package com.icetech.order.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.BetweenFormatter;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.google.common.collect.Lists;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.BasePark;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.basics.service.park.impl.ParkInoutdeviceServiceImpl;
import com.icetech.basics.service.park.impl.ParkServiceImpl;
import com.icetech.cloudcenter.api.order.OrderCarInfoService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.Page;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.order.domain.dto.OrderBackQueryDTO;
import com.icetech.order.domain.entity.OrderBack;
import com.icetech.order.domain.entity.OrderBackReview;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderSonInfo;
import com.icetech.order.domain.vo.CountVO;
import com.icetech.order.domain.vo.OrderBackVO;
import com.icetech.order.service.OrderBackGroupService;
import com.icetech.order.service.OrderBackService;
import com.icetech.oss.OssService;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.freespace.UpdateParkFreeSpaceServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 折返回场记录表 服务实现类
 * <p>
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author fangct
 * @version 1.0 @date 2023-07-20
 */
@Service
@Slf4j
public class OrderBackGroupServiceImpl implements OrderBackGroupService {
    @Resource
    private OrderBackService orderBackService;
    @Resource
    private ParkServiceImpl parkService;
    @Autowired
    private OrderCarInfoService orderCarInfoService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private OssService ossService;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private ParkInoutdeviceServiceImpl parkInoutdeviceService;
    @Autowired
    private OrderSonInfoServiceImpl orderSonInfoService;
    @Autowired
    private UpdateParkFreeSpaceServiceImpl updateParkFreeSpaceService;

    /**
     * 车辆折返
     *
     * @param queryDTO 查询参数
     * @param parkIds  车场
     * @return list
     */
    @Override
    public Page<OrderBackVO> listPage(OrderBackQueryDTO queryDTO, List<Long> parkIds) {
        // 获取订单分页列表
        Page<OrderBack> pageList = orderBackService.getPageList(queryDTO, parkIds);
        // 创建返回值分页对象
        Page<OrderBackVO> backVoPage = new Page<>();
        // 复制分页属性
        BeanUtil.copyProperties(pageList, backVoPage);
        // 判断分页列表是否为空
        if (Objects.isNull(pageList) || CollectionUtils.isEmpty(pageList.getRows())) {
            return backVoPage;
        }
        Map<Integer, ParkDevice> parkDeviceMap = new HashMap<>(16);
        if (queryDTO.getInandoutType() == 1) {
            Set<Long> channelIds = pageList.getRows().stream().map(OrderBack::getChannelId).collect(Collectors.toSet());
            ObjectResponse<List<ParkDevice>> response = parkDeviceService.getDeviceListByChannelIds(new ArrayList<>(channelIds), 9);
            if (ObjectResponse.isSuccess(response)) {
                parkDeviceMap = response.getData().stream().collect(Collectors.toMap(ParkDevice::getChannelId, parkDevice -> parkDevice));
            }
        }
        // 确保 parkDeviceMap 是不可变的
        final Map<Integer, ParkDevice> finalParkDeviceMap = parkDeviceMap;
        // 提取订单中的停车场ID并转换为Set集合
        Set<Long> parkIdSet = pageList.getRows().stream().map(OrderBack::getParkId).collect(Collectors.toSet());
        // 根据停车场ID列表获取停车场信息并转换为Map
        Map<Long, BasePark> parkMap = parkService.listByIds(parkIdSet).stream().collect(Collectors.toMap(BasePark::getId, Function.identity()));
        // 提取订单编号并转换为Set集合
        Set<String> orderNums = pageList.getRows().stream().map(OrderBack::getOrderNum).collect(Collectors.toSet());
        // 获取订单车辆信息列表并转换为Map
        Map<String, OrderCarInfo> orderCarInfoMap = orderCarInfoService.getCarInfoList(orderNums, null).getData().stream().collect(Collectors.toMap(OrderCarInfo::getOrderNum, Function.identity()));
        // 获取订单信息并转换为Map
        Map<String, OrderInfo> orderInfoMap = orderService.selectListByOrderNums(orderNums).stream().collect(Collectors.toMap(OrderInfo::getOrderNum, Function.identity()));
        // 转换订单列表为订单视图对象列表
        List<OrderBackVO> orderBacks = pageList.getRows().stream().map(t -> {
            OrderBackVO vo = new OrderBackVO();
            BeanUtil.copyProperties(t, vo); // 复制订单属性到视图对象
            if (StringUtils.isNotEmpty(t.getEvidenceVideo()) || Objects.nonNull(finalParkDeviceMap.get(Math.toIntExact(t.getChannelId())))) {
                vo.setHasMorCamera(1);
            }
            vo.setEvidenceVideo("");
            // 设置停车场信息
            BasePark basePark = parkMap.get(t.getParkId());
            if (basePark != null) {
                vo.setParkCode(basePark.getParkCode());
                vo.setParkName(basePark.getParkName());
                vo.setIsInterior(basePark.getIsInterior());
            }
            // 设置订单车辆信息
            OrderCarInfo orderCarInfo = orderCarInfoMap.get(t.getOrderNum());
            if (orderCarInfo != null) {
                vo.setEnterNo(orderCarInfo.getEnterNo());
                vo.setExitNo(orderCarInfo.getExitNo());
                // 根据条件加载图像URL
                if (queryDTO.getLoadImage()) {
                    vo.setEnterImg(ossService.getImageUrl(orderCarInfo.getSmallEnterImage()));
                    vo.setExitImg(ossService.getImageUrl(orderCarInfo.getSmallExitImage()));
                } else {
                    vo.setEnterImg("");
                    vo.setExitImg("");
                }
            }
            // 设置订单额外信息
            OrderInfo orderInfo = orderInfoMap.get(t.getOrderNum());
            if (orderInfo != null) {
                vo.setServiceStatus(orderInfo.getServiceStatus());
                vo.setHasSon(orderInfo.getHasSon());
                vo.setOrderId(orderInfo.getId());
                // 计算停车时长并格式化
                long exitTime = Optional.ofNullable(orderInfo.getExitTime()).orElse(DateUtil.currentSeconds());
                long duration = Math.max(exitTime - orderInfo.getEnterTime(), 0) * 1000;
                vo.setParkingTimeStr(DateUtil.formatBetween(duration, BetweenFormatter.Level.SECOND));
            }
            return vo;
        }).collect(Collectors.toList());
        // 设置视图对象列表到分页对象
        backVoPage.setRows(orderBacks);
        ;
        return backVoPage;
    }


    @Override
    public CountVO count(OrderBackQueryDTO queryDTO, List<Long> parkIds) {
        Integer total = orderBackService.getTotal(queryDTO, parkIds);
        return CountVO.builder().totalNum(total).build();
    }

    @Override
    public OrderBackVO detail(Long recordId, Long parkId) {
        OrderBack orderBack = orderBackService.getOrderBackById(recordId);
        if (orderBack == null) {
            return null;
        }
        if (!Objects.equals(orderBack.getParkId(), parkId)) {
            return null;
        }
        OrderBackVO vo = new OrderBackVO();
        BeanUtil.copyProperties(orderBack, vo);
        vo.setEvidenceVideo(ossService.getImageUrl(vo.getEvidenceVideo()));
        Park basePark = parkService.findByParkId(parkId).getData();
        if (Objects.nonNull(basePark)) {
            vo.setParkCode(basePark.getParkCode());
            vo.setParkName(basePark.getParkName());
            vo.setIsInterior(basePark.getIsInterior());
        }
        OrderCarInfo orderCarInfo = orderCarInfoService.getCarInfo(orderBack.getOrderNum(), parkId).getData();
        if (Objects.nonNull(orderCarInfo)) {
            vo.setEnterImg(ossService.getImageUrl(orderCarInfo.getSmallEnterImage()));
            vo.setExitImg(ossService.getImageUrl(orderCarInfo.getSmallExitImage()));
            vo.setEnterNo(orderCarInfo.getEnterNo());
            vo.setExitNo(orderCarInfo.getExitNo());
        }
        OrderInfo orderInfo = orderService.findByOrderNum(orderBack.getOrderNum()).getData();
        if (Objects.nonNull(orderInfo)) {
            vo.setServiceStatus(orderInfo.getServiceStatus());
            vo.setHasSon(orderInfo.getHasSon());
            vo.setOrderId(orderInfo.getId());
            Long exitTime = DateUtil.currentSeconds();
            if (Objects.nonNull(orderInfo.getExitTime())) {
                exitTime = orderInfo.getExitTime();
            }
            long bet = Math.max(exitTime - orderInfo.getEnterTime(), 0) * 1000;
            vo.setParkingTimeStr(DateUtil.formatBetween(bet, BetweenFormatter.Level.SECOND));
        }
        return vo;
    }


    @Override
    public OrderBackVO detail(String orderNum, Long parkId) {
        List<OrderBack> listByOrderNum = orderBackService.getListByOrderNum(orderNum);
        if (CollectionUtils.isNotEmpty(listByOrderNum)) {
            OrderBack orderBack = CollectionUtil.getLast(listByOrderNum);
            return detail(orderBack.getId(), parkId);
        }
        return null;
    }

    @Override
    public List<OrderBackVO> listByOrderNum(String orderNum) {
        List<OrderBack> orderBacks = orderBackService.getListByOrderNum(orderNum);
        List<OrderBackVO> vos = Lists.newArrayList();
        orderBacks.forEach(o -> {
            OrderBackVO vo = new OrderBackVO();
            BeanUtil.copyProperties(o, vo);
            vo.setEvidenceVideo(ossService.getImageUrl(vo.getEvidenceVideo()));
            Park basePark = parkService.findByParkId(o.getParkId()).getData();
            if (Objects.nonNull(basePark)) {
                vo.setParkCode(basePark.getParkCode());
                vo.setParkName(basePark.getParkName());
                vo.setIsInterior(basePark.getIsInterior());
            }
            OrderCarInfo orderCarInfo = orderCarInfoService.getCarInfo(o.getOrderNum(), o.getParkId()).getData();
            if (Objects.nonNull(orderCarInfo)) {
                vo.setEnterImg(ossService.getImageUrl(orderCarInfo.getEnterImage()));
                vo.setExitImg(ossService.getImageUrl(orderCarInfo.getExitImage()));
                vo.setEnterNo(orderCarInfo.getEnterNo());
                vo.setExitNo(orderCarInfo.getExitNo());
            }
            OrderInfo orderInfo = orderService.findByOrderNum(o.getOrderNum()).getData();
            if (Objects.nonNull(orderInfo)) {
                vo.setOrderId(orderInfo.getId());
                vo.setServiceStatus(orderInfo.getServiceStatus());
            }
            vos.add(vo);
        });
        return vos;
    }

    /**
     * @param review 入参
     * @return 状态
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse<Void> review(OrderBackReview review) {
        OrderBack orderBack = orderBackService.getById(review.getId());
        if (Objects.isNull(orderBack)) {
            return ObjectResponse.failed("500", "审核失败，暂无此ID");
        }
        if (orderBack.getReviewStatus() == 1) {
            return ObjectResponse.failed("500", "审核失败，此单已审核");
        }
        orderBack.setReviewStatus(1);
        orderBackService.modifyOrderBack(orderBack);
        //1- 确认进场
        if (review.getReviewType() == 1) {
            return ObjectResponse.success();
        }
        //获取通道信息
        ParkInoutdevice inoutDevice = parkInoutdeviceService.getOne(Wrappers.lambdaQuery(ParkInoutdevice.class)
                .eq(ParkInoutdevice::getId, orderBack.getChannelId()));
        if (Objects.isNull(inoutDevice)) {
            return ObjectResponse.failed("500", "审核失败，暂无此通道");
        }
        //主订单
        ObjectResponse<OrderInfo> orderInfoResponse = orderService.findByOrderNum(orderBack.getOrderNum());
        if (!ObjectResponse.isSuccess(orderInfoResponse)) {
            return ObjectResponse.failed("500", "审核失败，暂无此订单");
        }
        OrderInfo orderInfo = orderInfoResponse.getData();
        orderInfo.setServiceStatus(3);
        orderService.updateOrderInfo(orderInfo);
        if (Integer.valueOf(1).equals(orderInfo.getHasSon())) {
            //更新子订单
            orderSonInfoService.updateServiceStatusByOrderNums(Collections.singletonList(orderInfo.getOrderNum()), OrderStatusConstants.CANCEL);
        }
        //场中场
        if (inoutDevice.getIsMaster() == 0) {
            List<OrderSonInfo> sonInfos = orderSonInfoService.getInfosByOrderNum(orderBack.getParkId(), orderBack.getOrderNum());
            sonInfos.forEach(sonInfo -> {
                sonInfo.setServiceStatus(4);
                orderSonInfoService.save(sonInfo);
            });
        }
        updateParkFreeSpaceService.resetFreeSpace(orderBack.getParkId());
        return ObjectResponse.success();
    }
}
