package com.icetech.order.dao;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderSonInfo;
import com.icetech.cloudcenter.domain.order.update.OrderInfoUpdate;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PlateTypeEnum;
import com.icetech.db.mybatis.base.mapper.SuperMapper;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.stereotype.Repository;

import java.util.List;

/**
 * Description : 子订单信息表操作DAO
 *
 * @author zys
 */
@Repository
public interface OrderSonInfoDao extends SuperMapper<OrderSonInfo> {
    default OrderSonInfo selectOneByEntity(OrderSonInfo entity) {
        return selectLimitOne(Wrappers.lambdaQuery(entity).orderByDesc(OrderSonInfo::getEnterTime));
    }

    default int updateByOrderNum(OrderSonInfo sonInfo) {
        Long id = sonInfo.getId();
        String orderNum = sonInfo.getOrderNum();
        sonInfo.setId(null);
        sonInfo.setOrderNum(null);
        int result = update(sonInfo, Wrappers.lambdaUpdate(OrderSonInfo.class).eq(OrderSonInfo::getOrderNum, orderNum));
        sonInfo.setId(id);
        sonInfo.setOrderNum(orderNum);
        return result;
    }
    default int updateByOrderNumAndRegionId(OrderSonInfo sonInfo) {
        Long id = sonInfo.getId();
        String orderNum = sonInfo.getOrderNum();
        Long regionId = sonInfo.getRegionId();
        if (orderNum == null || regionId == null) {
            return 0;
        }
        sonInfo.setId(null);
        sonInfo.setOrderNum(null);
        sonInfo.setRegionId(null);
        int result = update(sonInfo, Wrappers.lambdaUpdate(OrderSonInfo.class)
                .eq(OrderSonInfo::getOrderNum, orderNum)
                .eq(OrderSonInfo::getRegionId, regionId));
        sonInfo.setId(id);
        sonInfo.setOrderNum(orderNum);
        sonInfo.setRegionId(regionId);
        return result;
    }

    default int insertWithPlateNum2(OrderSonInfo orderInfo) {
        orderInfo.setPlateNum2(StringUtils.substring(orderInfo.getPlateNum(), 1));
        return insert(orderInfo);
    }

    /**
     * 车牌验证
     *
     * @param parkId
     * @param plateNumPart
     * @return
     */
    default OrderSonInfo fuzzyInParkPlate(Long parkId, String plateNumPart) {
        return selectLimitOne(Wrappers.lambdaQuery(OrderSonInfo.class)
                .eq(OrderSonInfo::getParkId, parkId)
                .eq(OrderSonInfo::getServiceStatus, OrderStatusConstants.IN_PARK)
                .eq(OrderSonInfo::getPlateNum2, plateNumPart)
                .orderByDesc(OrderSonInfo::getEnterTime));
    }

    default List<OrderSonInfo> selectListByPlates(OrderSonInfo entity, List<String> plateList) {
        return selectList(Wrappers.lambdaQuery(entity)
                .eq(OrderInfo::getParkId, entity.getParkId())
                .eq(OrderInfo::getServiceStatus, entity.getServiceStatus())
                .eq(entity.getType() != null, OrderInfo::getType, entity.getType())
                .eq(entity.getRegionId() != null && entity.getRegionId() != 0, OrderInfo::getRegionId, entity.getRegionId())
                .in(OrderInfo::getPlateNum, plateList)
                .orderByAsc(OrderInfo::getEnterTime));
    }

    /**
     * 修改状态
     *
     * @param orderInfoUpdate
     * @return
     */
    Integer updateStatus(OrderInfoUpdate orderInfoUpdate);

    int setInPark(@Param("orderSonId") long orderSonId);

    int setException(@Param("orderNum") String orderNum, @Param("exitTime") Long exitTime, @Param("oddStatus") Integer oddStatus);

    /**
     * 查询子订单是否已经存在
     *
     * @param plateNum
     * @param parkId
     * @return
     */
    default OrderSonInfo selectOrderSonInpark(String plateNum, Long parkId) {
        return selectLimitOne(Wrappers.lambdaQuery(OrderSonInfo.class)
                .eq(OrderSonInfo::getParkId, parkId)
                .eq(OrderSonInfo::getServiceStatus, OrderStatusConstants.IN_PARK)
                .eq(OrderSonInfo::getPlateNum, plateNum)
                .orderByDesc(OrderSonInfo::getEnterTime));
    }

    /**
     * 统计月卡多位多车车牌在当前区域的在场订单个数
     *
     * @param parkId    车场ID
     * @param plateNums 车牌列表
     * @param regionId  区域ID
     * @return 总在场个数
     */
    default int countInParkMonthCardByPlateNums(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums, @Param("regionId") Long regionId) {
        return selectCount(Wrappers.lambdaQuery(OrderSonInfo.class)
                .eq(OrderSonInfo::getParkId, parkId)
                .eq(OrderSonInfo::getServiceStatus, OrderStatusConstants.IN_PARK)
                .eq(OrderSonInfo::getType, 2)
                .eq(OrderSonInfo::getRegionId, regionId)
                .in(OrderSonInfo::getPlateNum, plateNums));
    }
    default int countExitRecordsFromStartTime(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums,
                                              @Param("regionId") Long regionId, @Param("startTime") Long startTime) {
        return selectCount(Wrappers.lambdaQuery(OrderSonInfo.class)
                .eq(OrderSonInfo::getParkId, parkId)
                .eq(OrderSonInfo::getRegionId, regionId)
                .in(OrderSonInfo::getPlateNum, plateNums)
                .and(query -> query.ge(OrderSonInfo::getExitTime, startTime).lt(OrderSonInfo::getEnterTime, startTime)
                        .or(query2 -> query2.eq(OrderSonInfo::getServiceStatus, OrderStatusConstants.IN_PARK)
                                .eq(OrderSonInfo::getType, PlateTypeEnum.月卡车.getType())
                        )
                )
        );
    }
    List<String> selectInParkMonthCardByPlateNums(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums, @Param("regionId") Long regionId);
}
