package com.icetech.order.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.common.utils.DateTools;
import com.icetech.order.dao.OrderPayDiscountMapper;
import com.icetech.order.domain.dto.PayDiscountListDTO;
import com.icetech.order.domain.entity.OrderPayDiscount;
import com.icetech.order.domain.vo.OrderPayDiscountPageVo;
import com.icetech.order.domain.vo.PayDiscountListVo;
import com.icetech.order.service.OrderPayDiscountService;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.park.domain.entity.park.Park;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 支付优惠关联表 服务实现类
 * <p>
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author fangchangtong
 * @version 1.0 @date 2024-06-03
 */
@Service
public class OrderPayDiscountServiceImpl extends BaseServiceImpl<OrderPayDiscountMapper, OrderPayDiscount> implements OrderPayDiscountService {

    @Autowired
    private ParkService parkService;

    @Override
    public OrderPayDiscountPageVo<PayDiscountListVo> listPage(PayDiscountListDTO queryDTO) {
        QueryWrapper<OrderPayDiscount> wrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(queryDTO.getParkIdList())) {
            wrapper.in("park_id", queryDTO.getParkIdList());
        }
        if (CollectionUtils.isNotEmpty(queryDTO.getDiscountSources())) {
            wrapper.and(wrapper1 -> {
                if (queryDTO.getDiscountSources().contains(1)) {
                    wrapper1.gt("merchant_discount_price", 0);
                }
                if (queryDTO.getDiscountSources().contains(2)) {
                    wrapper1.or().gt("third_discount_price", 0);
                }
                if (queryDTO.getDiscountSources().contains(3)) {
                    wrapper1.or().gt("vip_discount_price", 0);
                }
                if (queryDTO.getDiscountSources().contains(4)) {
                    wrapper1.or().gt("toll_discount_price", 0);
                }
            });
        }
        wrapper.like(StringUtils.isNotEmpty(queryDTO.getPlateNum()), "plate_num", queryDTO.getPlateNum());
        wrapper.ge(StringUtils.isNotBlank(queryDTO.getStartTime()), "pay_date", queryDTO.getStartTime());
        wrapper.le(StringUtils.isNotBlank(queryDTO.getEndTime()), "pay_date", queryDTO.getEndTime());
        wrapper.orderByDesc("id");
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<OrderPayDiscount> pageList = page(wrapper, queryDTO.getPageNo(), queryDTO.getPageSize());

        wrapper.select("ifnull(sum(total_price), 0) sumTotalPrice, ifnull(sum(discount_price), 0) sumDiscountPrice, ifnull(sum(paid_price), 0) sumPaidPrice, " +
                "ifnull(sum(merchant_discount_price), 0) sumMerchantDiscountPrice, ifnull(sum(third_discount_price), 0) sumThirdDiscountPrice, " +
                "ifnull(sum(vip_discount_price), 0) sumVipDiscountPrice, ifnull(sum(toll_discount_price), 0) sumTollDiscountPrice");
        Map<String, Object> sumPriceMap = getMap(wrapper);
        OrderPayDiscountPageVo<PayDiscountListVo> voPage = new OrderPayDiscountPageVo<>();
        if (pageList == null || CollectionUtils.isEmpty(pageList.getRecords())) {
            voPage.setTotal(0L);
            voPage.setTotalPage(0);
            return voPage;
        }
        voPage.setSumTotalPrice(((BigDecimal) sumPriceMap.get("sumTotalPrice")).doubleValue());
        voPage.setSumDiscountPrice(((BigDecimal) sumPriceMap.get("sumDiscountPrice")).doubleValue());
        voPage.setSumPaidPrice(((BigDecimal) sumPriceMap.get("sumPaidPrice")).doubleValue());
        voPage.setSumMerchantDiscountPrice(((BigDecimal) sumPriceMap.get("sumMerchantDiscountPrice")).doubleValue());
        voPage.setSumThirdDiscountPrice(((BigDecimal) sumPriceMap.get("sumThirdDiscountPrice")).doubleValue());
        voPage.setSumVipDiscountPrice(((BigDecimal) sumPriceMap.get("sumVipDiscountPrice")).doubleValue());
        voPage.setSumTollDiscountPrice(((BigDecimal) sumPriceMap.get("sumTollDiscountPrice")).doubleValue());
        voPage.setTotalPage((int) pageList.getPages());
        List<Long> parkIdList = pageList.getRecords().stream().map(OrderPayDiscount::getParkId).distinct().collect(Collectors.toList());
        List<Park> parkList = parkService.findByParkIds(parkIdList);
        Map<Long, Park> parkMap = parkList.stream().collect(Collectors.toMap(Park::getId, park -> park));

        List<PayDiscountListVo> pageListVoList = new ArrayList<>();

        for (OrderPayDiscount record : pageList.getRecords()) {
            PayDiscountListVo vo = new PayDiscountListVo();
            BeanUtils.copyProperties(record, vo);
            Park park = parkMap.get(record.getParkId());
            if (park != null) {
                vo.setParkName(park.getParkName());
            }
            vo.setParkTime(record.getParkTime());
            if (record.getParkTime() != null) {
                vo.setParkingTimeStr(DateTools.secondToSecondsTime(record.getParkTime().intValue()));
            }
            vo.setPayTime(record.getPayDate() != null ? record.getPayDate().getTime() / 1000 : null);
            pageListVoList.add(vo);
        }
        voPage.setRows(pageListVoList);
        voPage.setTotal(pageList.getTotal());
        return voPage;
    }

    /**
     * Description: 根据ID查询 支付优惠关联表对象信息 <br>
     * Version1.0 2024-06-03 by fangchangtong创建
     *
     * @param id 对象id
     * @return OrderPayDiscount
     */
    @Override
    public OrderPayDiscount getOrderPayDiscountById(Long id) {
        return getById(id);
    }

    /**
     * Description: 新增  支付优惠关联表对象信息 <br>
     * Version1.0 2024-06-03 by fangchangtong创建
     *
     * @param entity 对象信息
     * @return Boolean 返回新增后主键
     */
    @Override
    public Boolean addOrderPayDiscount(OrderPayDiscount entity) {
        return save(entity);
    }

    /**
     * Description: 修改 支付优惠关联表对象信息 <br>
     * Version1.0 2024-06-03 by fangchangtong创建
     *
     * @param entity 对象信息
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean modifyOrderPayDiscount(OrderPayDiscount entity) {
        return updateById(entity);
    }

    /**
     * Description: 删除  支付优惠关联表对象信息 <br>
     * Version1.0 2024-06-03 by fangchangtong创建
     *
     * @param id 对象id
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean removeOrderPayDiscountById(Long id) {
        return removeById(id);
    }

}
