package com.icetech.order.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.basics.constants.CommonConsts;
import com.icetech.basics.dao.park.ParkInoutdeviceDao;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.city.road.api.OrderApi;
import com.icetech.city.road.domain.constant.OrderRecoveryFlagConsts;
import com.icetech.city.road.domain.constant.OrderStatusConstant;
import com.icetech.city.road.domain.constant.OrderTypeConstant;
import com.icetech.city.road.domain.entity.request.param.OrderQueryParam;
import com.icetech.city.road.domain.table.Order;
import com.icetech.cloudcenter.api.order.OrderCarInfoService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.vo.NotPayDetailVo;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.Page;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderNotpayDao;
import com.icetech.order.dao.OrderTagsDao;
import com.icetech.order.domain.dto.NotPayRecordQueryDTO;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.order.domain.entity.OrderTags;
import com.icetech.order.domain.vo.CountVO;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.third.service.third.MqPushService;
import com.icetech.third.utils.DateRangeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * 订单欠费记录表 服务实现类
 * <p>
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author fangct
 * @version 1.0 @date 2023-03-08
 */
@Service
@Slf4j
public class OrderNotpayServiceImpl extends BaseServiceImpl<OrderNotpayDao, OrderNotpay> implements OrderNotpayService {
    @Autowired
    private ParkService parkService;
    @Autowired
    private ParkInoutdeviceDao parkInoutdeviceDao;
    @Autowired
    private OrderTagsDao orderTagsDao;
    @Autowired
    private OrderCarInfoService orderCarInfoService;
    @Autowired
    private MqPushService mqPushService;
    @Autowired
    private OrderApi roadOrderApi;

    @Override
    public List<OrderNotpay> queryNotPayFee(Long parkId, String plateNum, Integer exType) {
        ObjectResponse<ParkConfig> objectResponse = parkService.getParkConfig(parkId);
        if (!ObjectResponse.isSuccess(objectResponse)) {
            return null;
        }
        ParkConfig parkConfig = objectResponse.getData();
        return queryNotPayFee(parkId, plateNum, exType, parkConfig);
    }

    @Override
    public List<OrderNotpay> queryNotPayFee(Long parkId, String plateNum, Integer exType, ParkConfig parkConfig) {
        if (!CommonConsts.YES.equals(parkConfig.getOnlyRecoverySelf())) {
            parkId = null;
        }
        if (exType == null) {
            if (NumberUtils.toPrimitive(parkConfig.getEnterpayType()) == 1
                    && NumberUtils.toPrimitive(parkConfig.getExitpayType()) == 1) {
                exType = 2;
            } else if (NumberUtils.toPrimitive(parkConfig.getEnterpayType()) == 1) {
                exType = 1;
            } else {
                exType = 2;
            }
        }
        List<OrderNotpay> list = null;
        if (exType == 1) {
            if (NumberUtils.toPrimitive(parkConfig.getEnterpayType()) == 1) {
                if (NumberUtils.toPrimitive(parkConfig.getEnterpayRangeDay()) > 0) {
                    long endTime = DateTools.unixTimestamp();
                    long startTime = endTime - parkConfig.getEnterpayRangeDay() * 3600 * 24;
                    LambdaQueryWrapper<OrderNotpay> queryWrapper = Wrappers.lambdaQuery(OrderNotpay.class)
                            .eq(parkId != null, OrderNotpay::getParkId, parkId)
                            .eq(OrderNotpay::getPlateNum, plateNum)
                            .eq(OrderNotpay::getStatus, 1)
//                            .ge(OrderNotpay::getExitTime, startTime)
//                            .lt(OrderNotpay::getExitTime, endTime)
                            .orderByDesc(OrderNotpay::getId);
                    list = list(queryWrapper);
                }
            }
        } else {
            if (NumberUtils.toPrimitive(parkConfig.getExitpayType()) == 1) {
                if (NumberUtils.toPrimitive(parkConfig.getExitpayRangeDay()) > 0) {
                    long endTime = DateTools.unixTimestamp();
                    long startTime = endTime - parkConfig.getExitpayRangeDay() * 3600 * 24;
                    LambdaQueryWrapper<OrderNotpay> queryWrapper = Wrappers.lambdaQuery(OrderNotpay.class)
                            .eq(parkId != null, OrderNotpay::getParkId, parkId)
                            .eq(OrderNotpay::getPlateNum, plateNum)
                            .eq(OrderNotpay::getStatus, 1)
//                            .ge(OrderNotpay::getExitTime, startTime)
//                            .lt(OrderNotpay::getExitTime, endTime)
                            .orderByDesc(OrderNotpay::getId);
                    list = list(queryWrapper);
                }
            }
        }
        if (CommonConsts.YES.equals(parkConfig.getAllowRecoveryRoad())) {
            List<OrderNotpay> roadOrders = queryRoadUnpaidOrders(plateNum);
            if (list == null) list = roadOrders;
            else list.addAll(roadOrders);
        }
        return list;
    }

    /**
     * Description: 根据ID查询 订单欠费记录表对象信息 <br>
     * Version1.0 2023-03-08 by fangct创建
     *
     * @param id 对象id
     * @return OrderNotpay
     */
    @Override
    public OrderNotpay getOrderNotpayById(Long id) {
        return getById(id);
    }

    /**
     * Description: 新增  订单欠费记录表对象信息 <br>
     * Version1.0 2023-03-08 by fangct创建
     *
     * @param entity 对象信息
     * @return Boolean 返回新增后主键
     */
    @Override
    public Boolean addOrderNotpay(OrderNotpay entity) {
        orderTagsDao.insert(OrderTags.builder().parkId(entity.getParkId())
                .orderNum(entity.getOrderNum()).tagId(OrderTags.TagIdEnum.NOT_PAY_CAR.getType()).build());
        boolean result = save(entity);
        mqPushService.pushOrderNotPay(entity);
        return result;
    }

    @Override
    public Boolean dealNotpayOrder(OrderNotpay orderNotpay) {
        orderTagsDao.delete(Wrappers.lambdaQuery(OrderTags.class)
                .eq(OrderTags::getOrderNum, orderNotpay.getOrderNum())
                .eq(OrderTags::getTagId, OrderTags.TagIdEnum.NOT_PAY_CAR.getType()));
        return update(orderNotpay, Wrappers.lambdaUpdate(OrderNotpay.class).eq(OrderNotpay::getId, orderNotpay.getId())
                .eq(OrderNotpay::getParkId, orderNotpay.getParkId()).eq(OrderNotpay::getStatus, 1));
    }

    @Override
    public OrderNotpay getOrderNotpayByOrderNum(String orderNum) {
        return getOne(Wrappers.lambdaQuery(OrderNotpay.class).eq(OrderNotpay::getOrderNum, orderNum));
    }

    @Override
    public List<OrderNotpay> getListByOrderNums(List<String> orderNums) {
        return list(Wrappers.lambdaQuery(OrderNotpay.class).eq(OrderNotpay::getStatus, 1).in(OrderNotpay::getOrderNum, orderNums).orderByDesc(OrderNotpay::getId));
    }

    @Override
    public List<OrderNotpay> getPaidPriceGroupParkId(Integer status, String month) {
        QueryWrapper<OrderNotpay> wrapper = new QueryWrapper<>();
        wrapper.select("park_id as parkId, sum(paid_price) as paidPrice");
        wrapper.eq("status", status);
        wrapper.likeRight("oper_time", month);
        wrapper.groupBy("park_id");
        wrapper.orderByDesc("id");
        return list(wrapper);
    }

    @Override
    public Page<OrderNotpay> getNotPayRecord(NotPayRecordQueryDTO queryDTO, List<Long> parkIds) {
        QueryWrapper<OrderNotpay> wrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(parkIds)) {
            wrapper.in(org.apache.commons.collections4.CollectionUtils.isNotEmpty(parkIds) && !parkIds.contains(-1L), "park_id", parkIds);
        }
        wrapper.eq(Objects.nonNull(queryDTO.getStatus()), "status", queryDTO.getStatus());
        wrapper.in(CollectionUtils.isNotEmpty(queryDTO.getMendPayType()), "mend_pay_type", queryDTO.getMendPayType());
        wrapper.eq(StringUtils.isNotEmpty(queryDTO.getPlateNum()), "plate_num", queryDTO.getPlateNum());
        if (Objects.nonNull(queryDTO.getStartDate()) && Objects.nonNull(queryDTO.getEndDate())) {
            wrapper.between(queryDTO.getStatus() == 1 ? "FROM_UNIXTIME(exit_time)" : "oper_time", queryDTO.getStartDate(),
                    queryDTO.getEndDate());
        }

        wrapper.orderByDesc("id");
        // 查询列表
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<OrderNotpay> page = page(wrapper, queryDTO.getPageNo(), queryDTO.getPageSize());
        return Page.instance(Math.toIntExact(page.getPages()), page.getTotal(), page.getRecords());
    }

    @Override
    public CountVO getTotal(NotPayRecordQueryDTO queryDTO, List<Long> parkIds) {
        CountVO count = new CountVO();
        QueryWrapper<OrderNotpay> wrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(parkIds)) {
            wrapper.in(org.apache.commons.collections4.CollectionUtils.isNotEmpty(parkIds) && !parkIds.contains(-1L), "park_id", parkIds);
        }
        wrapper.eq(Objects.nonNull(queryDTO.getStatus()), "status", queryDTO.getStatus());
        wrapper.in(CollectionUtils.isNotEmpty(queryDTO.getMendPayType()), "mend_pay_type", queryDTO.getMendPayType());
        wrapper.eq(StringUtils.isNotEmpty(queryDTO.getPlateNum()), "plate_num", queryDTO.getPlateNum());
        if (Objects.nonNull(queryDTO.getStartDate()) && Objects.nonNull(queryDTO.getEndDate())) {
            wrapper.between(queryDTO.getStatus() == 1 ? "FROM_UNIXTIME(exit_time)" : "oper_time", queryDTO.getStartDate(),
                    queryDTO.getEndDate());
        }
        // 查询车牌号或者金额
        wrapper.select("count(*) as 'totalNum', sum(total_price) as 'totalPrice', sum(paid_price) as 'paidPrice'");
        List<Map<String, Object>> maps = listMaps(wrapper);
        if (CollectionUtils.isNotEmpty(maps)) count = JSON.parseObject(JSON.toJSONString(maps.get(0)), CountVO.class);
        return count;
    }

    @Override
    public Boolean deleteByOrderNum(String orderNum) {
        OrderNotpay orderNotpay = this.getOrderNotpayByOrderNum(orderNum);
        if (Objects.nonNull(orderNotpay)) {
            orderTagsDao.delete(Wrappers.lambdaQuery(OrderTags.class)
                    .eq(OrderTags::getOrderNum, orderNum)
                    .eq(OrderTags::getTagId, 3));
            return removeById(orderNotpay.getId());
        }
        return Boolean.FALSE;
    }


    @Autowired
    private OrderInfoDao orderInfoDao;

    @Override
    public void updatePaidByOrderNums(List<NotPayDetailVo> vos, String channelCode, String operAccount) {
        log.info("[补缴订单更新] orderNums {}", vos);
        ParkInoutdevice parkInoutdevice = null;
        if (StringUtils.isNotEmpty(channelCode)) {
            parkInoutdevice = parkInoutdeviceDao.selectLimitOne(Wrappers.lambdaQuery(ParkInoutdevice.class).eq(ParkInoutdevice::getInandoutCode, channelCode));
        }

        DateTime current = DateUtil.date();
        ParkInoutdevice finalParkInoutdevice = parkInoutdevice;
        for (NotPayDetailVo vo : vos) {
            if (!vo.isParkOrder()) continue;
            try {
                OrderNotpay orderNotpay = getOrderNotpayByOrderNum(vo.getOrderNum());
                if (Objects.nonNull(orderNotpay)) {
                    orderNotpay.setStatus(2);
                    orderNotpay.setPaidPrice(new BigDecimal(vo.getUnPayPrice()));
                    if (finalParkInoutdevice != null) {
                        orderNotpay.setMendPayType(finalParkInoutdevice.getInandoutType());
                        orderNotpay.setChannelName(finalParkInoutdevice.getInandoutName());
                    }
                    orderNotpay.setOperTime(current);
                    updateById(orderNotpay);

                    OrderInfo orderInfo = orderInfoDao.selectByOrderNum(orderNotpay.getOrderNum());
                    boolean isHistoryTable = false;
                    List<String> yearQuarterRangeTableName = new ArrayList<>();
                    if (orderInfo == null) {
                        Date startDate = DateUtil.offsetDay(DateUtil.date(), -90);
                        yearQuarterRangeTableName = DateRangeUtils.getYearQuarterRangeTableName(startDate);
                        List<OrderInfo> orderInfos = orderInfoDao.selectWithHistoryByOrderNumList(yearQuarterRangeTableName, Collections.singletonList(orderNotpay.getOrderNum()));
                        if (orderInfos.size() == 0) {
                            log.error("[补缴订单更新] 失败, 订单号不存在 {}", orderNotpay.getOrderNum());
                            continue;
                        }
                        orderInfo = orderInfos.get(0);
                        isHistoryTable = true;
                    }
                    orderInfo.setServiceStatus(OrderStatusConstants.LEAVED_PARK);
                    orderInfo.setOperAccount(operAccount);
                    orderInfo.setPaidPrice(orderNotpay.getPaidPrice().setScale(2, BigDecimal.ROUND_HALF_UP).toString());
                    orderInfo.setTotalPrice(orderNotpay.getTotalPrice().setScale(2, BigDecimal.ROUND_HALF_UP).toString());
                    orderInfo.setDiscountPrice(orderNotpay.getDiscountPrice().setScale(2, BigDecimal.ROUND_HALF_UP).toString());

                    if (!isHistoryTable) {
                        orderInfoDao.update(orderInfo, Wrappers.lambdaUpdate(OrderInfo.class).eq(OrderInfo::getId, orderInfo.getId()));
                    } else {
                        for (String tableName : yearQuarterRangeTableName) {
                            int n = orderInfoDao.updateHistoryTable(tableName, orderInfo);
                            if (n > 0) {
                                break;
                            }
                        }
                    }
                    log.info("[补缴订单更新] orderNotpay {}, orderInfo {}", orderNotpay, orderInfo);

                    if (StringUtils.isNotBlank(operAccount)) {
                        //更新
                        OrderCarInfo orderCarInfo = orderCarInfoService.getByOrderNumWithHistory(orderNotpay.getOrderNum());
                        if (orderCarInfo != null) {
                            orderCarInfo.setExitOperAccount(operAccount);
                            orderCarInfoService.updateByIdWithHistory(orderCarInfo);
                        }
                    }
                    orderTagsDao.delete(Wrappers.lambdaQuery(OrderTags.class)
                            .eq(OrderTags::getOrderNum, orderNotpay.getOrderNum())
                            .eq(OrderTags::getTagId, OrderTags.TagIdEnum.NOT_PAY_CAR.getType()));
                    orderTagsDao.insert(OrderTags.builder()
                            .parkId(orderNotpay.getParkId())
                            .orderNum(orderNotpay.getOrderNum())
                            .tagId(OrderTags.TagIdEnum.BACK_PAY.getType()).build());
                }
            } catch (Exception e) {
                log.error("补缴订单更新异常, e=", e);
            }
        }
    }

    @Override
    public void largeOrder(OrderNotpay notPay) {
        ObjectResponse<ParkConfig> configResp = parkService.getParkConfig(notPay.getParkId());
        if (!ObjectResponse.isSuccess(configResp)) {
            return;
        }
        ParkConfig config = configResp.getData();
        if (config.getIsOrderBigAmt() == 0) {
            return;
        }
        BigDecimal orderBigAmt = Optional.ofNullable(config.getOrderBigAmt()).orElse(BigDecimal.ZERO);
        if (notPay.getTotalPrice().compareTo(orderBigAmt) > 0) {
            notPay.setStatus(4);
        } else {
            notPay.setStatus(1);
        }
    }

    public List<OrderNotpay> queryRoadUnpaidOrders(String plateNum) {
        OrderQueryParam param = new OrderQueryParam();
        param.setPlateNum(plateNum);
        param.setOrderStatus(OrderStatusConstant.NORMAL);
        param.setOrderType(OrderTypeConstant.EXIT);
        param.setHasUnpaidPrice(true);
        param.setRecoveryFlag(OrderRecoveryFlagConsts.RECOVERY);
        List<Order> unpaidOrders = roadOrderApi.queryOrders(param);
        log.debug("路内欠费订单查询|{}|{}", param, unpaidOrders);
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(unpaidOrders)) return Collections.emptyList();
        List<OrderNotpay> orderNotpays = new ArrayList<>(unpaidOrders.size());
        for (Order order : unpaidOrders) {
            OrderNotpay orderNotpay = new OrderNotpay();
            orderNotpay.setParkId(order.getParkId().longValue());
            orderNotpay.setOrderNum(order.getOrderNum());
            orderNotpay.setPlateNum(order.getPlateNum());
            orderNotpay.setType(1);
            orderNotpay.setEnterTime(order.getEnterTime());
            orderNotpay.setExitTime(order.getExitTime());
            orderNotpay.setCarType(order.getCarType());
            orderNotpay.setOrderSource(OrderNotpay.ORDER_SOURCE_ROAD);
            orderNotpay.setTotalPrice(order.getTotalPrice());
            orderNotpay.setDiscountPrice(order.getDiscountPrice());
            orderNotpay.setPaidPrice(order.getPaidPrice());
            orderNotpay.setStatus(1);
            orderNotpays.add(orderNotpay);
        }
        return orderNotpays;
    }
}
