package com.icetech.order.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateUtil;
import com.alibaba.csp.sentinel.annotation.SentinelResource;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.google.common.collect.Lists;
import com.icetech.basics.dao.park.ParkDao;
import com.icetech.basics.sentinel.ExceptionUtils;
import com.icetech.cloudcenter.api.order.OrderCarInfoService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.order.query.OrderEnexCount;
import com.icetech.cloudcenter.domain.request.FlowRequest;
import com.icetech.cloudcenter.domain.request.OrderEnexQueryRequest;
import com.icetech.cloudcenter.domain.response.EnexCountDto;
import com.icetech.cloudcenter.domain.response.EnexDetailDto;
import com.icetech.cloudcenter.domain.response.EnterMatchDto;
import com.icetech.cloudcenter.domain.response.ParkEnterOrexitCountDto;
import com.icetech.cloudcenter.domain.response.ParkEnterOrexitNumDto;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.order.dao.OrderCarInfoDao;
import com.icetech.order.dao.OrderEnexCountDao;
import com.icetech.order.dao.OrderInfoDao;
import com.icetech.order.dao.OrderSonCarInfoDao;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderSonCarInfo;
import com.icetech.oss.OssService;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.redis.handle.RedisHandle;
import com.icetech.third.anno.DS_SLAVE;
import com.icetech.third.utils.DateRangeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.icetech.third.utils.DateRangeUtils.get24DayHours;
import static com.icetech.third.utils.DateRangeUtils.get24HourTime;
import static com.icetech.third.utils.DateRangeUtils.get24Hours;
import static com.icetech.third.utils.DateRangeUtils.getDayTime;
import static com.icetech.third.utils.DateRangeUtils.getDays;
import static com.icetech.third.utils.DateRangeUtils.getDaysOfMonth;
import static com.icetech.third.utils.DateRangeUtils.getHour;
import static com.icetech.third.utils.DateRangeUtils.getMonthDays;
import static com.icetech.third.utils.DateRangeUtils.getNHourTime;
import static com.icetech.third.utils.DateRangeUtils.getNextTimeHourTime;
import static com.icetech.third.utils.DateRangeUtils.getNowHourTime;
import static com.icetech.third.utils.DateRangeUtils.getNowHours;
import static com.icetech.third.utils.DateRangeUtils.getTimeHourTime;
import static com.icetech.third.utils.DateRangeUtils.getTodayTime;
import static com.icetech.third.utils.DateRangeUtils.getYesterdayHourTime;

@Slf4j
@Service("orderCarInfoService")
public class OrderCarInfoServiceImpl extends BaseServiceImpl<OrderCarInfoDao, OrderCarInfo> implements OrderCarInfoService {
    @Resource
    private OrderCarInfoDao orderCarInfoDao;
    @Resource
    private OrderEnexCountDao orderEnexCountDao;
    @Resource
    private ParkDao parkDao;
    @Resource
    private ParkService parkService;
    @Resource
    private OssService ossService;
    @Resource
    private OrderInfoDao orderInfoDao;
    @Resource
    private OrderSonCarInfoDao orderSonCarInfoDao;
    @Resource
    private RedisHandle redisHandle;

    @Override
    @DS_SLAVE
    @SentinelResource(value = "OrderEnexService.enexCount", defaultFallback = "defaultFallbackHandle", fallbackClass = {ExceptionUtils.class})
    public ObjectResponse<EnexCountDto> enexCount(FlowRequest flowRequest) {
        EnexCountDto enexCountDto = new EnexCountDto();
        long enterCount = orderInfoDao.enterCount(flowRequest);
        long exitCount = orderInfoDao.exitCount(flowRequest);
        enexCountDto.setTotalEnter(enterCount);
        enexCountDto.setTotalExit(exitCount);
        return ObjectResponse.success(enexCountDto);
    }

    @Override
    @DS_SLAVE
    @SentinelResource(value = "OrderEnexService.enexDetail", defaultFallback = "defaultFallbackHandle", fallbackClass = {ExceptionUtils.class})
    public ObjectResponse<List<EnexDetailDto>> enexDetail(FlowRequest flowRequest) {
        String parkIdList = flowRequest.getParkIdList();

        List<EnexDetailDto> enterDetails = orderInfoDao.enterDetail(flowRequest);
        List<EnexDetailDto> exitDetails = orderInfoDao.exitDetail(flowRequest);
        exitDetails.addAll(enterDetails);
        exitDetails.sort(Comparator.<EnexDetailDto>comparingLong(detail -> detail.getExitTime() == null ? detail.getEnterTime() : detail.getExitTime()).reversed());
        List<EnexDetailDto> enexDetailDtos = exitDetails.subList(0, Integer.min(exitDetails.size(), 30));
        if (CollUtil.isNotEmpty(enexDetailDtos)) {
            List<Park> parks = parkDao.selectByIds(parkIdList);
            Map<Long, String> parkMap = parks.stream().collect(Collectors.toMap(Park::getId, Park::getParkName));
            for (EnexDetailDto dto : enexDetailDtos) {
                int serviceStatus = dto.getServiceStatus();
                if (serviceStatus == 1) {
                    dto.setStatus("入场");
                    dto.setGateName(dto.getEnterNo());
                    dto.setActionTime(DateTools.getFormat(dto.getEnterTime() * 1000));
                } else if(serviceStatus == 2 || serviceStatus == 4) {
                    dto.setStatus("出场");
                    dto.setGateName(dto.getExitNo());
                    if (dto.getExitTime()!=null){
                        dto.setActionTime(DateTools.getFormat(dto.getExitTime() * 1000));
                    }else {
                        dto.setActionTime(DateTools.getFormat(dto.getEnterTime() * 1000));
                    }

                }
                dto.setParkName(parkMap.get(dto.getParkId()));
            }
        }
        return ObjectResponse.success(enexDetailDtos);
    }

    @Override
    @DS_SLAVE
    @SentinelResource(value = "OrderEnexService.countDaysEnterOrExitNum", defaultFallback = "defaultFallbackHandle", fallbackClass = {ExceptionUtils.class})
    public ObjectResponse<List<ParkEnterOrexitCountDto>> countDaysEnterOrExitNum(String parkIdList, Integer day, String monthCount) {
        return ObjectResponse.success(countEnterOrExitNum(parkIdList, day, monthCount));
    }

    @Override
    @DS_SLAVE
    @SentinelResource(value = "OrderEnexService.countDaysEnterOrExitNum2", defaultFallback = "defaultFallbackHandle", fallbackClass = {ExceptionUtils.class})
    public ObjectResponse<List<ParkEnterOrexitCountDto>> countDaysEnterOrExitNum(String parkIdList, Integer day) {
        return ObjectResponse.success(countEnterOrExitNum(parkIdList, day, null));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ObjectResponse countHoursEnterOrExitNum(String time) {
        log.info("[统计当前时间的进出场流量],{}", time);
        long toTime;
        long fromTime;
        if (StringUtils.isNotBlank(time)) {
            //获取当前传入时间时间整点的时间戳
            fromTime = getTimeHourTime(time);
            //获取下一个小时的时间戳
            toTime = getNextTimeHourTime(fromTime, 1);
            log.info("[统计当前时间的进出场流量],开始时间:{},结束时间:{}", fromTime, toTime);
        } else {
            //获取当前时间整点的时间戳
            toTime = getNowHourTime();
            //获取上一个小时的时间戳
            fromTime = getNextTimeHourTime(toTime, -1);
            log.info("[统计当前时间的进出场流量],开始时间:{},结束时间:{}", fromTime, toTime);
        }
        //统计进口车辆
        List<ParkEnterOrexitNumDto> parkEnterNumDtos = orderInfoDao.countHoursEnterNum(fromTime, toTime - 1);
        //统计出口车辆
        List<ParkEnterOrexitNumDto> parkExitNumDtos = orderInfoDao.countHoursExitNum(fromTime, toTime - 1);

        if (CollectionUtils.isEmpty(parkEnterNumDtos) && CollectionUtils.isEmpty(parkExitNumDtos)) {
            return ObjectResponse.success(null);
        }
        //更新或添加进口数据
        if (!CollectionUtils.isEmpty(parkEnterNumDtos)) {
            parkEnterNumDtos.forEach(parkEnterOrexitNumDto -> {
                //获取统计当前的时间点
                Date current = new Date(fromTime * 1000);
                //插入数据
                OrderEnexCount orderEnexCount = new OrderEnexCount();
                orderEnexCount.setParkId(parkEnterOrexitNumDto.getParkId());
                orderEnexCount.setTotalEnter(parkEnterOrexitNumDto.getTotalEnter());
                orderEnexCount.setTotalExit(0);
                orderEnexCount.setCountTime(current);
                OrderEnexCount count = orderEnexCountDao.selectByCountTime(parkEnterOrexitNumDto.getParkId(), current);
                if (Objects.isNull(count)) {
                    orderEnexCountDao.insert(orderEnexCount);
                } else {
                    count.setTotalEnter(parkEnterOrexitNumDto.getTotalEnter());
                    orderEnexCountDao.update(count);
                }
            });
        }

        //更新或添加出口数据
        if (!CollectionUtils.isEmpty(parkExitNumDtos)) {
            parkExitNumDtos.forEach(orexitNumDto -> {
                //获取统计当前的时间点
                Date current = new Date(fromTime * 1000);
                // 插入数据
                OrderEnexCount orderEnexCount = new OrderEnexCount();
                orderEnexCount.setParkId(orexitNumDto.getParkId());
                orderEnexCount.setTotalEnter(0);
                orderEnexCount.setTotalExit(orexitNumDto.getTotalExit());
                orderEnexCount.setCountTime(current);
                OrderEnexCount count = orderEnexCountDao.selectByCountTime(orexitNumDto.getParkId(), current);
                if (Objects.isNull(count)) {
                    orderEnexCountDao.insert(orderEnexCount);
                } else {
                    count.setTotalExit(orexitNumDto.getTotalExit());
                    orderEnexCountDao.update(count);
                }
            });
        }
        return ObjectResponse.success(null);
    }

    @Override
    @DS_SLAVE
    @SentinelResource(value = "OrderEnexService.getEnterMatchList", defaultFallback = "defaultFallbackHandle", fallbackClass = {ExceptionUtils.class})
    public ObjectResponse<List<EnterMatchDto>> getEnterMatchList(OrderEnexQueryRequest orderEnexQueryRequest) {
        String parkCode = orderEnexQueryRequest.getParkCode();
        Date startTime = orderEnexQueryRequest.getStartTime();
        Date endTime = orderEnexQueryRequest.getEndTime();
        Boolean isNoPlate = orderEnexQueryRequest.getIsNoPlate();
        String plateNum = orderEnexQueryRequest.getPlateNum();
        Park park = parkService.findByParkCode(parkCode).getData();
        if (park == null) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }
        //PageHelper.startPage(orderEnexQueryRequest.getPageNo(),orderEnexQueryRequest.getPageSize());
        List<String> plateNums = new ArrayList<>();
        if (plateNum != null && plateNum.length() >= 7) {
            plateNums = getPlateNums(plateNum);
            plateNum= null;
        }
        //获取入场记录
        List<EnterMatchDto> enterMatchDtos = orderInfoDao.selectEnterMatchListNew(park.getId(),
                Objects.isNull(startTime) ? null : startTime.getTime() / 1000, Objects.isNull(endTime) ? null : endTime.getTime() / 1000,
                plateNum, isNoPlate, plateNums);
        enterMatchDtos.removeIf(Objects::isNull);
        if (CollectionUtils.isEmpty(enterMatchDtos)) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }

        //过滤并分页
        Map<String, EnterMatchDto> collect = enterMatchDtos.stream().collect(Collectors.groupingBy(EnterMatchDto::getPlateNumber,
                Collectors.collectingAndThen(Collectors.toList(), value -> value.get(0))));
        enterMatchDtos = new ArrayList<>(collect.values());
        enterMatchDtos.sort(Comparator.comparing(EnterMatchDto::getEnterTimeLong).reversed());
        int pageNo = NumberUtils.toPrimitive(orderEnexQueryRequest.getPageNo(), 1);
        int pageSize = NumberUtils.toPrimitive(orderEnexQueryRequest.getPageSize(), 10);
        int s = (pageNo - 1) * pageSize;
        int e = Math.min((pageNo - 1) * pageSize + pageSize, enterMatchDtos.size());
        if (s > enterMatchDtos.size()) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }
        List<EnterMatchDto> dataList = enterMatchDtos.subList(s, e);
        if (CollectionUtils.isEmpty(dataList)) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }
        //查询
        List<String> orderNums = dataList.stream().map(EnterMatchDto::getOrderNum).collect(Collectors.toList());
        //根据 orderNums 获取 url
        List<OrderCarInfo> carInfos = orderCarInfoDao.selectList(Wrappers.lambdaQuery(OrderCarInfo.class).select(OrderCarInfo::getOrderNum, OrderCarInfo::getEnterImage).in(OrderCarInfo::getOrderNum, orderNums));
        if (CollectionUtils.isEmpty(carInfos)) {
            return ObjectResponse.failed(CodeConstantsEnum.ERROR_404);
        }
        Map<String, String> enterImageMap = carInfos.stream().collect(Collectors.toMap(OrderCarInfo::getOrderNum, carInfo -> carInfo.getEnterImage() == null ? "" : carInfo.getEnterImage()));
        dataList.forEach(enterMatchDto -> {
            String imageUrl = enterImageMap.get(enterMatchDto.getOrderNum());
            if (StringUtils.isNotBlank(imageUrl)) {
                enterMatchDto.setImgUrl(ossService.getImageUrl(imageUrl));
            }
            // 计算停车时长
            if (enterMatchDto.getEnterTimeLong() != null) {
                enterMatchDto.setParkTime(DateTools.secondToSecondsTime(Math.toIntExact(System.currentTimeMillis() / 1000 - enterMatchDto.getEnterTimeLong())));
            }
        });
        return ObjectResponse.success(dataList);
    }

    public List<String> getPlateNums(String plateNum) {
        List<String> lists = new ArrayList<>();
        String str = StringUtils.substring(plateNum, 2, 7);
        lists.add(StringUtils.substring(str, 0, 3));
        lists.add(StringUtils.substring(str, 1, 4));
        lists.add(StringUtils.substring(str, 2, 5));
        return lists;
    }
    @Override
    public ObjectResponse<OrderCarInfo> getCarInfo(String orderNum, Long parkId) {
        OrderInfo orderInfo = new OrderInfo();
        orderInfo.setParkId(parkId);
        orderInfo.setOrderNum(orderNum);
        orderInfo = orderInfoDao.selectLimitOneOrderByEnterDesc(orderInfo);
        if (orderInfo == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        OrderCarInfo carInfo = orderCarInfoDao.selectByOrderNum(orderNum);
        if (carInfo == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        return ObjectResponse.success(carInfo);
    }

    @Override
    public ObjectResponse<String> addCarInfo(OrderCarInfo carInfo) {
        if (StringUtils.isBlank(carInfo.getOrderNum())) {
            carInfo.setOrderNum(CodeTools.GenerateOrderNum());
        }
        orderCarInfoDao.insert(carInfo);
        return ObjectResponse.success(carInfo.getOrderNum());
    }

    @Override
    public ObjectResponse<Integer> updateCarInfo(OrderCarInfo carInfo) {
        int lines = orderCarInfoDao.updateByOrderNum(carInfo);
        if (lines > 0) {
            return ObjectResponse.success(lines);
        } else {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    @DS_SLAVE
    public ObjectResponse<List<OrderCarInfo>> getCarInfoList(Collection<String> orderNums, Long parkId) {
        List<OrderCarInfo> orderCarInfos = orderCarInfoDao.selectList(Wrappers.lambdaQuery(OrderCarInfo.class).in(OrderCarInfo::getOrderNum, orderNums));
        return ObjectResponse.success(orderCarInfos);
    }



    @Override
    public ObjectResponse<OrderSonCarInfo> getOrderSonCarInfo(Long orderSonId) {
        OrderSonCarInfo orderSonCarInfo = orderSonCarInfoDao.selectByOrderSonId(orderSonId);
        return ObjectResponse.returnNotFoundIfNull(orderSonCarInfo);
    }

    @Override
    public OrderCarInfo getByOrderNumWithHistory(String orderNum) {
        OrderCarInfo orderCarInfo = orderCarInfoDao.selectByOrderNum(orderNum);
        if (orderCarInfo != null) {
            return orderCarInfo;
        }
        Date startDate = DateUtil.offsetDay(DateUtil.date(), -90);
        List<String> yearQuarterRangeTableName = DateRangeUtils.getYearQuarterRangeTableName(startDate);
        return orderCarInfoDao.selectByOrderNumListWithHistory(orderNum, yearQuarterRangeTableName);
    }

    @Override
    public OrderCarInfo getByOrderNum(String orderNum) {
        return orderCarInfoDao.selectByOrderNum(orderNum);
    }

    @Override
    public int updateByIdWithHistory(OrderCarInfo updateCarInfo) {
        int i = orderCarInfoDao.updateById(updateCarInfo);
        if (i == 0) {
            Date startDate = DateUtil.offsetDay(DateUtil.date(), -90);
            List<String> yearQuarterRangeTableName = DateRangeUtils.getYearQuarterRangeTableName(startDate);
            for (String table : yearQuarterRangeTableName) {
                i = orderCarInfoDao.updateByIdWithHistory(updateCarInfo, table);
                if (i > 0) {
                    break;
                }
            }
        }
        return i;
    }

    /**
     * 统计出入口流量
     *
     * @param parkIdList
     * @param day        1=近24小时 2=昨天 3=今天 7=7天 30=30天
     * @param monthCount 指定月
     * @return
     */
    private List<ParkEnterOrexitCountDto> countEnterOrExitNum(String parkIdList, Integer day, String monthCount) {
        List<ParkEnterOrexitCountDto> resluts = Lists.newArrayList();
        List<ParkEnterOrexitCountDto> parkEnterOrexitCountDtos;
        Long fromTime;
        long toTime;
        if (day <= 3 && day != 0) {
            //获取近24小时的进出口车辆
            fromTime = get24HourTime();
            List<Integer> hours = get24Hours();
            toTime = getNowHourTime();
            if (day == 2) {//昨天
                fromTime = getYesterdayHourTime(24 + getHour(new Date()));
                toTime = getYesterdayHourTime(getHour(new Date())) - 1;
                hours = get24DayHours();
            }
            if (day == 3) {//今天
                fromTime = getNHourTime(getHour(new Date()));
                toTime = getNowHourTime();
                hours = getNowHours();
            }

            String cacheKey = "countHoursEnterOrExitNum:" + parkIdList + "_" + fromTime + "_" + toTime;
            String finalFromTime = DateTools.getFormat(DateTools.DF, new Date(fromTime * 1000));
            String finalToTime = DateTools.getFormat(DateTools.DF, new Date(toTime * 1000));
            parkEnterOrexitCountDtos = redisHandle.cacheList(cacheKey, ParkEnterOrexitCountDto.class,
                    () -> orderEnexCountDao.countHoursEnterOrExitNum(parkIdList, finalFromTime, finalToTime),
                    3600000);
            hours.forEach(s -> {
                ParkEnterOrexitCountDto ret = new ParkEnterOrexitCountDto();
                ret.setTimeRange(s + ":00-" + (s + 1) + ":00");
                if (CollectionUtils.isEmpty(parkEnterOrexitCountDtos)) {
                    ret.setTotalEnter(0);
                    ret.setTotalExit(0);
                } else {
                    ParkEnterOrexitCountDto res = parkEnterOrexitCountDtos.stream()
                            .filter(enex -> Integer.valueOf(enex.getTimeRange()).equals(s))
                            .findFirst()
                            .orElse(null);
                    if (Objects.nonNull(res)) {
                        ret.setTotalExit(res.getTotalExit());
                        ret.setTotalEnter(res.getTotalEnter());
                    } else {
                        ret.setTotalExit(0);
                        ret.setTotalEnter(0);
                    }
                }
                resluts.add(ret);
            });
            return resluts;
        } else {
            //获取近7/30天的进出口车辆
            fromTime = getDayTime(day);
            toTime = getTodayTime();
            List<String> days = getDays(day);
            if (StringUtils.isNotEmpty(monthCount)) {
                try {
                    SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
                    fromTime = sdf.parse(monthCount + "-01 00:00:00").getTime() / 1000;
                    SimpleDateFormat sdf2 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    toTime = sdf2.parse(monthCount + "-" + getDaysOfMonth(sdf.parse(monthCount + "-01")) + " 23:59:59").getTime() / 1000;
                    days = getMonthDays(monthCount);
                } catch (Exception e) {
                    log.error("[统计出入口流量]处理失败: {}. parkIdList[{}], day[{}], monthCount[{}]", e.getMessage(), parkIdList, day, monthCount, e);
                }
            }
            String cacheKey = "countDaysEnterOrExitNum:" + parkIdList + "_" + fromTime + "_" + toTime;
            String finalFromTime = DateTools.getFormat(DateTools.DF, new Date(fromTime * 1000));
            String finalToTime = DateTools.getFormat(DateTools.DF, new Date(toTime * 1000));
            parkEnterOrexitCountDtos = redisHandle.cacheList(cacheKey, ParkEnterOrexitCountDto.class,
                    () -> orderEnexCountDao.countDaysEnterOrExitNum(parkIdList, finalFromTime, finalToTime),
                    3600000);
            if (parkEnterOrexitCountDtos.size() > 0) {
                days.forEach(s -> {
                    ParkEnterOrexitCountDto ret = new ParkEnterOrexitCountDto();
                    ret.setTimeRange(s);
                    if (CollectionUtils.isEmpty(parkEnterOrexitCountDtos)) {
                        ret.setTotalEnter(0);
                        ret.setTotalExit(0);
                    } else {
                        ParkEnterOrexitCountDto res = parkEnterOrexitCountDtos.stream()
                                .filter(enex -> enex.getTimeRange().equals(s))
                                .findFirst()
                                .orElse(null);
                        if (Objects.nonNull(res)) {
                            ret.setTotalExit(res.getTotalExit());
                            ret.setTotalEnter(res.getTotalEnter());
                        } else {
                            ret.setTotalExit(0);
                            ret.setTotalEnter(0);
                        }
                    }
                    resluts.add(ret);
                });
            }
            return resluts;
        }
    }
}
