package com.icetech.order.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.common.domain.Page;
import com.icetech.common.utils.NumberUtils;
import com.icetech.order.dao.OrderBackMapper;
import com.icetech.order.domain.dto.OrderBackQueryDTO;
import com.icetech.order.domain.entity.OrderBack;
import com.icetech.order.service.OrderBackService;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.third.service.third.MqPushService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * 折返回场记录表 服务实现类
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author fangct
 * @version 1.0 @date 2023-07-20
 */
@Service
public class OrderBackServiceImpl extends BaseServiceImpl<OrderBackMapper, OrderBack> implements OrderBackService {
    @Autowired
    private MqPushService mqPushService;

    /**
     * Description: 根据ID查询 折返回场记录表对象信息 <br>
     * Version1.0 2023-07-20 by fangct创建
     *
     * @param id 对象id
     * @return OrderBack
     */
    @Override
    public OrderBack getOrderBackById (Long id) {
        return getById(id);
    }

    /**
     * Description: 新增  折返回场记录表对象信息 <br>
     * Version1.0 2023-07-20 by fangct创建
     *
     * @param entity 对象信息
     * @return Boolean 返回新增后主键
     */
    @Override
    public Boolean addOrderBack (OrderBack entity) {
        boolean result = save(entity);
        mqPushService.pushOrderBack(entity);
        return result;
    }

    /**
     * Description: 修改 折返回场记录表对象信息 <br>
     * Version1.0 2023-07-20 by fangct创建
     *
     * @param entity 对象信息
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean modifyOrderBack (OrderBack entity) {
        return updateById(entity);
    }

    /**
     * Description: 删除  折返回场记录表对象信息 <br>
     * Version1.0 2023-07-20 by fangct创建
     *
     * @param id 对象id
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean removeOrderBackById (Long id) {
        return removeById(id);
    }

    @Override
    public Page<OrderBack> getPageList(OrderBackQueryDTO queryDTO, List<Long> parkIds) {
        QueryWrapper<OrderBack> wrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(parkIds)){
            wrapper.in(org.apache.commons.collections4.CollectionUtils.isNotEmpty(parkIds) && !parkIds.contains(-1L),"park_id",parkIds);
        }
        wrapper.like(StringUtils.isNotEmpty(queryDTO.getPlateNum()),"plate_num",queryDTO.getPlateNum());
        wrapper.eq("inandout_type",queryDTO.getInandoutType());
        if (NumberUtils.toPrimitive(queryDTO.getInandoutType(), 2) == 1){
            wrapper.eq("review_status",0);
        }
        wrapper.between(Objects.nonNull(queryDTO.getStartDate()) && Objects.nonNull(queryDTO.getEndDate()),"create_time",queryDTO.getStartDate(),queryDTO.getEndDate());
        wrapper.orderByDesc("id");
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<OrderBack> page = page(wrapper, queryDTO.getPageNo(), queryDTO.getPageSize());

        return Page.instance(Math.toIntExact(page.getPages()),page.getTotal(),page.getRecords());
    }

    @Override
    public Integer getTotal(OrderBackQueryDTO queryDTO, List<Long> parkIds) {
        QueryWrapper<OrderBack> wrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(parkIds)){
            wrapper.in(org.apache.commons.collections4.CollectionUtils.isNotEmpty(parkIds) && !parkIds.contains(-1L),"park_id",parkIds);
        }
        wrapper.like(StringUtils.isNotEmpty(queryDTO.getPlateNum()),"plate_num",queryDTO.getPlateNum());
        wrapper.eq("inandout_type",queryDTO.getInandoutType());
        if (NumberUtils.toPrimitive(queryDTO.getInandoutType(), 2) == 1){
            wrapper.eq("review_status",0);
        }
        wrapper.between(Objects.nonNull(queryDTO.getStartDate()) && Objects.nonNull(queryDTO.getEndDate()),"create_time",queryDTO.getStartDate(),queryDTO.getEndDate());
        return count(wrapper);
    }

    @Override
    public List<OrderBack> getListByOrderNum(String orderNum) {
        return list(Wrappers.lambdaQuery(OrderBack.class)
                .eq(OrderBack::getOrderNum, orderNum));
    }

}
