package com.icetech.order.dao;

import java.util.List;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.cloudcenter.domain.request.OrderQueryRequest;
import com.icetech.order.domain.dto.OrderInfoDto;
import com.icetech.order.domain.entity.OrderExitEnterTimeVO;
import com.icetech.cloudcenter.domain.order.OrderVO;
import com.icetech.cloudcenter.domain.request.FlowRequest;
import com.icetech.cloudcenter.domain.response.EnexDetailDto;
import com.icetech.cloudcenter.domain.response.EnterMatchDto;
import com.icetech.cloudcenter.domain.response.ParkEnterOrexitNumDto;
import com.icetech.cloudcenter.domain.order.update.OrderInfoUpdate;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.common.domain.response.PageQuery;
import com.icetech.db.mybatis.base.mapper.SuperMapper;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.stereotype.Repository;

/**
 * Description : 订单信息表操作DAO
 *
 * @author fangct
 */
@Repository
public interface OrderInfoDao extends SuperMapper<OrderInfo> {
    default OrderInfo selectByOrderNum(String orderNum) {
        return selectOne(Wrappers.lambdaQuery(OrderInfo.class).eq(OrderInfo::getOrderNum, orderNum));
    }

    default OrderInfo selectLimitOneOrderByEnterDesc(OrderInfo entity) {
        return selectLimitOne(Wrappers.lambdaQuery(entity).orderByDesc(OrderInfo::getEnterTime).orderByDesc(OrderInfo::getId));
    }

    default OrderInfo selectLimitOneNotOrderByEnterDesc(OrderInfo entity, String orderNum) {
        return selectLimitOne(Wrappers.lambdaQuery(entity).notIn(OrderInfo::getOrderNum, orderNum)
                .orderByDesc(OrderInfo::getEnterTime).orderByDesc(OrderInfo::getId));
    }

    default List<OrderInfo> selectListOrderByEnterDesc(OrderInfo entity) {
        return selectList(Wrappers.lambdaQuery(entity).orderByDesc(OrderInfo::getEnterTime).orderByDesc(OrderInfo::getId));
    }

    default OrderInfo selectLimitOneByIdDesc(OrderInfo entity) {
        return selectLimitOne(Wrappers.lambdaQuery(entity).orderByDesc(OrderInfo::getEnterTime).orderByDesc(OrderInfo::getId));
    }

    List<OrderInfo> selectPageList(PageQuery<OrderInfo> query);

    default int updateByOrderNum(OrderInfo orderInfo) {
        Long id = orderInfo.getId();
        String orderNum = orderInfo.getOrderNum();
        orderInfo.setId(null);
        orderInfo.setOrderNum(null);
        int result = update(orderInfo, Wrappers.lambdaUpdate(OrderInfo.class).eq(OrderInfo::getOrderNum, orderNum));
        orderInfo.setId(id);
        orderInfo.setOrderNum(orderNum);
        return result;
    }

    default int updateByOrderNumAndRegionId(OrderInfo orderInfo) {
        Long id = orderInfo.getId();
        String orderNum = orderInfo.getOrderNum();
        Long regionId = orderInfo.getRegionId();
        orderInfo.setId(null);
        orderInfo.setOrderNum(null);
        orderInfo.setRegionId(null);
        int result = update(orderInfo, Wrappers.lambdaUpdate(OrderInfo.class)
                .eq(OrderInfo::getOrderNum, orderNum)
                .eq(OrderInfo::getRegionId, regionId)
        );
        orderInfo.setId(id);
        orderInfo.setOrderNum(orderNum);
        orderInfo.setRegionId(regionId);
        return result;
    }

    default int insertWithPlateNum2(OrderInfo orderInfo) {
        orderInfo.setPlateNum2(StringUtils.substring(orderInfo.getPlateNum(), 1));
        return insert(orderInfo);
    }

    /**
     * 查询在场车牌列表
     *
     * @param entity    parkId、serviceStatus、type为条件
     * @param plateList 车牌列表
     * @return 结果
     */
    default List<OrderInfo> selectListByPlates(OrderInfo entity, List<String> plateList) {
        return selectList(Wrappers.lambdaQuery(entity)
                .eq(OrderInfo::getParkId, entity.getParkId())
                .eq(OrderInfo::getServiceStatus, entity.getServiceStatus())
                .eq(entity.getType() != null, OrderInfo::getType, entity.getType())
                .eq(entity.getRegionId() != null && entity.getRegionId() != 0, OrderInfo::getRegionId, entity.getRegionId())
                .in(OrderInfo::getPlateNum, plateList)
                .orderByAsc(OrderInfo::getEnterTime));
    }

    /**
     * 查询多车场的在场车牌
     *
     * @param entity    parkId、serviceStatus、type为条件
     * @param plateList 车牌列表
     * @return 结果
     */
    default List<OrderInfo> selectListByPlates(OrderInfo entity, List<Long>parkIds, List<String> plateList) {
        return selectList(Wrappers.lambdaQuery(entity)
                .in(OrderInfo::getParkId, parkIds)
                .eq(OrderInfo::getServiceStatus, entity.getServiceStatus())
                .eq(entity.getType() != null, OrderInfo::getType, entity.getType())
                .eq(entity.getRegionId() != null && entity.getRegionId() != 0, OrderInfo::getRegionId, entity.getRegionId())
                .in(OrderInfo::getPlateNum, plateList)
                .orderByAsc(OrderInfo::getEnterTime));
    }

    /**
     * 查询需要人修改车牌的订单
     *
     * @return
     */
    List<OrderInfo> selectMadeOrders(@Param("parkId") String parkId, @Param("startTime") String startTime);

    /**
     * 查询人工修改
     *
     * @param orderNum
     * @return
     */
    String selectMadeNum(@Param("orderNum") String orderNum);

    /**
     * 查询在场记录
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @param plateNumber
     * @return
     */
    Integer countEnterRecords(@Param("parkId") String parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber, @Param("type") List<Integer> type, @Param("reliability") Integer reliability);

    /**
     * 离场车辆总条数
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @param plateNumber
     * @param type
     * @return
     */
    Integer countExitRecords(@Param("parkId") String parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber, @Param("type") List<Integer> type, @Param("enterStartTime") Long enterStartTime, @Param("enterEndTime") Long enterEndTime, @Param("tagIds") List<Integer> tagIds);

    Integer countExitRecordsFromStartTime(@Param("parkId") Long parkId, @Param("startTime") Long startTime, @Param("plateNumList") List<String> plateNumList);

    /**
     * 修改状态
     *
     * @param orderInfoUpdate
     * @return
     */
    Integer updateStatus(OrderInfoUpdate orderInfoUpdate);

    /**
     * 查询在场记录
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @param plateNumber
     * @param channelCodes
     * @return
     */
    List<OrderInfo> selectEnterRecords(@Param("parkId") String parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber, @Param("type") List<Integer> type, @Param("reliability") Integer reliability,
                                       @Param("tagIds") List<Integer> tagIds, @Param("channelCodes") List<String> channelCodes);

    /**
     * 查询离场记录
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @param plateNumber
     * @param freeFlag
     * @param channelCodes
     * @param plateNums
     * @return
     */
    List<OrderInfo> selectExitRecords(@Param("parkId") String parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber,
                                      @Param("type") List<Integer> type, @Param("enterStartTime") Long enterStartTime, @Param("enterEndTime") Long enterEndTime, @Param("noplateFee") Integer noplateFee, @Param("orderNum") String orderNum,
                                      @Param("tagIds") List<Integer> tagIds, @Param("freeFlag") Integer freeFlag, @Param("channelCodes") List<String> channelCodes, @Param("plateNums") List<String> plateNums);

    List<OrderInfo> countParkingTime(@Param("parkId") String parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime);

    List<OrderExitEnterTimeVO> parkingTimeStatistics(@Param("startTime") Long startTime, @Param("endTime") Long endTime, @Param("parkId") Long parkId);

    /**
     * 模糊匹配车牌
     *
     * @param parkId
     * @param plateNumPart
     * @returnplateNumPart
     */
    OrderInfo fuzzyOrderByPlate(@Param("parkId") long parkId, @Param("plateNumPart") String plateNumPart, @Param("status") int status);

    /**
     * 模糊匹配车牌
     *
     * @param parkId
     * @param plateNums
     * @returnplateNumPart
     */
    OrderInfo fuzzyInParkPlate(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums);

    OrderInfo fuzzyInParkPlate2(@Param("parkId") Long parkId, @Param("plateNumPart") String plateNumPart);

    /**
     * 模糊查询最近的离场车牌
     *
     * @param parkId
     * @param plateNumPart
     * @return
     */
    OrderInfo fuzzyOutParkPlate(@Param("parkId") Long parkId, @Param("plateNumPart") String plateNumPart);

    /**
     * 匹配入场记录
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @param plateNumber
     * @param isNoPalte
     * @return
     */
    List<EnterMatchDto> selectEnterMatchList(@Param("parkId") Long parkId, @Param("startTime") Long startTime,
                                             @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber,
                                             @Param("isNoPalte") Boolean isNoPalte);

    List<EnterMatchDto> selectEnterMatchListNew(@Param("parkId") Long parkId, @Param("startTime") Long startTime,
                                                @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber,
                                                @Param("isNoPalte") Boolean isNoPalte, @Param("plateNums") List<String> plateNums);

    /**
     * 查询时间段内订单
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @param plateNumber
     * @return
     */
    List<OrderInfo> selectListByParam(@Param("parkId") Long parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime, @Param("plateNumber") String plateNumber);

    /**
     * 统计在场车辆数量
     *
     * @param parkId
     * @return
     */
    Integer countEnterCar(Long parkId);

    /**
     * 监控大屏今日进总条数
     *
     * @param request
     * @return
     */
    long enterCount(FlowRequest request);

    /**
     * 今日入场车辆数
     *
     * @param request
     * @return
     */
    long enterInparkCount(FlowRequest request);

    /**
     * 监控大屏今日离总条数
     *
     * @param request
     * @return
     */
    long exitCount(FlowRequest request);

    List<EnexDetailDto> enterDetail(FlowRequest request);

    List<EnexDetailDto> exitDetail(FlowRequest request);

    /**
     * 按小时时间段返回进场车辆统计
     *
     * @param fromTime
     * @param toTime   截至时间戳
     * @return
     */
    List<ParkEnterOrexitNumDto> countHoursEnterNum(@Param("fromTime") long fromTime, @Param("toTime") long toTime);

    /**
     * 按小时时间段返回出场车辆统计
     *
     * @param fromTime
     * @param toTime   截至时间戳
     * @return
     */
    List<ParkEnterOrexitNumDto> countHoursExitNum(@Param("fromTime") long fromTime, @Param("toTime") long toTime);

    /**
     * 根据多个车牌号查询在场记录
     *
     * @param parkId
     * @param plateNums
     * @return
     */
    int countInParkMonthCardByPlateNums(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums);

    /**
     * 根据多个车牌号查询在场记录(集团月卡多车场)
     * @param parkIds
     * @param plateNums
     * @return
     */
    int countInMoreParkMonthCardByPlateNums(@Param("parkIds") List<Long> parkIds, @Param("plateNums") List<String> plateNums);

    /**
     * 统计场中场月卡在场记录
     *
     * @param parkId    车场ID
     * @param plateNums 车牌列表
     * @return 在场数
     */
    int countMultipleAreaInParkByPlateNums(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums, @Param("regionId") Long regionId);

    int countMultipleAreaExitRecordsFromStartTime(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums, @Param("regionId") Long regionId,
                                                  @Param("startTime") Long startTime);

    /**
     * 批量更新订单为离场状态
     *
     * @param orderInfos
     * @return
     */
    int batchExitOrder(@Param("orderInfos") List<OrderInfo> orderInfos);

    /**
     * 查询订单数
     *
     * @return
     */
    Integer findTheMaxIdByDate(@Param("parkId") Long parkId, @Param("startTime") String startTime, @Param("endTime") String endTime);

    /**
     * 某时间段进场且正常离场的订单数
     *
     * @param parkId
     * @param startTime
     * @param endTime
     * @return
     */
    Integer countRecentExitByEnterTime(@Param("parkId") Long parkId, @Param("plateNum") String plateNum, @Param("startTime") Long startTime, @Param("endTime") Long endTime);

    int countEnexTimesByEnterTime(@Param("parkId") Long parkId, @Param("plateNum") String plateNum, @Param("startTime") Long startTime, @Param("endTime") Long endTime);

    /**
     * 获取盘点可以清理数据数量
     *
     * @param parkId
     * @param createTime
     * @return
     */
    int selectSurplusCount(@Param("parkId") Long parkId, @Param("createTime") String createTime);

    /**
     * 获取盘点可以清理数据列表
     *
     * @param parkId
     * @param createTime
     * @return
     */
    List<OrderVO> selectSurplusList(@Param("parkId") Long parkId, @Param("createTime") String createTime);

    /**
     * 查询含历史表的订单信息
     *
     * @param tables   历史表
     * @param orderNum 订单号
     * @return 订单信息
     */
    OrderInfo selectWithHistoryByOrderNum(@Param("tables") List<String> tables, @Param("orderNum") String orderNum);

    /**
     * 查询最近的历史表订单
     *
     * @param tables
     * @param parkId
     * @param plateNums
     * @return
     */
    OrderInfo fuzzyRecentHistoryOrder(@Param("tables") List<String> tables, @Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums);

    OrderInfo fuzzyByPlateNums(@Param("parkId") Long parkId, @Param("plateNums") List<String> plateNums);

    List<OrderInfo> selectExitOrderBy(@Param("parkId") Long parkId, @Param("startTime") Long startTime, @Param("endTime") Long endTime);

    List<OrderInfo> selectWithHistoryByOrderNumList(@Param("tables") List<String> tables, @Param("orderNums") List<String> orderNums);

    OrderInfo selectByLocalOrderNumWithHistory(@Param("parkId") Long parkId, @Param("tables") List<String> tables, @Param("localOrderNum") String localOrderNum);

    int updateHistoryTable(@Param("tableName") String tableName, @Param("orderInfo") OrderInfo orderInfo);

    Integer selectChargeLiveCount(@Param("parkId") Long parkId);

    Integer selectOilLiveCount(@Param("parkId") Long parkId);

    /**
     * 获取在场车辆车牌
     *
     * @param parkId
     * @param limitNum
     * @return
     */
    List<String> selectLivePlateNum(@Param("parkId") Long parkId, @Param("limitNum") Integer limitNum);

    List<OrderInfoDto> selectMpOrderInfos(OrderQueryRequest request);
}
