package com.icetech.fee.service.storecard.impl;

import com.github.pagehelper.PageHelper;
import com.icetech.cloudcenter.domain.request.ExitCommonRequest;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.cloudcenter.api.store.StoreCardService;
import com.icetech.common.utils.NumberUtils;
import com.icetech.order.domain.dto.StoredCardRecordDto;
import com.icetech.third.anno.DS_SLAVE;
import com.icetech.third.dao.send.SendinfoDao;
import com.icetech.fee.dao.storecard.StoreCardDao;
import com.icetech.fee.dao.storecard.StoreCardPlateDao;
import com.icetech.fee.dao.storecard.StoreCardUsedDao;
import com.icetech.fee.dao.storecard.StoredCardRecordDao;
import com.icetech.fee.domain.entity.storecard.StoreCard;
import com.icetech.fee.domain.entity.storecard.StoreCardPlate;
import com.icetech.fee.domain.entity.storecard.StoreCardRecord;
import com.icetech.fee.domain.entity.storecard.StoreCardUsed;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.common.constants.CodeConstants;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.DateTools;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

/**
 *
 *
 * @author zhangpengzhan
 * @date 2020/8/3 14:16  日期
 * @description: 储值卡业务操作服务类
 */
@Slf4j
@Service("StoreCardService")
public class StoreCardServiceImpl implements StoreCardService {

    /**
     * 储值卡表操作
     */
    @Autowired
    private StoreCardDao storeCardDao;

    /**
     * 储值卡车牌绑定表操作
     */
    @Autowired
    private StoreCardPlateDao storeCardPlateDao;
    @Resource
    private StoredCardRecordDao storedCardRecordDao;
    @Resource
    private StoreCardUsedDao storeCardUsedDao;
    @Resource
    private SendinfoDao sendinfoDao;

    /**
     * 获取储值卡详情
     * @param cardId  储值卡ID
     * @return 查询结果
     */
    @Override
    public ObjectResponse<StoreCard> getStoreCardDetail(Integer cardId) {
        log.info("getStoreCardDetail======>{}", cardId);
        if (null == cardId) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        StoreCard storeCard = storeCardDao.selectByCardId(cardId);
        if (storeCard == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "储值卡[" + cardId + "]信息不存在");
        }
        return ObjectResponse.success(storeCard);
    }

    /**
     * 根据车牌储值卡绑定信息批量查询储值卡
     * @param storeCardPlateList
     * @return
     */
    @Override
    public ObjectResponse<List<StoreCard>> getStoreCardList(List<StoreCardPlate> storeCardPlateList) {
        log.info("getStoreCardList===========>{}",storeCardPlateList);
        if (null == storeCardPlateList || storeCardPlateList.size() == 0){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        List<StoreCard> storeCardList = storeCardDao.selectByCardPlates(storeCardPlateList);
        boolean isSuccess = null != storeCardList && storeCardList.size() > 0;
        return new ObjectResponse<>(isSuccess ? CodeConstants.SUCCESS : CodeConstants.ERROR_404,
                isSuccess ? CodeConstants.getName(CodeConstants.SUCCESS) : CodeConstants.getName(CodeConstants.ERROR_404), storeCardList);
    }

    /**
     * 获取有效期内的储值卡
     * @param plate
     * @param parkId
     * @return
     */
    @Override
    public ObjectResponse<StoreCard> getValidStoreCards(String plate, Long parkId) {
        //查询储值卡
        ObjectResponse<List<StoreCardPlate>> objectResponse = getStoreCardPlate(plate);
        //匹配到储值卡
        StoreCard storeCard = null;
        if (CodeConstants.SUCCESS.equals(objectResponse.getCode())){
            //查储值卡信息
            List<StoreCardPlate> storeCardPlate = objectResponse.getData();
            if (null != storeCardPlate && storeCardPlate.size() >0) {
                //匹配储值卡信息
                ObjectResponse<List<StoreCard>> storeCardObjectResponse = getStoreCardList(storeCardPlate);
                if (CodeConstants.SUCCESS.equals(storeCardObjectResponse.getCode())) {
                    List<StoreCard> storeCardList = storeCardObjectResponse.getData();
                    if (null != storeCardList && storeCardList.size() > 0) {
                        Date nowDate = new Date();
                        for (StoreCard item : storeCardList
                        ) {
                            if (null != item && null != parkId && (item.getParkId().equals(parkId.intValue()))) {
                                storeCard = item;
                            }
                        }
                        if (null != storeCard && storeCard.getStartTime().getTime() <= nowDate.getTime() && dateCompare(nowDate, storeCard.getEndTime()) < 1) {
                            //储值卡有效
                            return ObjectResponse.success(storeCard);

                        }
                    }
                }
            }
        }
        return ObjectResponse.failed(CodeConstants.ERROR_404);
    }

    /**
     * 根据车牌号获取绑定的储值卡号
     * @param plateNum  车牌
     * @return 查询结果
     */
    @Override
    public ObjectResponse<List<StoreCardPlate>> getStoreCardPlate(String plateNum) {
        if (null == plateNum){
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        List<StoreCardPlate> storeCardPlates = storeCardPlateDao.selectByPlate(plateNum);
        boolean isSuccess = storeCardPlates != null && storeCardPlates.size() > 0 ;
        ObjectResponse<List<StoreCardPlate>> storeCardPlateObjectResponse = new ObjectResponse<>();
        storeCardPlateObjectResponse.setData(storeCardPlates);
        storeCardPlateObjectResponse.setCode(isSuccess ? CodeConstants.SUCCESS : CodeConstants.ERROR_404);
        storeCardPlateObjectResponse.setMsg(isSuccess ? CodeConstants.getName(CodeConstants.SUCCESS) : CodeConstants.getName(CodeConstants.ERROR_404));
        return storeCardPlateObjectResponse;
    }

    /**
     * 更新储值卡余额
     * @param storeCard 需要更新的储值卡
     * @return 更新
     */
    @Override
    public ObjectResponse updateStoreCardBalance(StoreCard storeCard) {
        storeCardDao.updateBalance(storeCard);
        return ObjectResponse.success();
    }

    /**
     * 根据车牌查找储值卡并且更新储值卡扣费 流程
     * @param queryOrderFeeResponse 扣费信息
     * @param parkId 车场id
     * @return 扣费结果
     */
    @Override
    public ObjectResponse<StoreCard> chargeStoreCard(QueryOrderFeeResponse queryOrderFeeResponse,Long parkId) {
        String plate = queryOrderFeeResponse.getPlateNum();
        //查询储值卡
        ObjectResponse<List<StoreCardPlate>> objectResponse = getStoreCardPlate(plate);
        //匹配到储值卡
        StoreCard storeCard = null;
        if (CodeConstants.SUCCESS.equals(objectResponse.getCode())){
            //查储值卡信息
            List<StoreCardPlate> storeCardPlate = objectResponse.getData();
            //匹配储值卡信息
            ObjectResponse<List<StoreCard>> storeCardObjectResponse = getStoreCardList(storeCardPlate);
            if (CodeConstants.SUCCESS.equals(storeCardObjectResponse.getCode())){
                List<StoreCard> storeCardList = storeCardObjectResponse.getData();

                for (StoreCard item:storeCardList
                     ) {
                    if (null != item && parkId.equals(item.getParkId().longValue())){
                        storeCard = item;
                    }
                }
                if (null == storeCard){
                    //没有匹配到储值卡信息
                    return ObjectResponse.failed(CodeConstants.ERROR_404);
                }
                log.info("查询储值卡结果： {}",storeCard);
                Date nowDate = new Date();
                BigDecimal unPay =  new BigDecimal(queryOrderFeeResponse.getUnpayPrice());
                // 卡余额 减去 未支付费用
                BigDecimal oldBalance = storeCard.getBalance();
                BigDecimal needUpdateBalance = oldBalance.subtract(unPay);
                //判断储值卡有效期区间
                if (storeCard.getStartTime().getTime()<= nowDate.getTime() && dateCompare(nowDate,storeCard.getEndTime()) < 1 &&
                    new BigDecimal(0).compareTo(needUpdateBalance) < 1 ){
                    //储值卡有效,更新余额
                    storeCard.setBalance(needUpdateBalance);
                    storeCard.setUpdateTime(nowDate);
                    updateStoreCardBalance(storeCard);

                    // 添加储值卡使用记录
                    StoreCardUsed cardUsed = new StoreCardUsed();
                    cardUsed.setParkId(parkId);
                    cardUsed.setOrderNum(queryOrderFeeResponse.getOrderNum());
                    cardUsed.setPlateNum(queryOrderFeeResponse.getPlateNum());
                    cardUsed.setEnterTime(queryOrderFeeResponse.getEnterTime());
                    cardUsed.setExitTime(nowDate.getTime() / 1000);
                    cardUsed.setPayTime(nowDate);
                    cardUsed.setTotalPrice(Double.valueOf(queryOrderFeeResponse.getTotalAmount()));
                    cardUsed.setDiscountPrice(Double.valueOf(queryOrderFeeResponse.getDiscountAmount()));
                    cardUsed.setCardId(storeCard.getId().longValue());
                    cardUsed.setCardBeforeAmount(oldBalance.doubleValue());
                    cardUsed.setCardDeductAmount(unPay.doubleValue());
                    cardUsed.setCardAfterAmount(needUpdateBalance.doubleValue());
                    storeCardUsedDao.insert(cardUsed);
                }else {
                    //没有有效储值卡
                    return ObjectResponse.failed(CodeConstants.ERROR_404);
                }
            }else {
                //没有匹配到储值卡信息
                return ObjectResponse.failed(CodeConstants.ERROR_404);
            }

        }else {
           //没有查到储值卡绑定信息
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
        return ObjectResponse.success(storeCard);
    }

    @Override
    @Transactional
    public ObjectResponse<StoreCard> deductStoreCard(ExitCommonRequest request, ExitCommonRequest.PaidInfo paidInfo, OrderInfo orderInfo) {
        Double balancePrice = paidInfo.getBalancePrice();

        StoreCard storeCard = storeCardDao.selectValidateCard(request.getParkId(), request.getPlateNum());
        if (storeCard == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }

        BigDecimal oldBalance = storeCard.getBalance();
        BigDecimal balance = oldBalance.subtract(BigDecimal.valueOf(balancePrice));
        if (balance.compareTo(BigDecimal.ZERO) < 0) {
            balance = BigDecimal.ZERO;
            log.warn("储值卡[{}][{}][{}]余额不足", storeCard.getId(), request.getParkId(), request.getPlateNum());
        }

        storeCard.setBalance(balance);
        Date now = new Date();
        storeCard.setUpdateTime(now);
        storeCardDao.updateBalance(storeCard);

        // 添加储值卡使用记录
        StoreCardUsed cardUsed = new StoreCardUsed();
        cardUsed.setParkId(request.getParkId());
        cardUsed.setOrderNum(orderInfo.getOrderNum());
        cardUsed.setPlateNum(request.getPlateNum());
        cardUsed.setEnterTime(orderInfo.getEnterTime());
        cardUsed.setExitTime(orderInfo.getExitTime());
        cardUsed.setPayTime(new Date(paidInfo.getPayTime() * 1000));
        cardUsed.setTotalPrice(Double.valueOf(request.getTotalAmount()));
        cardUsed.setDiscountPrice(Double.valueOf(request.getDiscountAmount()));
        cardUsed.setCardId(storeCard.getId().longValue());
        cardUsed.setCardBeforeAmount(oldBalance.doubleValue());
        cardUsed.setCardDeductAmount(balancePrice);
        cardUsed.setCardAfterAmount(balance.doubleValue());
        storeCardUsedDao.insert(cardUsed);

        // 写入发送记录表以下发储值卡信息
        SendInfo sendInfo = new SendInfo();
        sendInfo.setParkId(storeCard.getParkId().longValue());
        sendInfo.setServiceType(DownServiceEnum.储值卡车辆信息.getServiceType());
        sendInfo.setServiceId(cardUsed.getId());
        sendInfo.setInclude("use");
        sendInfo.setLevel(2);
        sendInfo.setNextFireTime((int)DateTools.unixTimestamp());
        sendInfo.setStatus(0);
        sendInfo.setSendType(0);
        sendInfo.setSendNum(0);
        sendInfo.setTargetService(1);
        sendInfo.setCreateTime(now);
        sendInfo.setUpdateTime(now);
        sendinfoDao.insert(sendInfo);
        return ObjectResponse.success(storeCard);
    }

    @Override
    public ObjectResponse<BigDecimal> getStoreCardBalance(Long parkId, String plateNum) {
        StoreCard storeCard = storeCardDao.selectValidateCard(parkId, plateNum);
        if (storeCard != null){
            BigDecimal balance = storeCard.getBalance();
            return ObjectResponse.success(balance);
        }else{
            return ObjectResponse.failed(CodeConstants.ERROR_404);
        }
    }

    @Override
    public ObjectResponse<StoreCardRecord> getStoreCardRecordById(long recordId) {
        StoreCardRecord record = storedCardRecordDao.selectById(recordId);
        if (record == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "储值卡操作记录[" + recordId + "]不存在");
        }

        return ObjectResponse.success(record);
    }

    @Override
    public ObjectResponse<StoreCardUsed> getStoreCardUsedById(long usedId) {
        StoreCardUsed storeCardUsed = storeCardUsedDao.selectById(usedId);
        if (storeCardUsed == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "储值卡使用记录不存在");
        }

        return ObjectResponse.success(storeCardUsed);
    }

    @Override
    public List<StoreCardRecord> getStoreCarRecords(List<Long> parkIds,String startTime, String endTime, Integer type) {
        return storedCardRecordDao.getList(parkIds,startTime,endTime,type);
    }

    @Override
    public Integer countRecords(List<Long> parkIds, String startTime, String endTime, Integer type) {
        return storedCardRecordDao.countRecords(parkIds,startTime,endTime,type);
    }

    @Override
    public Double sumRecords(List<Long> parkIds, String startTime, String endTime, Integer type) {
        return storedCardRecordDao.sumRecords(parkIds,startTime,endTime,type);
    }

    @Override
    @DS_SLAVE
    public List<StoredCardRecordDto> getStoredCardRecords(StoredCardRecordDto dto) {
        PageHelper.startPage(NumberUtils.toPrimitive(dto.getPageNo(), 1),
            NumberUtils.toPrimitive(dto.getPageSize(), 10));
        return storedCardRecordDao.selectRecordList(dto);
    }

    @Override
    @DS_SLAVE
    public StoreCardRecord getStoredCardRecordById(Long id) {
        return storedCardRecordDao.selectById(id);
    }

    /**
     * @title: dateCompare
     * @description: 比较日期大小
     * @param date1 日期1
     * @param date2 日期2
     * @return  date1 > date2 = 1,date1 = date2 = 0, date1 < date2 = -1
     */
    public static int dateCompare(Date date1, Date date2) {
        if (date1 == null || date2 == null) {
            return 1;
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
        String dateFirst = dateFormat.format(date1);
        String dateLast = dateFormat.format(date2);
        int dateFirstIntVal = Integer.parseInt(dateFirst);
        int dateLastIntVal = Integer.parseInt(dateLast);
        if (dateFirstIntVal > dateLastIntVal) {
            return 1;
        } else if (dateFirstIntVal < dateLastIntVal) {
            return -1;
        }
        return 0;
    }
}
