package com.icetech.web.exception;

import com.alibaba.fastjson.JSON;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.web.bean.ServiceConfig;
import com.icetech.web.bean.ServiceContext;
import com.icetech.web.message.ServiceErrorEnum;
import com.icetech.web.message.ServiceErrorFactory;
import com.icetech.web.result.ServiceResultBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.util.UriUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author wgy
 */
@Slf4j
public class ExceptionHolder {
    /**
     * 与网关约定好的状态码，表示业务出错
     */
    private static final int BIZ_ERROR_CODE = 4000;

    /**
     * 与网关约定好的系统错误状态码
     */
    private static final int SYSTEM_ERROR_CODE = 5050;

    /**
     * header中的错误code
     */
    private static final String X_SERVICE_ERROR_HEADER_NAME = "x-service-error-code";

    /**
     * header中的错误信息
     */
    private static final String X_SERVICE_ERROR_MESSAGE = "x-service-error-message";

    /**
     * header中的返回信息
     */
    private static final String X_SERVICE_ERROR_RESPONSE = "x-service-error-response";

    private static final String LEFT_TOKEN = "{";
    private static final String RIGHT_TOKEN = "}";
    private static final String EQ = "=";
    private static final String COMMA = ",";
    private static final Object[] EMPTY_OBJ_ARRAY = {};

    /**
     * 处理微服务异常信息，做到不与原系统的错误处理相冲突
     *
     * @param request   request
     * @param response  response
     * @param exception exception
     */
    public static void hold(HttpServletRequest request, HttpServletResponse response, Exception exception) {
        if (exception instanceof ResponseBodyException) {
            ResponseBodyException responseBodyException = (ResponseBodyException) exception;
            exception = new ServiceException(responseBodyException.getErrCode(), responseBodyException.getMessage());
            log.warn("业务错误", exception);
        } else {
            log.error("系统错误", exception);
        }
        int code = exception instanceof ServiceException ? BIZ_ERROR_CODE : SYSTEM_ERROR_CODE;
        // 需要设置两个值，这样网关会收到错误信息
        // 并且会统计到监控当中
        response.setHeader(X_SERVICE_ERROR_HEADER_NAME, String.valueOf(code));
        String responseBody = buildResponse(request, response, exception);
        response.setHeader(X_SERVICE_ERROR_RESPONSE, UriUtils.encode(responseBody, StandardCharsets.UTF_8));

        // 如果是未知错误，还需要收集异常信息
        if (code == SYSTEM_ERROR_CODE) {
            // 扩展系统异常类型
            if (exception instanceof HttpRequestMethodNotSupportedException) {
                responseBody = buildResponse(request, response, ServiceErrorEnum.ISV_METHOD_NOT_ALLOWED.getErrorMeta().getException());
                response.setHeader(X_SERVICE_ERROR_RESPONSE, UriUtils.encode(responseBody, StandardCharsets.UTF_8));
                return;
            }
            // JSR-303 异常处理
            if (exception instanceof BindException) {
                BindingResult bindingResult = ((BindException) exception).getBindingResult();
                ServiceException serviceException = getValidateBizParamException(bindingResult.getFieldError().getDefaultMessage());
                responseBody = buildResponse(request, response, serviceException);
                response.setHeader(X_SERVICE_ERROR_RESPONSE, UriUtils.encode(responseBody, StandardCharsets.UTF_8));
                return;
            }
            if (exception instanceof ConstraintViolationException) {
                String message = null;
                Optional<ConstraintViolation<?>> optional = ((ConstraintViolationException) exception).getConstraintViolations().stream().findFirst();
                if (optional.isPresent()) {
                    ConstraintViolation<?> constraintViolation = optional.get();
                    message = constraintViolation.getMessage();
                }
                ServiceException serviceException = getValidateBizParamException(Objects.requireNonNull(message));
                responseBody = buildResponse(request, response, serviceException);
                response.setHeader(X_SERVICE_ERROR_RESPONSE, UriUtils.encode(responseBody, StandardCharsets.UTF_8));
                return;
            }
            if (exception instanceof MethodArgumentNotValidException) {
                BindingResult bindingResult =  ((MethodArgumentNotValidException)exception).getBindingResult();
                String message = bindingResult.getFieldErrors().stream()
                        .map(FieldError::getDefaultMessage)
                        .collect(Collectors.joining(", "));
                ServiceException serviceException = getValidateBizParamException(Objects.requireNonNull(message));
                responseBody = buildResponse(request, response, serviceException);
                response.setHeader(X_SERVICE_ERROR_RESPONSE, UriUtils.encode(responseBody, StandardCharsets.UTF_8));
                return;
            }
            StringBuilder msg = new StringBuilder();
            msg.append(exception.getMessage());
            StackTraceElement[] stackTrace = exception.getStackTrace();
            // 取5行错误内容
            int lineCount = 5;
            for (int i = 0; i < stackTrace.length && i < lineCount; i++) {
                StackTraceElement stackTraceElement = stackTrace[i];
                msg.append("\n at ").append(stackTraceElement.toString());
            }
            response.setHeader(X_SERVICE_ERROR_MESSAGE, UriUtils.encode(msg.toString(), StandardCharsets.UTF_8));
        }
    }

    /**
     * 处理异常
     *
     * @param request   request
     * @param response  response
     * @param exception 异常信息
     * @return 返回最终结果
     */
    private static String buildResponse(HttpServletRequest request, HttpServletResponse response, Exception exception) {
        ServiceResultBuilder serviceResultBuilder = ServiceConfig.getInstance().getServiceResultBuilder();
        Object result = serviceResultBuilder.buildError(request, response, exception);
        return JSON.toJSONString(result);
    }

    public static ServiceException getValidateBizParamException(String errorMsg) {
        String subCode = ServiceErrorEnum.ISV_PARAM_ERROR.getErrorMeta().getSubCode();
        String[] msgToken = errorMsg.split(EQ);
        String msg = msgToken[0];
        if (msg.startsWith(LEFT_TOKEN) && msg.endsWith(RIGHT_TOKEN)) {
            String module = msg.substring(1, msg.length() - 1);
            Object[] params = buildParams(msgToken);
            String error = ServiceErrorFactory.getErrorMessage(module, ServiceContext.getCurrentContext().getLocale(), params);
            return new ServiceException(subCode, error);
        } else {
            return new ServiceException(subCode, errorMsg);
        }
    }

    private static Object[] buildParams(String[] msgToken) {
        if (msgToken.length == 2) {
            return msgToken[1].split(COMMA);
        } else {
            return EMPTY_OBJ_ARRAY;
        }
    }
}
