package com.icetech.web.exception;

import com.fasterxml.jackson.databind.exc.MismatchedInputException;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.exception.base.BaseException;
import com.icetech.common.utils.HttpTools;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.nio.file.AccessDeniedException;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.text.ParseException;
import java.util.stream.Collectors;

/**
 * 全局的的异常拦截器
 *
 * @author wangzw
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {
    /**
     * 请求内容异常
     */
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public ObjectResponse<Void> httpMediaTypeNotSupportedException(HttpServletRequest request, HttpMediaTypeNotSupportedException e) {
        log.warn("不支持的请求内容|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getContentType());
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_403);
    }

    /**
     * 请求方式异常
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public ObjectResponse<Void> httpRequestMethodNotSupportedException(HttpServletRequest request, HttpRequestMethodNotSupportedException e) {
        log.warn("不支持的HTTP请求方式|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getMethod());
        return ObjectResponse.failed(CodeConstants.ERROR_403, "不支持的HTTP请求方式");
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    public ObjectResponse<String> handleMissingServletRequestParameterException(HttpServletRequest request, MissingServletRequestParameterException e) {
        log.warn("缺少请求参数|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getParameterName());
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400);
    }

    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public ObjectResponse<Void> methodArgumentTypeMismatchException(HttpServletRequest request, MethodArgumentTypeMismatchException e) {
        log.warn("请求参数类型有误|{}|{}|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request),
                e.getParameter().getParameterName(), e.getParameter().getParameterType(), e.getRequiredType());
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400);
    }

    /**
     * 参数非法异常
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ObjectResponse<Void> methodArgumentNotValidException(HttpServletRequest request, MethodArgumentNotValidException e) {
        String message = e.getBindingResult().getAllErrors().stream().map(ObjectError::getDefaultMessage).collect(Collectors.joining(","));
        log.warn("参数校验未通过|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), message);
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), message);
    }
    
    @ExceptionHandler(BindException.class)
    public ObjectResponse<Void> bindException(HttpServletRequest request, BindException e) {
        String message = e.getAllErrors().stream().map(ObjectError::getDefaultMessage).collect(Collectors.joining(","));
        log.warn("参数校验未通过|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), message);
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), message);
    }

    @ExceptionHandler(ConstraintViolationException.class)
    public ObjectResponse<String> constraintViolationException(HttpServletRequest request, ConstraintViolationException e) {
        String message = e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(","));
        log.warn("参数校验未通过|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), message);
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400.getCode(), message);
    }

    @ExceptionHandler(MismatchedInputException.class)
    public ObjectResponse<Void> mismatchedInputException(HttpServletRequest request, MismatchedInputException e) {
        log.warn("请求内容丢失|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getMessage());
        return ObjectResponse.failed(CodeConstants.ERROR_400, "请求内容无效");
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    public ObjectResponse<String> httpMessageNotReadableException(HttpServletRequest request, HttpMessageNotReadableException e) {
        String message = e.getMessage();
        if (StringUtils.isNotBlank(message)) {
            int index = message.indexOf(":");
            if (index > 0) {
                message = message.substring(0, index);
            }
        }
        log.warn("请求内容不全|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), message);
        return ObjectResponse.failed(CodeConstants.ERROR_400, "请求内容不全");
    }

    @ExceptionHandler(ParseException.class)
    public ObjectResponse<String> handleParseException(HttpServletRequest request, ParseException e) {
        log.warn("参数转换失败|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getMessage());
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_400);
    }

    /**
     * 无权限访问
     */
    @ExceptionHandler(AccessDeniedException.class)
    public ObjectResponse<Void> accessDeniedException(HttpServletRequest request, AccessDeniedException e) {
        log.warn("请求无权访问|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getMessage());
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_401);
    }

    /**
     * 业务异常
     */
    @ExceptionHandler(ResponseBodyException.class)
    public ObjectResponse<Void> responseBodyException(HttpServletRequest request, ResponseBodyException e) {
        log.warn("业务处理异常|{}|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getErrCode(), e.getMessage());
        //此处Msg不加 CodeConstants.getName(e.getErrCode())，因为前端需要直接使用msg中的url跳转
        return ObjectResponse.failed(e.getErrCode(), e.getMessage());
    }

	@ExceptionHandler(BaseException.class)
	public ObjectResponse<Object> baseException(BaseException e) {
		log.warn("base异常|{}|{}|{}", e.getMessage(), e.getCode(), e.getArgs());
		return ObjectResponse.failed(e.getCode(), e.getArgs());
	}

    /**
     * SQL异常
     */
    @ExceptionHandler(SQLIntegrityConstraintViolationException.class)
    public ObjectResponse<Void> sqlIntegrityConstraintViolationException(HttpServletRequest request, SQLException e) {
        log.error("SQL执行异常|{}|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e.getErrorCode(), e);
        return ObjectResponse.failed(CodeConstantsEnum.ERROR_405);
    }

    /**
     * 全局异常
     */
    @ExceptionHandler(Exception.class)
    public ObjectResponse<Void> exception(HttpServletRequest request, Exception e) {
        log.error("请求处理异常|{}|{}", request.getRequestURI(), HttpTools.getIpAddr(request), e);
        return ObjectResponse.failed(CodeConstantsEnum.ERROR);
    }
}
