package com.icetech.web.filter;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.alibaba.fastjson.JSON;
import com.icetech.common.utils.HttpTools;
import com.icetech.web.wrapper.HttpServletRequestDecorator;
import com.icetech.web.wrapper.HttpServletResponseDecorator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.util.AntPathMatcher;

/**
 * HTTP日志记录器
 *
 * @since 1.0.5
 */
@Slf4j
public class LoggingHttpFilter implements Filter {
    @Value("${ice.web.filter.logging-http.trunc-length:2048}")
    private int truncLength;
    @Value("#{'${ice.web.filter.logging-http.ignore-path:}'.split(',')}")
    private String[] ignorePaths;
    private static final Pattern PATTERN_FORM_FILE_NAME = Pattern.compile("filename=\"([^\\f\\n\\r\\t\\v\"]+)\"");
    private static final byte[] SUFFIX_TRUNC = " ......".getBytes(StandardCharsets.UTF_8);
    private static final AntPathMatcher PATH_MATCHER = new AntPathMatcher();

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        if (request instanceof HttpServletRequest) {
            HttpServletRequest httpServletRequest = (HttpServletRequest) request;
            if (ignorePaths != null && ignorePaths.length > 0) {
                for (int i = 0; i < ignorePaths.length; i++) {
                    String ignorePath = ignorePaths[i];
                    if (PATH_MATCHER.match(ignorePath, httpServletRequest.getRequestURI())) {
                        chain.doFilter(request, response);
                        return;
                    }
                }
            }
        }

        long begin = System.currentTimeMillis();
        String requestURI = "";
        String requestAddr = "";
        try {
            HttpServletRequestDecorator requestWrapper = null;
            if (request instanceof HttpServletRequestDecorator) {
                requestWrapper = (HttpServletRequestDecorator) request;
            } else if (request instanceof HttpServletRequest) {
                requestWrapper = new HttpServletRequestDecorator((HttpServletRequest) request);
                request = requestWrapper;
            }
            if (requestWrapper != null) {
                requestURI = requestWrapper.getRequestURI();
                requestAddr = HttpTools.getIpAddr((HttpServletRequest) request);
                byte[] requestBytes = truncBytes(requestWrapper.getCacheBody());
                String requestBody = new String(requestBytes, StringUtils.isBlank(requestWrapper.getCharacterEncoding()) ? StandardCharsets.UTF_8 : Charset.forName(requestWrapper.getCharacterEncoding()));
                String content = requestURI + "|" + requestAddr + "|" + requestWrapper.getMethod() + "|" + requestWrapper.getDispatcherType() + "|" + requestWrapper.getContentType() + "|"
//                        + buildQueryLog(requestWrapper) + "|"
                        + buildParameterLog(requestWrapper) + "|"
                        + buildPartLog(requestWrapper);
                log.info(content + "|" + requestBody);
            }
            if (response instanceof HttpServletResponse && !(response instanceof HttpServletResponseDecorator)) {
                response = new HttpServletResponseDecorator((HttpServletResponse) response);
            }

            chain.doFilter(request, response);
        } catch (Exception e) {
            log.error("日志记录器出错|{}|{}|{}ms", requestURI, System.currentTimeMillis() - begin, e);
        } finally {
            if (response instanceof HttpServletResponseDecorator) {
                HttpServletResponseDecorator responseWrapper = (HttpServletResponseDecorator) response;
                String disposition = responseWrapper.getHeader(HttpHeaders.CONTENT_DISPOSITION);
                String responseBody = null;
                if (StringUtils.startsWith(disposition, "attachment;filename=")) {
                    disposition = disposition.replaceFirst("attachment;filename=", "");
                    responseBody = "FILE:" + disposition;
                } else {
                    byte[] responseBytes = truncBytes(responseWrapper.getContentAsByteArray());
//                    responseWrapper.copyBodyToResponse();
//                String responseBody = new String(responseBytes, StringUtils.isBlank(response.getCharacterEncoding()) ? StandardCharsets.UTF_8 : Charset.forName(response.getCharacterEncoding()));
                    responseBody = new String(responseBytes, StandardCharsets.UTF_8);
                }
                log.info("{}|{}|{}|{}ms", requestURI, responseWrapper.getContentType(), responseBody, System.currentTimeMillis() - begin);
            }
        }
    }

    public void init(FilterConfig filterConfig) throws ServletException {
        log.info("LoggingFilter 初始化");
        log.info("设定日志字符串最大输出长度为: {}", truncLength);
        log.info("设定忽略日志记录路径: {}", Arrays.toString(ignorePaths));
    }

    public void destroy() {
        log.info("LoggingFilter 销毁");
    }

    private String buildQueryLog(HttpServletRequest request) {
        if (request.getParameterMap() != null && !request.getParameterMap().isEmpty()) {
            return "[query:" + request.getQueryString() + "]";
        }
        return "";
    }

    private String buildParameterLog(HttpServletRequest request) {
        if (request.getParameterMap() != null && !request.getParameterMap().isEmpty()) {
            return "[parameter:" + JSON.toJSONString(request.getParameterMap()) + "]";
        }
        return "";
    }

    private String buildPartLog(HttpServletRequest request) throws IOException, ServletException {
        String contentType = request.getContentType();
        if (StringUtils.containsAny(contentType, "multipart/form-data", "multipart/mixed")
                && request.getParts() != null
                && !request.getParts().isEmpty()) {
            List<String> parts = request.getParts().stream().map(part -> {
                String disposition = part.getHeader(HttpHeaders.CONTENT_DISPOSITION);
                Matcher matcher = PATTERN_FORM_FILE_NAME.matcher(disposition);
                if (matcher.find()) {
                    String fileName = matcher.group(1);
                    return part.getName() + "=" + fileName + ":" + part.getSize();
                }
                return null;
            }).filter(Objects::nonNull).collect(Collectors.toList());
            return "[part:" + JSON.toJSONString(parts) + "]";
        }
        return "";
    }

    private byte[] truncBytes(byte[] source) {
        int maxLength = truncLength + SUFFIX_TRUNC.length;
        if (source.length > maxLength) {
            byte[] truncated = new byte[maxLength];
            System.arraycopy(source, 0, truncated, 0, truncLength);
            System.arraycopy(SUFFIX_TRUNC, 0, truncated, truncLength, SUFFIX_TRUNC.length);
            return truncated;
        }

        return source;
    }
}
