package com.icetech.redis.config;

import com.icetech.redis.handle.IceProxyMsgHandle;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceClientConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettucePoolingClientConfiguration;

/**
 * Description SocketAloneRedisInject
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2022/2/21 2:36 下午
 */
@Configuration
public class SocketAloneRedisInject implements EnvironmentAware {

    /**
     * 配置信息的前缀 
     */
    private static final String ALONE_PREFIX = "ice-socket.alone-redis";

    @Autowired(required = false)
    private IceProxyMsgHandle iceProxyMsgHandle;
    
    @Override
    public void setEnvironment(Environment environment) {

        // 如果配置文件不包含相关配置，则不进行任何操作 
        if(environment.getProperty(ALONE_PREFIX + ".host") == null) {
            return;
        }
        // ------------------- 开始注入 
        // 获取cfg对象 
        RedisProperties cfg = Binder.get(environment).bind(ALONE_PREFIX, RedisProperties.class).get();

        // 1. Redis配置 
        RedisStandaloneConfiguration redisConfig = new RedisStandaloneConfiguration();
        redisConfig.setHostName(cfg.getHost());
        redisConfig.setPort(cfg.getPort());
        redisConfig.setDatabase(cfg.getDatabase());
        redisConfig.setPassword(RedisPassword.of(cfg.getPassword()));
        // 2. 连接池配置 
        GenericObjectPoolConfig poolConfig = new GenericObjectPoolConfig();
        // pool配置 
        RedisProperties.Lettuce lettuce = cfg.getLettuce();
        if(lettuce.getPool() != null) {
            RedisProperties.Pool pool = cfg.getLettuce().getPool();
            // 连接池最大连接数
            poolConfig.setMaxTotal(pool.getMaxActive());
            // 连接池中的最大空闲连接 
            poolConfig.setMaxIdle(pool.getMaxIdle());
            // 连接池中的最小空闲连接
            poolConfig.setMinIdle(pool.getMinIdle());
            // 连接池最大阻塞等待时间（使用负值表示没有限制）
            poolConfig.setMaxWaitMillis(pool.getMaxWait().toMillis());
        }
        LettucePoolingClientConfiguration.LettucePoolingClientConfigurationBuilder builder = LettucePoolingClientConfiguration.builder();
        // timeout 
        if(cfg.getTimeout() != null) {
            builder.commandTimeout(cfg.getTimeout());
        }
        // shutdownTimeout 
        if(lettuce.getShutdownTimeout() != null) {
            builder.shutdownTimeout(lettuce.getShutdownTimeout());
        }
        // 创建Factory对象 
        LettuceClientConfiguration clientConfig = builder.poolConfig(poolConfig).build();
        LettuceConnectionFactory factory = new LettuceConnectionFactory(redisConfig, clientConfig);
        factory.afterPropertiesSet();
        iceProxyMsgHandle.init(factory);
    }
}
