package com.icetech.redis.config;

import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.DeserializationProblemHandler;
import com.fasterxml.jackson.databind.jsontype.TypeIdResolver;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.SerializationException;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.regex.Pattern;

public class CustomStringRedisSerializer implements RedisSerializer<String> {
    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private static final Pattern PATTERN = Pattern.compile("^\\[\"(\\w+\\.)+\\w+\",");
    private Jackson2JsonRedisSerializer<Object> jackson2JsonRedisSerializer;

    @Override
    public byte[] serialize(String value) throws SerializationException {
        if (value == null) {
            return new byte[0];
        }

        return value.getBytes(DEFAULT_CHARSET);
    }

    @Override
    public String deserialize(byte[] bytes) throws SerializationException {
        if (ArrayUtils.isEmpty(bytes)) {
            return null;
        }

        String cache = new String(bytes, DEFAULT_CHARSET);
        if (PATTERN.matcher(cache).find()) { // 兼容代码
            Object object = jackson2JsonRedisSerializer.deserialize(bytes);
            if (object == null) {
                return null;
            }
            return JSON.toJSONString(object);
        } else {
            return cache;
        }
    }

    public CustomStringRedisSerializer() {
        jackson2JsonRedisSerializer = new Jackson2JsonRedisSerializer<>(Object.class);
        // ObjectMapper 将Json反序列化成Java对象,当java客户端调用当时候，会在直接转化成对象当java对象，
        // 可以去掉
        ObjectMapper om = new ObjectMapper();
        om.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        om.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL);
        om.addHandler(new DeserializationProblemHandler() {
            @Override
            public JavaType handleUnknownTypeId(DeserializationContext ctxt, JavaType baseType, String subTypeId, TypeIdResolver idResolver, String failureMsg) throws IOException {
                return om.getTypeFactory().constructRawMapType(LinkedHashMap.class);
            }

            @Override
            public JavaType handleMissingTypeId(DeserializationContext ctxt, JavaType baseType, TypeIdResolver idResolver, String failureMsg) throws IOException {
                return om.getTypeFactory().constructRawMapType(LinkedHashMap.class);
            }
        });
        jackson2JsonRedisSerializer.setObjectMapper(om);
    }
}