package com.icetech.redis.handle;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.icetech.common.constants.RedisKeyConstants;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.collections4.map.SingletonMap;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;

@Slf4j
@AllArgsConstructor
public class RedisHandle {
    private RedisTemplate<Object, Object> redisTemplate;
    private StringRedisTemplate stringRedisTemplate;

    /**
     * <p>从Redis获取缓存对象</p>
     * <p>缓存获取为空则执行supplier获取对象</p>
     * <p>缓存获取为{@link RedisKeyConstants#NONE_VALUE}则直接返回为空</p>
     * <p>supplier返回为空则将{@link RedisKeyConstants#NONE_VALUE}放入至缓存</p>
     * <p>supplier返回不为空则将返回对象转换为JSON放入缓存</p>
     *
     * @param cacheKey     缓存KEY
     * @param clazz        缓存对象Class
     * @param supplier     获取缓存失败后的操作
     * @param expireMillis 放入缓存后的超时毫秒数
     * @param <T>          对象类型
     * @return 对象信息
     */
    public <T> T cacheObject(String cacheKey, Class<T> clazz, Supplier<T> supplier, long expireMillis) {
        return cacheValue(cacheKey, value -> JSONObject.parseObject(value, clazz), supplier, JSONObject::toJSONString, expireMillis);
    }

    public <T> T cacheObject(String cacheKey, TypeReference<T> reference, Supplier<T> supplier, long expireMillis) {
        return cacheValue(cacheKey, value -> JSONObject.parseObject(value, reference), supplier, JSONObject::toJSONString, expireMillis);
    }

    @Nonnull
    public <T> List<T> cacheList(String cacheKey, Class<T> clazz, Supplier<List<T>> supplier, long expireMillis) {
        List<T> list = cacheValue(cacheKey, value -> JSONArray.parseArray(value, clazz), supplier, JSONObject::toJSONString, expireMillis);
        if (list == null) {
            return Collections.emptyList();
        }
        return list;
    }

    @Nonnull
    public <T> List<T> cacheList(String cacheKey, TypeReference<List<T>> reference, Supplier<List<T>> supplier, long expireMillis) {
        List<T> list = cacheValue(cacheKey, value -> JSONArray.parseObject(value, reference), supplier, JSONObject::toJSONString, expireMillis);
        if (list == null) {
            return Collections.emptyList();
        }
        return list;
    }

    public <T> T cacheValue(String cacheKey, Function<String, T> stringToValue, Supplier<T> supplier, Function<T, String> valueToString, long expireMillis) {
        String cacheValue = stringRedisTemplate.opsForValue().get(cacheKey);
        if (StringUtils.isNotBlank(cacheValue)) {
            if (RedisKeyConstants.NONE_VALUE.equals(cacheValue)) {
                return null;
            }
            //所有车场权限时，转换成统一Long类型
            String all = "ALL";
            if (cacheKey.startsWith(RedisKeyConstants.KEY_PREFIX_USER_PARKS) && cacheValue.equals(all)) {
                cacheValue = "[-1]";
            }
            return stringToValue.apply(cacheValue);
        }
        if (supplier == null) {
            return null;
        }

        T data = supplier.get();
        if (data == null) {
            stringRedisTemplate.opsForValue().set(cacheKey, RedisKeyConstants.NONE_VALUE, expireMillis, TimeUnit.MILLISECONDS);
            return null;
        }

        cacheValue = valueToString.apply(data);
        stringRedisTemplate.opsForValue().set(cacheKey, cacheValue, expireMillis, TimeUnit.MILLISECONDS);
        return data;
    }

    public <T> T getObjectFromMap(String cacheKey, String mapKey, Class<T> clazz, Supplier<Map<String, T>> supplier, long expireMillis) {
        return getValueFromMap(cacheKey, mapKey, value -> JSONObject.parseObject(value, clazz), JSONObject::toJSONString, supplier, expireMillis);
    }

    @Nonnull
    public <T> List<T> getListFromMap(String cacheKey, String mapKey, Class<T> clazz, Supplier<Map<String, List<T>>> supplier, long expireMillis) {
        List<T> list = getValueFromMap(cacheKey, mapKey, value -> JSONArray.parseArray(value, clazz), JSONArray::toJSONString, supplier, expireMillis);
        if (list == null) {
            return Collections.emptyList();
        }

        return list;
    }

    public <T> T getValueFromMap(String cacheKey, String mapKey, Function<String, T> stringToValue, Function<T, String> valueToString, Supplier<Map<String, T>> supplier, long expireMillis) {
        HashOperations<String, String, String> operations = stringRedisTemplate.opsForHash();
        String cacheValue = operations.get(cacheKey, mapKey);
        if (StringUtils.isNotBlank(cacheValue)) {
            if (RedisKeyConstants.NONE_VALUE.equals(cacheValue)) {
                return null;
            }

            return stringToValue.apply(cacheValue);
        }

        Map<String, T> map = cacheMap(cacheKey, Function.identity(), stringToValue, supplier, valueToString, expireMillis);
        return map.get(mapKey);
    }

    @Nonnull
    public <T> Map<String, T> cacheObjectMap(String cacheKey, Class<T> valueClass, Supplier<Map<String, T>> supplier, long expireMillis) {
        return cacheMap(cacheKey, Function.identity(), value -> JSONObject.parseObject(value, valueClass), supplier, JSONObject::toJSONString, expireMillis);
    }

    @Nonnull
    public <T> Map<String, T> cacheObjectMap(String cacheKey, TypeReference<T> reference, Supplier<Map<String, T>> supplier, long expireMillis) {
        return cacheMap(cacheKey, Function.identity(), value -> JSONObject.parseObject(value, reference), supplier, JSONObject::toJSONString, expireMillis);
    }

    @Nonnull
    public <T> Map<String, List<T>> cacheListMap(String cacheKey, Class<T> valueClass, Supplier<Map<String, List<T>>> supplier, long expireMillis) {
        return cacheMap(cacheKey, Function.identity(), value -> JSONArray.parseArray(value, valueClass), supplier, JSONArray::toJSONString, expireMillis);
    }

    @Nonnull
    public <T> Map<String, List<T>> cacheListMap(String cacheKey, TypeReference<List<T>> reference, Supplier<Map<String, List<T>>> supplier, long expireMillis) {
        return cacheMap(cacheKey, Function.identity(), value -> JSONArray.parseObject(value, reference), supplier, JSONArray::toJSONString, expireMillis);
    }

    public <K, V> Map<K, V> cacheMap(String cacheKey, Function<String, K> keyFunction, Function<String, V> stringToValue, Supplier<Map<K, V>> supplier, Function<V, String> valueToString, long expireMillis) {
        HashOperations<String, String, String> operations = stringRedisTemplate.opsForHash();
        Map<String, String> cacheMap = operations.entries(cacheKey);
        if (MapUtils.isNotEmpty(cacheMap)) {
            if (cacheMap.containsKey(RedisKeyConstants.NONE_VALUE)) {
                return Collections.emptyMap();
            }

            Map<K, V> map = new HashMap<>(cacheMap.size());
            for (Map.Entry<String, String> cacheEntry : cacheMap.entrySet()) {
                K key = keyFunction.apply(cacheEntry.getKey());
                if (StringUtils.isBlank(cacheEntry.getValue()) || RedisKeyConstants.NONE_VALUE.equals(cacheEntry.getValue())) {
                    map.put(key, null);
                } else {
                    map.put(key, stringToValue.apply(cacheEntry.getValue()));
                }
            }
            return map;
        }
        if (supplier == null) {
            return Collections.emptyMap();
        }

        Map<K, V> map = supplier.get();
        if (MapUtils.isEmpty(map)) {
            cacheMap = new SingletonMap<>(RedisKeyConstants.NONE_VALUE, RedisKeyConstants.NONE_VALUE);
            map = Collections.emptyMap();
        } else {
            cacheMap = new HashMap<>(map.size());
            for (Map.Entry<K, V> entry : map.entrySet()) {
                cacheMap.put(entry.getKey().toString(), entry.getValue() == null ? RedisKeyConstants.NONE_VALUE : valueToString.apply(entry.getValue()));
            }
        }
        stringRedisTemplate.opsForHash().putAll(cacheKey, cacheMap);
        stringRedisTemplate.expire(cacheKey, expireMillis, TimeUnit.MILLISECONDS);
        return map;
    }
    
    /**
     * Description: 删除key
     * Version1.0 2022-10-21 by wgy 创建
     * 
     * @param key   key
     * @return java.lang.Boolean       
     */
    public Boolean delete(String key) {
        return redisTemplate.delete(key);
    }
    
}
