package com.icetech.redis.handle;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.P2cBaseMsg;
import com.icetech.common.domain.WebSocketMessage;
import com.icetech.common.domain.response.ObjectResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.StringRedisTemplate;

/**
 * Description socket 消息发送助手
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2022/2/21 2:27 下午
 */
@Slf4j
public class IceProxyMsgHandle {

    /**
     * String专用
     */
    public StringRedisTemplate stringRedisTemplate;

    @Autowired
    public void init(RedisConnectionFactory connectionFactory) {
        // 构建StringRedisTemplate
        StringRedisTemplate stringTemplate = new StringRedisTemplate();
        stringTemplate.setConnectionFactory(connectionFactory);
        stringTemplate.afterPropertiesSet();
        // 开始初始化相关组件 
        this.stringRedisTemplate = stringTemplate;
    }

    /**
     * Description: 发送socket 消息
     * Version1.0 2022-02-21 by wgy 创建
     * 
     * @param p2cBaseMsg    参数
     * @return com.icetech.common.domain.response.ObjectResponse<java.lang.Boolean>       
     */
    public ObjectResponse<Boolean> sendSocketMsg(P2cBaseMsg p2cBaseMsg) {
        try {
            log.info("消息发放内容,{}", JSONUtil.toJsonStr(p2cBaseMsg));
            stringRedisTemplate.convertAndSend(RedisKeyConstants.SOCKET_MSG_TOPIC, JSON.toJSONString(p2cBaseMsg));
        } catch (Exception e) {
            log.error("[集群下广播失败]|设备[{}]|消息[{}]", p2cBaseMsg.getSn(), JSON.toJSONString(p2cBaseMsg), e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }
        return ObjectResponse.success(true);
    }

    public ObjectResponse<Boolean> sendSocketMsg(WebSocketMessage message) {
        try {
            String content = JSON.toJSONString(message);
            log.info("消息发放内容,{}", content);
            stringRedisTemplate.convertAndSend(RedisKeyConstants.SOCKET_MSG_TOPIC, content);
        } catch (Exception e) {
            log.error("集群下广播失败|{}|{}", message.getModule(), message.getChannel(), e);
            return ObjectResponse.failed(CodeConstantsEnum.ERROR);
        }
        return ObjectResponse.success(true);
    }
}
