package com.icetech.mq.config;

import java.util.LinkedHashMap;
import java.util.Map;

import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.Accessors;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.core.AbstractExchange;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.Exchange;
import org.springframework.amqp.core.ExchangeBuilder;
import org.springframework.amqp.core.ExchangeTypes;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.core.QueueBuilder;

/*
#ice.rabbitmq.declare.exchange.e_name.declare = false
#ice.rabbitmq.declare.exchange.e_name.admin =
#ice.rabbitmq.declare.exchange.e_name.name = e_name
#ice.rabbitmq.declare.exchange.e_name.type = direct
#ice.rabbitmq.declare.exchange.e_name.durable = true
#ice.rabbitmq.declare.exchange.e_name.auto-delete = false
#ice.rabbitmq.declare.exchange.e_name.internal = false
#ice.rabbitmq.declare.exchange.e_name.delayed = false
#ice.rabbitmq.declare.exchange.e_name.alternate =
#ice.rabbitmq.declare.exchange.e_name.arguments.xxx = xxx

#ice.rabbitmq.declare.queue.q_name.declare = true
#ice.rabbitmq.declare.queue.q_name.admin =
#ice.rabbitmq.declare.queue.q_name.name = q_name
#ice.rabbitmq.declare.queue.q_name.durable = true
#ice.rabbitmq.declare.queue.q_name.exclusive = false
#ice.rabbitmq.declare.queue.q_name.auto-delete = false
#ice.rabbitmq.declare.queue.q_name.ttl = 0
#ice.rabbitmq.declare.queue.q_name.expires = 0
#ice.rabbitmq.declare.queue.q_name.max-length = 0
#ice.rabbitmq.declare.queue.q_name.max-length-bytes = 0
#ice.rabbitmq.declare.queue.q_name.overflow =
#ice.rabbitmq.declare.queue.q_name.dead-letterexchange =
#ice.rabbitmq.declare.queue.q_name.dead-letterRoutingKey =
#ice.rabbitmq.declare.queue.q_name.max-priority = 0
#ice.rabbitmq.declare.queue.q_name.lazy = false
#ice.rabbitmq.declare.queue.q_name.master-locator =
#ice.rabbitmq.declare.queue.q_name.single-active-consumer = false
#ice.rabbitmq.declare.queue.q_name.quorum = false
#ice.rabbitmq.declare.queue.q_name.delivery-limit = 0
#ice.rabbitmq.declare.queue.q_name.arguments.xxx = xxx

#ice.rabbitmq.declare.binding.e_q.declare = true
#ice.rabbitmq.declare.binding.e_q.admin =
#ice.rabbitmq.declare.binding.e_q.destination = q_name
#ice.rabbitmq.declare.binding.e_q.exchange = e_name
#ice.rabbitmq.declare.binding.e_q.routing-key = r_key
#ice.rabbitmq.declare.binding.e_q.destination-type = QUEUE
#ice.rabbitmq.declare.binding.e_q.arguments.xxx = xxx
*/

@Getter
@Setter
@ToString
@Accessors(chain = true)
public class RabbitDeclareProperties {
    protected final Map<String, ExchangeInfo> exchange;
    protected final Map<String, QueueInfo> queue;
    protected final Map<String, BindingInfo> binding;

    public RabbitDeclareProperties() {
        exchange = new LinkedHashMap<>();
        queue = new LinkedHashMap<>();
        binding = new LinkedHashMap<>();
    }

    public void setExchange(Map<String, ExchangeInfo> exchange) {
        if (MapUtils.isNotEmpty(exchange))
            this.exchange.putAll(exchange);
    }

    public void setQueue(Map<String, QueueInfo> queue) {
        if (MapUtils.isNotEmpty(queue))
            this.queue.putAll(queue);
    }

    public void setBinding(Map<String, BindingInfo> binding) {
        if (MapUtils.isNotEmpty(binding))
            this.binding.putAll(binding);
    }

    @Getter
    @Setter
    @ToString
    @Accessors(chain = true)
    public static class ExchangeInfo {
        private boolean declare = true;
        private String admin;
        private String name;
        private String type = ExchangeTypes.DIRECT;
        private boolean durable = true;
        private boolean autoDelete = false;
        private boolean internal = false;
        private boolean delayed = false;
        private String alternate;
        private Map<String, Object> arguments;

        public Exchange buildExchange() {
            ExchangeBuilder builder = new ExchangeBuilder(name, type).durable(durable);
            if (autoDelete) builder.autoDelete();
            if (StringUtils.isNotBlank(alternate)) builder.alternate(alternate);
            if (MapUtils.isNotEmpty(arguments)) builder.withArguments(arguments);
            AbstractExchange exchange = builder.build();
            exchange.setInternal(internal);
            exchange.setDelayed(delayed);
            return exchange;
        }
    }


    @Getter
    @Setter
    @ToString
    @Accessors(chain = true)
    public static class QueueInfo {
        private boolean declare = true;
        private String admin;
        private String name;
        private boolean durable = true;
        private boolean exclusive = false;
        private boolean autoDelete = false;
        private int ttl = -1;
        private int expires = -1;
        private int maxLength = -1;
        private int maxLengthBytes = -1;
        private QueueBuilder.Overflow overflow;
        private String deadLetterExchange;
        private String deadLetterRoutingKey;
        private int maxPriority = -1;
        private boolean lazy = false;
        private QueueBuilder.MasterLocator masterLocator;
        private boolean singleActiveConsumer = false;
        private boolean quorum = false;
        private int deliveryLimit = -1;
        private Map<String, Object> arguments;

        public Queue buildQueue() {
            QueueBuilder builder = durable ? QueueBuilder.durable(name) : QueueBuilder.nonDurable(name);
            if (exclusive) builder.exclusive();
            if (autoDelete) builder.autoDelete();
            if (ttl > 0) builder.ttl(ttl);
            if (expires > 0) builder.expires(expires);
            if (maxLength > 0) builder.maxLength(maxLength);
            if (maxLengthBytes > 0) builder.maxLengthBytes(maxLengthBytes);
            if (overflow != null) builder.overflow(overflow);
            if (!StringUtils.isAllBlank(deadLetterExchange, deadLetterRoutingKey))
                builder.deadLetterExchange(deadLetterExchange).deadLetterRoutingKey(deadLetterRoutingKey);
            if (maxPriority > 0) builder.maxPriority(maxPriority);
            if (lazy) builder.lazy();
            if (masterLocator != null) builder.masterLocator(masterLocator);
            if (singleActiveConsumer) builder.singleActiveConsumer();
            if (quorum) builder.quorum();
            if (deliveryLimit > 0) builder.deliveryLimit(deliveryLimit);
            if (MapUtils.isNotEmpty(arguments)) builder.withArguments(arguments);
            return builder.build();
        }
    }


    @Getter
    @Setter
    @ToString
    @Accessors(chain = true)
    public static class BindingInfo {
        private boolean declare = true;
        private String admin;
        private String destination;
        private String exchange;
        private String routingKey;
        private Binding.DestinationType destinationType = Binding.DestinationType.QUEUE;
        private Map<String, Object> arguments;

        public Binding buildBinding() {
            return new Binding(destination, destinationType, exchange, routingKey, arguments);
        }
    }
}