package com.icetech.mq.config;

import com.icetech.mq.sender.RabbitSender;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.rabbit.listener.RabbitListenerContainerFactory;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.ImportAutoConfiguration;
import org.springframework.boot.autoconfigure.amqp.SimpleRabbitListenerContainerFactoryConfigurer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * RabbitMQ 配置类
 *
 * @author wgy
 */
@Slf4j
@Configuration
@ImportAutoConfiguration({IceRabbitDeclareBeanPostProcessor.class})
public class RabbitConfig {
    @Bean
    public RabbitListenerContainerFactory<?> rabbitListenerContainerFactory(SimpleRabbitListenerContainerFactoryConfigurer configurer, ConnectionFactory connectionFactory
            , Jackson2JsonMessageConverter jsonMessageConverter){
        SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
        factory.setConnectionFactory(connectionFactory);
        configurer.configure(factory, connectionFactory);
        factory.setMessageConverter(jsonMessageConverter);
        return factory;
    }

    /**
     * 方法rabbitAdmin的功能描述:动态声明queue、exchange、routing
     *
     * @param connectionFactory
     * @return
     */
    @Bean
    public RabbitAdmin rabbitAdmin(ConnectionFactory connectionFactory) {
        return new RabbitAdmin(connectionFactory);
    }

    @Bean
    @ConfigurationProperties(prefix = "ice.rabbitmq.declare")
    public RabbitDeclareProperties rabbitDeclareProperties() {
        return new RabbitDeclareProperties();
    }

    @Bean
    @ConditionalOnMissingBean(RabbitSender.class)
    public RabbitSender rabbitSender(RabbitTemplate rabbitTemplate,
                                     ObjectProvider<RabbitMultipleAutoConfiguration> rabbitMultipleAutoConfiguration,
                                     @Value("${spring.rabbitmq.template.use-publisher-connection:true}") boolean usePublisherConnection) {
        rabbitTemplate.setUsePublisherConnection(usePublisherConnection);
        return new RabbitSender(rabbitTemplate, rabbitMultipleAutoConfiguration);
    }

    /**
     * 定义消息转换实例
     * @return
     */
    @Bean(name = "messageConverter")
    @ConditionalOnMissingBean(MessageConverter.class)
    MessageConverter jackson2JsonMessageConverter() {
        return new Jackson2JsonMessageConverter();
    }
}

