package com.icetech.mq.config;

import com.alibaba.fastjson.JSON;
import com.icetech.common.utils.FileUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static com.icetech.mq.config.RabbitDeclareProperties.*;

@Slf4j
public class IceRabbitDeclareBeanPostProcessor implements BeanPostProcessor, ApplicationContextAware {
    private ApplicationContext applicationContext;

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);
        if (!Object.class.equals(targetClass) && targetClass.isAssignableFrom(RabbitDeclareProperties.class)) {
            declare((RabbitDeclareProperties) bean);
        }
        return bean;
    }

    protected void declare(RabbitDeclareProperties properties) {
        readJson(properties);
        RabbitAdmin rabbitAdmin = applicationContext.getBean(RabbitAdmin.class);
        properties.getExchange().entrySet().stream().map(entry -> {
                    ExchangeInfo info = entry.getValue();
                    if (StringUtils.isBlank(info.getName()))
                        info.setName(entry.getKey());
                    return info;
                }).filter(ExchangeInfo::isDeclare)
                .map(ExchangeInfo::buildExchange)
                .forEach(exchange -> {
                    rabbitAdmin.declareExchange(exchange);
                    log.info("RabbitMQ 交换机创建成功|{}", exchange.getName());
                });
        properties.getQueue().entrySet().stream().map(entry -> {
                    QueueInfo info = entry.getValue();
                    if (StringUtils.isBlank(info.getName()))
                        info.setName(entry.getKey());
                    return info;
                }).filter(QueueInfo::isDeclare)
                .map(QueueInfo::buildQueue)
                .forEach(queue -> {
                    rabbitAdmin.declareQueue(queue);
                    log.info("RabbitMQ 队列创建成功|{}", queue.getName());
                });
        properties.getBinding().values().stream().filter(BindingInfo::isDeclare)
                .map(BindingInfo::buildBinding)
                .forEach(binding -> {
                    rabbitAdmin.declareBinding(binding);
                    log.info("RabbitMQ 队列绑定交换机成功|{}|{}|{}", binding.getDestination(), binding.getExchange(), binding.getRoutingKey());
                });
    }

    private void readJson(RabbitDeclareProperties properties) {
        String configPath = applicationContext.getEnvironment().getProperty("spring.rabbitmq.configPath", "config/mqcs.json");
        List<RabbitBean> rabbitBeans = Optional.of(configPath).filter(StringUtils::isNotBlank).map(path -> {
            try {
                return FileUtil.getReadFile(path);
            } catch (IOException e) {
                throw new RuntimeException("mq 配置文件异常，请检查配置的json 数据", e);
            }
        }).map(json -> JSON.parseArray(json, RabbitBean.class)).orElse(Collections.emptyList());
        for (RabbitBean rabbitBean : rabbitBeans) {
            properties.getQueue().computeIfAbsent(rabbitBean.getQueue(), key -> {
                QueueInfo info = new QueueInfo()
                        .setName(rabbitBean.getQueue())
                        .setDurable(rabbitBean.isDurable())
                        .setExclusive(rabbitBean.isExclusive())
                        .setAutoDelete(rabbitBean.isAutoDelete())
                        .setTtl(rabbitBean.getTtl());
                if (rabbitBean.isDeadQueue())
                    info.setDeadLetterExchange("dead-" + rabbitBean.getExchange())
                            .setDeadLetterRoutingKey(rabbitBean.getRouting());
                return info;
            });
            if (StringUtils.isAnyBlank(rabbitBean.getExchange(), rabbitBean.getRouting())) continue;
            properties.getExchange().computeIfAbsent(rabbitBean.getExchange(), key -> new ExchangeInfo()
                    .setName(rabbitBean.getExchange()).setDelayed(rabbitBean.isDelayed()));
            String bindingKey = rabbitBean.getExchange() + "&" + rabbitBean.getQueue() + "&" + rabbitBean.getRouting();
            properties.getBinding().computeIfAbsent(bindingKey, key -> new BindingInfo()
                    .setExchange(rabbitBean.getExchange()).setDestination(rabbitBean.getQueue()).setRoutingKey(rabbitBean.getRouting()));
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
