package com.icetech.mq.sender;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.icetech.common.utils.Slf4jUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.common.utils.UUIDTools;
import com.icetech.mq.CorrelationData;
import com.icetech.mq.IMQExceptionData;
import com.icetech.mq.config.DelayBean;
import com.icetech.mq.config.RabbitMultipleAutoConfiguration;
import com.icetech.mq.constants.Constant;
import com.icetech.mq.listener.ErrorMsgBean;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.amqp.AmqpException;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.PropertyMapper;

import java.util.Collection;
import java.util.Map;

/**
 * @Description: mq发送工具类
 * @Version1.0 wgy 创建
 */
@Slf4j
public class RabbitSender {

    /**
     * Rabbit MQ 客户端
     */
    private RabbitTemplate rabbitTemplate;

    /**
     * 异常处理
     */
    @Autowired(required = false)
    private IMQExceptionData iMQExceptionData;

    /**
     * 多mq 配置
     */
    private RabbitMultipleAutoConfiguration rabbitMultipleAutoConfiguration;


    /**
     * 默认mq 名字
     */
    @Value("${spring.rabbitmq.mqName:}")
    private String mqName;


    public RabbitSender() {
    }

    public RabbitSender(RabbitTemplate rabbitTemplate, ObjectProvider<RabbitMultipleAutoConfiguration> rabbitMultipleAutoConfiguration) {
        this.rabbitTemplate = rabbitTemplate;
        PropertyMapper map = PropertyMapper.get();
        map.from(rabbitMultipleAutoConfiguration::getIfUnique).whenNonNull().to(this::setRabbitMultipleAutoConfiguration);
        // 设置回调
        setCallback(mqName, rabbitTemplate);
        if (this.rabbitMultipleAutoConfiguration == null || MapUtils.isEmpty(this.rabbitMultipleAutoConfiguration.getRabbitTemplateMap())) {
            return;
        }
        Map<String, RabbitTemplate> rabbitTemplateMap = this.rabbitMultipleAutoConfiguration.getRabbitTemplateMap();
        if (MapUtils.isNotEmpty(rabbitTemplateMap)) {
            rabbitTemplateMap.forEach(this::setCallback);
        }
    }

    private void setRabbitMultipleAutoConfiguration(RabbitMultipleAutoConfiguration rabbitMultipleAutoConfiguration) {
        this.rabbitMultipleAutoConfiguration = rabbitMultipleAutoConfiguration;
    }


    /**
     * 设置 callback
     *
     * @param mqName         mq 名字
     * @param rabbitTemplate 模板
     */
    private void setCallback(String mqName, RabbitTemplate rabbitTemplate) {
        ConnectionFactory connectionFactory = getPublisherConnectionFactory(rabbitTemplate);
        if (connectionFactory.isPublisherConfirms()) {
            rabbitTemplate.setConfirmCallback((correlationData, ack, cause) -> {
                if (!ack) {
                    this.confirm(mqName, rabbitTemplate, correlationData, false, cause);
                    log.info("sender not send message to the right exchange correlationData={} ack={} cause{}", correlationData, false, cause);
                } else {
                    log.debug("sender send message to the right exchange correlationData={} ack={} cause{}", correlationData, true, cause);
                }
            });
        }
        if (connectionFactory.isPublisherReturns()) {
            //消息是否到达正确的消息队列，如果没有会把消息返回
            rabbitTemplate.setReturnCallback((message, replyCode, replyText, tmpExchange, tmpRoutingKey) -> {
                if (Constant.COMMON_MSG_ERROR_QUEUE.equals(tmpRoutingKey)) {
                    return;
                }
                this.returnedMessage(message, replyCode, replyText, tmpExchange, tmpRoutingKey);
            });
        }
    }

    /**
     * 发送MQ消息
     *
     * @param exchangeName 交换机名称
     * @param routingKey   路由名称
     * @param msg          发送消息体
     */
    public void sendMessage(String exchangeName, String routingKey, Object msg) {
        sendMsg(null, exchangeName, routingKey, msg, 0L);
    }

    /**
     * 发送延时消息
     *
     * @param exchangeName 交换机名称
     * @param routingKey   路由名称
     * @param msg          发送消息体
     * @param delay        延时队列的过期时间
     */
    public void sendMessage(String exchangeName, String routingKey, Object msg, Long delay) {
        sendMsg(null, exchangeName, routingKey, msg, delay);
    }

    /**
     * 发送延时消息
     *
     * @param exchangeName 交换机名称
     * @param routingKey   路由名称
     * @param msg          发送消息体
     * @param delay        延时队列的过期时间
     */
    public void sendMessage(String exchangeName, String routingKey, Object msg, Long delay, Collection<String> mqNames) {
        if (CollectionUtils.isEmpty(mqNames)) {
            return;
        }
        mqNames.forEach(mqName -> {
            try {
                sendMsg(mqName, exchangeName, routingKey, msg, delay);
            } catch (Exception e) {
                log.error("mq 信息配置异常,mqName:{}", mqName, e);
            }
        });
    }

    /**
     * 发送延时消息
     *
     * @param exchangeName 交换机名称
     * @param routingKey   路由名称
     * @param msg          发送消息体
     * @param delay        延时队列的过期时间
     */
    public void sendMessageAllMq(String exchangeName, String routingKey, Object msg, Long delay) {
        if (this.rabbitMultipleAutoConfiguration == null || MapUtils.isEmpty(this.rabbitMultipleAutoConfiguration.getRabbitTemplateMap())) {
            log.error("多 mq 信息未配置");
            return;
        }
        Map<String, RabbitTemplate> rabbitTemplateMap = this.rabbitMultipleAutoConfiguration.getRabbitTemplateMap();
        if (MapUtils.isNotEmpty(rabbitTemplateMap)) {
            rabbitTemplateMap.forEach((k, v) -> {
                try {
                    if (!Constant.commonMq.equals(k)) {
                        sendMsg(k, exchangeName, routingKey, msg, delay);
                    }
                } catch (Exception e) {
                    log.error("mq 信息配置异常,mqName:{}", k, e);
                }
            });
        }
    }

    /**
     * @param exchangeName 交换机名称
     * @param routingKey   路由名称
     * @param msg          发送消息体
     * @param delay        延时队列的过期时间
     * @param delayBean    延时队列的过期时间
     */
    public void sendMessage(String exchangeName, String routingKey, Object msg, Long delay, DelayBean delayBean, Collection<String> mqNames) {
        log.info("延迟消息中间队列发送消息，{}", JSON.toJSON(delayBean));
        if (CollectionUtils.isNotEmpty(mqNames)) {
            for (String mqName : mqNames) {
                try {
                    sendMsg(mqName, exchangeName, routingKey, msg, delay);
                } catch (Exception e) {
                    log.error("mq 信息获取异常, mqName:{}", mqName, e);
                }
            }
            return;
        }
        sendMsg(null, exchangeName, routingKey, msg, delay);
    }

    /**
     * @param mqName
     * @param routingKey
     * @param msg
     * @param delay
     */
    private void sendMsg(String mqName, String exchangeName, String routingKey, Object msg, Long delay) {
        RabbitTemplate rabbitTemplate;
        if (StringUtils.isEmpty(mqName)) {
            rabbitTemplate = this.rabbitTemplate;
        } else {
            try {
                rabbitTemplate = getRabbitTemplate(mqName);
            } catch (Exception e) {
                rabbitTemplate = this.rabbitTemplate;
            }
        }
        // 获取CorrelationData对象
        CorrelationData correlationData = this.correlationData(msg, exchangeName, routingKey);
        log.info("发送MQ消息，消息ID：{}，消息体:{}, exchangeName:{}, routingKey:{}, delay{}", correlationData.getId(), JSON.toJSONString(msg), exchangeName,
            routingKey, delay);
        // 如果延迟消息大于Integer.MAX_VALUE
        if (delay != null && delay > Integer.MAX_VALUE) {
            DelayBean delayBean = new DelayBean(correlationData.getId(), mqName, exchangeName, routingKey, delay, msg);
            // 发送到 公共mq 中 进行延迟消息再发送
            this.sendMessage(Constant.COMMON_DELAY_EXCHANGE, Constant.COMMON_MIDDLE_ROUTING, delayBean, (long) Integer.MAX_VALUE, Lists.newArrayList(Constant
                .commonMq));
            return;
        }
        // 发送消息
        this.convertAndSend(rabbitTemplate, exchangeName, routingKey, msg, correlationData, delay, null);
        // 记录延迟消息
        if (delay != null && delay > 0) {
            DelayBean delayBean = new DelayBean(correlationData.getId(), StringUtils.isEmpty(mqName) ? this.mqName : mqName, exchangeName, routingKey, delay, msg);
            // 发送mq 记录延迟消息
            sendMsg(Constant.commonMq, "", Constant.COMMON_DELAY_MSG_QUEUE, delayBean, 0L);
        }
    }


    /**
     * 用于实现消息发送到RabbitMQ交换器后接收ack回调。
     * 如果消息发送确认失败就进行重试。
     *
     * @param correlationData
     * @param ack
     * @param cause
     */
    public void confirm(String mqName, RabbitTemplate rabbitTemplate, org.springframework.amqp.rabbit.connection.CorrelationData correlationData, boolean ack, String
        cause) {
        // 消息回调确认失败处理
        if (!ack && correlationData instanceof CorrelationData) {
            CorrelationData correlationDataExtends = (CorrelationData) correlationData;

            //消息发送失败,就进行重试，重试过后还不能成功就记录到数据库
            /**
             * 最大重试次数
             */
            int maxRetryCount = 0;
            if (correlationDataExtends.getRetryCount() < maxRetryCount) {
                log.info("MQ消息发送失败，消息重发，消息ID：{}，重发次数：{}，消息体:{}, cause:{}", correlationDataExtends.getId(), correlationDataExtends.getRetryCount(), JSON
                    .toJSONString(correlationDataExtends.getMessage()), cause);
                // 将重试次数加一
                correlationDataExtends.setRetryCount(correlationDataExtends.getRetryCount() + 1);
                // 重发发消息
                this.convertAndSend(rabbitTemplate, correlationDataExtends.getExchange(), correlationDataExtends.getRoutingKey(), correlationDataExtends.getMessage(),
                    correlationDataExtends, 0L, correlationDataExtends.getMessage().getClass().getName());
            } else {
                //消息重试发送失败,将消息放到数据库等待补发
                log.error( "MQ消息重发失败，消息入库，消息ID：{}，消息体:{}, cause:{}", correlationData.getId(), JSON.toJSONString(correlationDataExtends.getMessage()),
                    cause);
                // 保存消息到数据库
                if (iMQExceptionData != null) {
                    iMQExceptionData.sendException(correlationDataExtends.getExchange(), correlationDataExtends.getRoutingKey(), correlationDataExtends.getMessage());
                } else {
                    // 统一处理 异常信息发送到队列
                    ErrorMsgBean errorMsgBean = new ErrorMsgBean(correlationDataExtends.getId(), mqName, correlationDataExtends.getExchange(), correlationDataExtends
                        .getRoutingKey(), correlationDataExtends.getMessage(), correlationDataExtends.getMessage().getClass().getName(), 1, 1, cause);
                    this.sendMessage("", Constant.COMMON_MSG_ERROR_QUEUE, errorMsgBean, 0L, Lists.newArrayList(Constant.commonMq));
                }
            }
        } else {
            log.info("消息发送成功,消息ID:{}", correlationData.getId());
        }
    }

    /**
     * 用于实现消息发送到RabbitMQ交换器，但无相应队列与交换器绑定时的回调。
     * 基本上来说线上不可能出现这种情况，除非手动将已经存在的队列删掉，否则在测试阶段肯定能测试出来。
     */
    public void returnedMessage(Message message, int replyCode, String replyText, String exchange, String routingKey) {

        // 如果是延迟消息，不进行处理
        if (StringUtils.getInt(message.getMessageProperties().getReceivedDelay()) > 0 || Constant.COMMON_MSG_ERROR_QUEUE.equals(routingKey)) {
            return;
        }
        log.error("MQ消息发送失败，replyCode:{}, replyText:{}，exchange:{}，routingKey:{}，消息体:{}", replyCode, replyText, exchange, routingKey, new String(message.getBody()));
        // 保存消息到数据库
        if (iMQExceptionData != null) {
            iMQExceptionData.sendException(exchange, routingKey, message);
        } else {
            String className = (String) message.getMessageProperties().getHeaders().get(Constant.className);
            // 统一处理 异常信息发送到队列
            ErrorMsgBean errorMsgBean = new ErrorMsgBean(message.getMessageProperties().getCorrelationId(), mqName, exchange, exchange, new String(message.getBody()), className,
                1, 2, replyText);
            this.sendMessage("", Constant.COMMON_MSG_ERROR_QUEUE, errorMsgBean, 0L, Lists.newArrayList(Constant.commonMq));
        }
    }

    /**
     * 消息相关数据（消息ID）
     *
     * @param message
     * @return
     */
    private CorrelationData correlationData(Object message, String exchange, String routingKey) {

        return new CorrelationData(UUIDTools.getUuid(), message, exchange, routingKey);
    }

    /**
     * 发送消息
     *
     * @param rabbitTemplate  发送模板
     * @param exchange        交换机名称
     * @param routingKey      路由key
     * @param msg             消息内容
     * @param correlationData 消息相关数据（消息ID）
     * @throws AmqpException
     */
    private void convertAndSend(RabbitTemplate rabbitTemplate, String exchange, String routingKey, final Object msg, CorrelationData correlationData, Long delay,
                                String className) throws AmqpException {
        try {
            rabbitTemplate.convertAndSend(exchange, routingKey, msg, message -> {
                // 设置消息属性-过期时间
                if (delay != null && delay > 0) {
                    message.getMessageProperties().setDelay(delay.intValue());
                }
                if (className != null) {
                    message.getMessageProperties().setHeader(Constant.className, className);
                }
                message.getMessageProperties().setMessageId(correlationData.getId());
                message.getMessageProperties().setCorrelationId(Slf4jUtils.putTraceIdIfAbsent());
                return message;
            }, correlationData);
        } catch (Exception e) {
            log.error("MQ消息发送异常，消息ID：{}，消息体:{}, exchangeName:{}, routingKey:{}", correlationData.getId(), JSON.toJSONString(msg), exchange,
                routingKey, e);
        }
    }

    /**
     * Description: 类型转换
     *
     * @param msg
     * @author wgy 2019-08-01 06:39:19
     * @return org.springframework.amqp.core.Message
     */
    public Message convertMessage(Object msg) {
        return rabbitTemplate.getMessageConverter().toMessage(msg, new MessageProperties());
    }

    private ConnectionFactory getPublisherConnectionFactory(RabbitTemplate rabbitTemplate) {
        return rabbitTemplate.isUsePublisherConnection() && rabbitTemplate.getConnectionFactory().getPublisherConnectionFactory() != null ?
                rabbitTemplate.getConnectionFactory().getPublisherConnectionFactory() : rabbitTemplate.getConnectionFactory();
    }

    /**
     * 获取mq 发行模板
     *
     * @param mqName
     * @return
     */
    private RabbitTemplate getRabbitTemplate(String mqName) throws Exception {
        if (rabbitMultipleAutoConfiguration != null && this.rabbitMultipleAutoConfiguration.getRabbitTemplateMap().containsKey(mqName)) {
            return rabbitMultipleAutoConfiguration.getRabbitTemplate(mqName);
        }
        throw new Exception("mq 信息 未配置");
    }

    public void setRabbitTemplate(RabbitTemplate rabbitTemplate) {
        this.rabbitTemplate = rabbitTemplate;
    }
}
