package com.icetech.oss.impl;

import java.io.File;
import java.io.InputStream;
import java.util.concurrent.TimeUnit;

import com.icetech.oss.OSSProperties;
import io.minio.DownloadObjectArgs;
import io.minio.GetObjectArgs;
import io.minio.GetObjectResponse;
import io.minio.GetPresignedObjectUrlArgs;
import io.minio.MinioClient;
import io.minio.ObjectWriteResponse;
import io.minio.PutObjectArgs;
import io.minio.StatObjectArgs;
import io.minio.StatObjectResponse;
import io.minio.UploadObjectArgs;
import io.minio.http.Method;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class MinioOssService extends BaseOssService<MinioClient> {
    @Override
    public boolean isExistFile(String fileName, String bucketName) {
        try {
            StatObjectResponse response = internalOssClient.statObject(StatObjectArgs.builder()
                    .bucket(bucketName)
                    .object(fileName)
                    .build());
            return response != null;
        } catch (Exception e) {
            log.error("OSS文件是否存在获取失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public String buildImageUrl(String bucketName, String fileName) {
        return ossProperties.getEndpoint() + (fileName.startsWith("/") ? fileName : "/" + fileName);
    }

    @Override
    public String getImageUrl(String bucketName, String name, Long expireSec) {
        return getImageUrl(publicOssClient, bucketName, name, expireSec);
    }

    @Override
    public String getInternalImageUrl(String name, String bucketName, Long expireSec) {
        return getImageUrl(internalOssClient, bucketName, name, expireSec);
    }

    public String getImageUrl(MinioClient minioClient, String bucketName, String name, Long expireSec) {
        try {
            if (expireSec == null || expireSec < 1) expireSec = ossProperties.getDefaultExpireSeconds();
            // 生成URL
            String url = minioClient.getPresignedObjectUrl(GetPresignedObjectUrlArgs.builder()
                    .bucket(bucketName)
                    .object(name)
                    .method(Method.GET)
                    .expiry(expireSec.intValue(), TimeUnit.SECONDS)
                    .build());
            return url;
        } catch (Exception e) {
            log.error("OSS文件生成访问地址失败|{}|{}|{}", bucketName, name, expireSec, e);
        }
        return null;
    }

    @Override
    public GetObjectResponse getOSS2InputStream(String name, String bucketName) {
        try {
            return internalOssClient.getObject(GetObjectArgs.builder()
                    .bucket(bucketName)
                    .object(name)
                    .build());
        } catch (Exception e) {
            log.error("OSS文件获取下载流失败|{}|{}", bucketName, name, e);
        }
        return null;
    }

    @Override
    public boolean getOSS2File(String fileName, String bucketName, File stored) {
        try {
            if (stored.exists() && !stored.delete()) {
                log.warn("OSS文件下载存储失败, 本地文件已存在且删除失败|{}|{}|{}", bucketName, fileName, stored.getPath());
                return false;
            }
            internalOssClient.downloadObject(DownloadObjectArgs.builder()
                    .bucket(bucketName)
                    .object(fileName)
                    .filename(stored.getPath())
                    .build());
            log.info("OSS文件下载存储成功|{}|{}|{}", bucketName, fileName, stored.getPath());
            return true;
        } catch (Exception e) {
            log.error("OSS文件下载存储失败|{}|{}|{}", bucketName, fileName, stored.getPath(), e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(File file, String fileName, String bucketName) {
        try {
            ObjectWriteResponse response = internalOssClient.uploadObject(UploadObjectArgs.builder()
                    .bucket(bucketName)
                    .object(fileName)
                    .filename(file.getPath())
                    .build());
            log.info("OSS文件上传成功|{}|{}|{}|{}|{}", file.getPath(), bucketName, fileName, response.etag(), response.versionId());
            return true;
        } catch (Exception e) {
            log.error("OSS文件上传失败|{}|{}|{}", file.getPath(), bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(InputStream inputStream, String fileName, String bucketName) {
        try {
            ObjectWriteResponse response = internalOssClient.putObject(PutObjectArgs.builder()
                    .bucket(bucketName)
                    .object(fileName)
                    .stream(inputStream, -1, PutObjectArgs.MAX_PART_SIZE)
                    .build());
            log.info("OSS输入流上传成功|{}|{}|{}|{}", bucketName, fileName, response.etag(), response.versionId());
            return true;
        } catch (Exception e) {
            log.error("OSS输入流上传失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    public MinioOssService(OSSProperties ossProperties) {
        super(ossProperties, MinioClient.builder()
                .endpoint(ossProperties.getEndpoint())
                .credentials(ossProperties.getAccessKeyId(), ossProperties.getAccessKeySecret())
                .build(), MinioClient.builder()
                .endpoint(ossProperties.getInternalEndpoint())
                .credentials(ossProperties.getAccessKeyId(), ossProperties.getAccessKeySecret())
                .build());
    }
}
