package com.icetech.oss.impl;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.Date;

import com.aliyun.oss.ClientBuilderConfiguration;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.PutObjectResult;
import com.icetech.oss.OSSProperties;
import com.icetech.oss.io.OSSObjectInputStream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class AliyunOssService extends BaseOssService<OSS> {
    @Override
    public boolean isExistFile(String fileName, String bucketName) {
        try {
            return internalOssClient.doesObjectExist(bucketName, fileName);
        } catch (Exception e) {
            log.error("OSS文件是否存在获取失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public String buildImageUrl(String bucketName, String fileName) {
        return "http://" + ossProperties.getBucketName() + "." + ossProperties.getEndpoint() + (fileName.startsWith("/") ? fileName : "/" + fileName);
    }

    @Override
    public String getImageUrl(String bucketName, String name, Long expireSec) {
        return getImageUrl(publicOssClient, bucketName, name, expireSec);
    }

    @Override
    public String getInternalImageUrl(String name, String bucketName, Long expireSec) {
        return getImageUrl(internalOssClient, bucketName, name, expireSec);
    }

    public String getImageUrl(OSS ossClient, String bucketName, String name, Long expireSec) {
        try {
            if (expireSec == null || expireSec < 1) expireSec = ossProperties.getDefaultExpireSeconds();
            // 设置URL过期时间为1小时
            Date expiration = new Date(System.currentTimeMillis() + expireSec * 1000);
            // 生成URL
            URL url = ossClient.generatePresignedUrl(bucketName, name, expiration);
            return url.toString().replace("http://", "https://");
        } catch (Exception e) {
            log.error("OSS文件生成访问地址失败|{}|{}|{}", bucketName, name, expireSec, e);
        }
        return null;
    }

    @Override
    public InputStream getOSS2InputStream(String name, String bucketName) {
        try {
            OSSObject ossObj = internalOssClient.getObject(bucketName, name);
            if (ossObj == null || ossObj.getObjectContent() == null) {
                log.warn("OSS文件获取下载流获取为空|{}|{}", bucketName, name);
                return null;
            }
            return new OSSObjectInputStream(ossObj);
        } catch (Exception e) {
            log.error("OSS文件获取下载流失败|{}|{}", bucketName, name, e);
        }
        return null;
    }

    @Override
    public boolean getOSS2File(String fileName, String bucketName, File stored) {
        try {
            internalOssClient.getObject(new GetObjectRequest(bucketName, fileName), stored);
            log.info("OSS文件下载存储成功|{}|{}|{}", bucketName, fileName, stored.getPath());
            return true;
        } catch (Exception e) {
            log.error("OSS文件下载存储失败|{}|{}|{}", bucketName, fileName, stored.getPath(), e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(File file, String fileName, String bucketName) {
        try {
            PutObjectResult result = internalOssClient.putObject(bucketName, fileName, file);
            log.info("OSS文件上传成功|{}|{}|{}|{}|{}", file.getPath(), bucketName, fileName, result.getETag(), result.getVersionId());
            return true;
        } catch (Exception e) {
            log.error("OSS文件上传失败|{}|{}|{}", file.getPath(), bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(InputStream inputStream, String fileName, String bucketName) {
        try {
            PutObjectResult result = internalOssClient.putObject(bucketName, fileName, inputStream);
            log.info("OSS文件上传成功|{}|{}|{}|{}", bucketName, fileName, result.getETag(), result.getVersionId());
            return true;
        } catch (Exception e) {
            log.error("[oss异常]e", e);
        }
        return false;
    }

    public AliyunOssService(OSSProperties ossProperties, ClientBuilderConfiguration clientBuilderConfiguration) {
        super(ossProperties, new OSSClientBuilder().build(
                ossProperties.getEndpoint(),
                ossProperties.getAccessKeyId(),
                ossProperties.getAccessKeySecret(), clientBuilderConfiguration), new OSSClientBuilder().build(
                ossProperties.getInternalEndpoint(),
                ossProperties.getAccessKeyId(),
                ossProperties.getAccessKeySecret(), clientBuilderConfiguration));
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            publicOssClient.shutdown();
            internalOssClient.shutdown();
        }));
    }
}
