package com.icetech.oss.impl;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.util.Base64;
import java.util.function.Function;

import com.icetech.common.utils.StringUtils;
import com.icetech.oss.OssService;
import com.icetech.oss.OSSProperties;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FilenameUtils;

@Slf4j
public abstract class BaseOssService<T> implements OssService {
    @Getter
    protected final OSSProperties ossProperties;
    @Getter
    protected final String tempPath;
    @Getter
    protected final T publicOssClient;
    @Getter
    protected final T internalOssClient;

    @Override
    public boolean isExistFile(String fileName) {
        return isExistFile(fileName, ossProperties.getBucketName());
    }

    @Override
    public String buildImageUrl(String fileName) {
        return buildImageUrl(ossProperties.getBucketName(), fileName);
    }

    @Override
    public String getImageUrl(String name) {
        return getImageUrlTime(name, ossProperties.getDefaultExpireSeconds());
    }

    @Override
    public String getImageUrlTime(String name, Long expireSec) {
        return getImageUrl(ossProperties.getBucketName(), name, expireSec);
    }

    @Override
    public String getImageUrl(String name, String bucketName) {
        return getImageUrl(bucketName, name, ossProperties.getDefaultExpireSeconds());
    }

    @Override
    public String getInternalImageUrl(String name) {
        return getInternalImageUrl(name, ossProperties.getDefaultExpireSeconds());
    }

    @Override
    public String getInternalImageUrl(String name, Long expireSec) {
        return getInternalImageUrl(name, ossProperties.getBucketName(), expireSec);
    }

    @Override
    public InputStream getOSS2InputStream(String name) {
        return getOSS2InputStream(name, ossProperties.getBucketName());
    }

    @Override
    public File getOSS2File(String fileName) {
        return getOSS2File(fileName, ossProperties.getBucketName());
    }

    @Override
    public File getOSS2File(String fileName, String bucketName) {
        String saveName = FilenameUtils.getName(fileName);
        File stored = new File(tempPath + saveName);
        if (getOSS2File(fileName, bucketName, stored)) {
            return stored;
        }
        return null;
    }

    @Override
    public boolean uploadBase64(String base64, String fileName) {
        return uploadBase64(base64, fileName, ossProperties.getBucketName());
    }

    @Override
    public boolean uploadBase64(String base64, String fileName, String bucketName) {
        byte[] bytes = Base64.getDecoder().decode(base64);
        return uploadFileStream(bytes, fileName, bucketName);
    }

    @Override
    public boolean uploadFileStream(byte[] fileStream, String fileName) {
        return uploadFileStream(fileStream, fileName, ossProperties.getBucketName());
    }

    @Override
    public boolean uploadFileStream(byte[] fileStream, String fileName, String bucketName) {
        try (ByteArrayInputStream bais = new ByteArrayInputStream(fileStream)) {
            return uploadFile(bais, fileName, bucketName);
        } catch (Exception e) {
            log.error("OSS服务上传文件字节失败|{}|{}", bucketName, fileName, e);
        }
        return false;
    }

    @Override
    public boolean uploadFile(File file, String fileName) {
        return uploadFile(file, fileName, ossProperties.getBucketName());
    }

    @Override
    public boolean uploadFile(InputStream inputStream, String fileName) {
        return uploadFile(inputStream, fileName, ossProperties.getBucketName());
    }

    public BaseOssService(OSSProperties ossProperties, T publicOssClient, T internalOssClient) {
        this(ossProperties, prop -> publicOssClient, prop -> internalOssClient);
    }

    public BaseOssService(OSSProperties ossProperties, Function<OSSProperties, T> publicOssClient, Function<OSSProperties, T> internalOssClient) {
        this.ossProperties = ossProperties;
        String tempPath = StringUtils.isBlank(ossProperties.getDownPath()) ? System.getProperty("java.io.tmpdir") : ossProperties.getDownPath();

        File tempDir = new File(tempPath);
        if (!tempDir.exists() || tempDir.isFile()) {
            boolean mkdirs = tempDir.mkdirs();
            if (mkdirs) {
                log.info("创建临时存储文件夹: " + tempPath);
            } else {
                throw new IllegalArgumentException("无法创建临时存储文件夹: " + tempPath);
            }
        }
        this.tempPath = tempDir.getPath() + File.separator;
        this.publicOssClient = publicOssClient.apply(ossProperties);
        this.internalOssClient = internalOssClient.apply(ossProperties);
        log.info("{} OSS客户端初始化成功|{}|{}|{}", ossProperties.getType(), ossProperties.getEndpoint(), ossProperties.getInternalEndpoint(), ossProperties.getBucketName());
    }
}
