package com.icetech.log;

import ch.qos.logback.classic.pattern.MessageConverter;
import ch.qos.logback.classic.spi.ILoggingEvent;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Description 日志脱敏控制器
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2021/8/2 4:36 下午
 */
public class SensitiveDataConverter extends MessageConverter {

    @Override
    public String convert(ILoggingEvent event) {
        // 获取原始日志
        String oriLogMsg = event.getFormattedMessage();

        // 获取脱敏后的日志
        String afterLogMsg = invokeMsg(oriLogMsg);
        return afterLogMsg;
    }

    /**
     * 日志脱敏开关
     */
    private static String converterCanRun = "true";
    /**
     * 日志脱敏关键字
     */
    private static String sensitiveDataKeys = "accessToken,password,base64Str,payCode";

    /**
     * 处理日志字符串，返回脱敏后的字符串
     *
     * @param oriMsg oriMsg
     * @return String
     */
    public String invokeMsg(final String oriMsg) {
        String tempMsg = oriMsg;
        if ("true".equals(converterCanRun)) {
            // 处理字符串
            if (sensitiveDataKeys != null && sensitiveDataKeys.length() > 0) {
                String[] keysArray = sensitiveDataKeys.split(",");
                for (String key : keysArray) {
                    int index = -1;
                    do {
                        index = tempMsg.indexOf(key, index + 1);
                        if (index != -1) {
                            // 判断key是否为单词字符
                            if (isWordChar(tempMsg, key, index)) {
                                continue;
                            }
                            // 寻找值的开始位置
                            int valueStart = getValueStartIndex(tempMsg, index + key.length());

                            // 查找值的结束位置（逗号，分号）........................
                            int valueEnd = getValueEndEIndex(tempMsg, valueStart);

                            // 对获取的值进行脱敏 
                            String subStr = tempMsg.substring(valueStart, valueEnd);
                            subStr = convertMsg(subStr, key);
                            ///
                            tempMsg = tempMsg.substring(0, valueStart) + subStr + tempMsg.substring(valueEnd);
                        }
                    } while (index != -1);
                }
            }
        }
        return tempMsg;
    }


    private static Pattern pattern = Pattern.compile("[0-9a-zA-Z]");

    /**
     * 判断从字符串msg获取的key值是否为单词 ， index为key在msg中的索引值
     *
     * @return
     */
    private boolean isWordChar(String msg, String key, int index) {
        // 必须确定key是一个单词............................
        if (index != 0) { // 判断key前面一个字符
            char preCh = msg.charAt(index - 1);
            Matcher match = pattern.matcher(preCh + "");
            if (match.matches()) {
                return true;
            }
        }
        // 判断key后面一个字符
        char nextCh = msg.charAt(index + key.length());
        Matcher match = pattern.matcher(nextCh + "");
        if (match.matches()) {
            return true;
        }
        return false;
    }

    /**
     * 获取value值的开始位置
     *
     * @param msg        要查找的字符串
     * @param valueStart 查找的开始位置
     * @return
     */
    private int getValueStartIndex(String msg, int valueStart) {
        // 寻找值的开始位置.................................
        do {
            char ch = msg.charAt(valueStart);
            if (ch == ':' || ch == '=') { // key与 value的分隔符
                valueStart++;
                ch = msg.charAt(valueStart);
                if (ch == '"') {
                    valueStart++;
                }
                break;    // 找到值的开始位置
            } else {
                valueStart++;
            }
        } while (true);
        return valueStart;
    }

    /**
     * 获取value值的结束位置
     *
     * @return
     */
    private int getValueEndEIndex(String msg, int valueEnd) {
        do {
            if (valueEnd == msg.length()) {
                break;
            }
            char ch = msg.charAt(valueEnd);

            if (ch == '"') { // 引号时，判断下一个值是结束，分号还是逗号决定是否为值的结束
                if (valueEnd + 1 == msg.length()) {
                    break;
                }
                char nextCh = msg.charAt(valueEnd + 1);
                if (nextCh == ';' || nextCh == ',') {
                    // 去掉前面的 \  处理这种形式的数据
                    while (valueEnd > 0) {
                        char preCh = msg.charAt(valueEnd - 1);
                        if (preCh != '\\') {
                            break;
                        }
                        valueEnd--;
                    }
                    break;
                } else {
                    valueEnd++;
                }
            } else if (ch == ';' || ch == ',' || ch == '}') {
                break;
            } else {
                valueEnd++;
            }

        } while (true);
        return valueEnd;
    }

    private String convertMsg(String subMsg, String key) {
        if ("phone".equals(key)) {
            return SensitiveInfoUtils.mobilePhone(subMsg);
        }
        if ("accessToken".equals(key)) {
            return SensitiveInfoUtils.token(subMsg);
        }
        return "******";
    }

    public static void main(String[] args) {
        String tempMsg = "[device-p2p]|2021-08-02 18:30:38.253|XNIO-1 task-1|INFO |7c2fb0bbf6a0420ea2a857cc474ae183|com.icetech.rpc.trace" +
                ".TraceIdFilter[55]|[client][SassUserApi.getCurrentWebUser]receive:[{\"accessToken\":\"32cc81b56bd54c19ae11b116e54999a4\"}]";
        String tempMsg1 = "{\"reason\":\"成功 \",\"result\":{\"payCode\":\"JH2131171027170837443588J6\",\"realname\":\"李哪娜\",\"bankcard\":\"6226430106137525\"," +
                "\"idcard\":\"130333198901192762\",\"mobile\":\"13210141605\",\"res\":\"1\",\"message\":\"验证成功\"},\"error_code\":0}";
        SensitiveDataConverter sc = new SensitiveDataConverter();
        System.out.println(sc.invokeMsg(tempMsg));
        System.out.println(sc.invokeMsg(tempMsg1));
    }
}

