package com.icetech.excel.convert;


import com.icetech.common.utils.SpringUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.excel.DictService;
import com.icetech.excel.anno.ExcelDictFormat;
import com.icetech.excel.utils.ExcelUtil;

import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Description DictContext
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 
 */
public class DictContext extends ConcurrentHashMap<String, String> {

    public static final String VALUE_PREFIX = "$";
    public static final String TITLE_PREFIX = "#";

    protected static Class<? extends DictContext> contextClass = DictContext.class;

    protected static final ThreadLocal<? extends DictContext> THREAD_LOCAL = ThreadLocal.withInitial(() -> {
        try {
            return contextClass.newInstance();
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    });


    public DictContext() {
        super();
    }

    public static DictContext getCurrentContext() {
        return THREAD_LOCAL.get();
    }

    /**
     * Description: 获取 字典项value 值
     * Version1.0 2022-03-07 by wgy 创建
     * 
     * @param label     标签
     * @param anno      注解
     * @return java.lang.String       
     */
    public String getValue(String label, ExcelDictFormat anno) {
        String type = anno.dictType();
        String value;
        if (StringUtils.isBlank(type)) {
            value = ExcelUtil.reverseByExp(label, anno.readConverterExp(), anno.separator());
        } else {
            DictContext dictContext = THREAD_LOCAL.get();
            String key = VALUE_PREFIX + type + "-" + label;
            if (Objects.nonNull(dictContext) && StringUtils.isNotEmpty(dictContext.get(key))) {
                return dictContext.get(key);
            }
            value = SpringUtils.getBean(DictService.class).getDictValue(type, label, anno.separator());
            set(key, value);
        }
        return value;
    }

    /**
     * Description: 获取 字典项value 值
     * Version1.0 2022-03-07 by wgy 创建
     *
     * @param value     值
     * @param anno      注解
     * @return java.lang.String
     */
    public String getLabel(String value, ExcelDictFormat anno) {
        String type = anno.dictType();
        String label;
        if (StringUtils.isBlank(type)) {
            label = ExcelUtil.convertByExp(value, anno.readConverterExp(), anno.separator());
        } else {
            DictContext dictContext = THREAD_LOCAL.get();
            String key = TITLE_PREFIX + type + "-" + value;
            if (Objects.nonNull(dictContext) && StringUtils.isNotEmpty(dictContext.get(key))) {
                return dictContext.get(key);
            }
            label = SpringUtils.getBean(DictService.class).getDictLabel(type, value, anno.separator());
            set(key, label);
        }
        return label;
    }

    /**
     * puts the key, value into the map. a null value will remove the key from the map
     *
     * @param key
     * @param value
     */
    public void set(String key, String value) {
        if (value != null) {
            put(key, value);
        } else {
            remove(key);
        }
    }

    /**
     * unsets the THREAD_LOCAL context. Done at the end of the request.
     */
    public void unset() {
        THREAD_LOCAL.remove();
    }
}
