package com.icetech.db.mybatis.base.service;

import java.util.Collection;
import java.util.List;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.core.toolkit.support.SFunction;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.IService;
import com.icetech.db.mybatis.base.mapper.SuperMapper;
import org.springframework.transaction.annotation.Transactional;

/**
 * Description 基础service 公共方法扩展
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @param <T> the type parameter
 * @author wgy
 * @version 1.0.5
 * @since Date 2021/4/29 10:33 AM
 */
public interface IBaseService<T> extends IService<T> {
    /**
     * <p>根据实体类查询单条记录</p>
     * <p>SQL层强制返回单条记录, 仅限于MySQL</p>
     *
     * @param entity 实体类, 非空字段为查询条件
     * @return 查询结果
     */
    default T selectLimitOne(T entity) {
        return selectLimitOne(Wrappers.query(entity));
    }

    /**
     * <p>根据查询条件查询单条记录</p>
     * <p>SQL层强制返回单条记录, 仅限于MySQL</p>
     *
     * @param wrapper 查询条件
     * @return 查询结果
     */
    default T selectLimitOne(QueryWrapper<T> wrapper) {
        return getSuperMapper().selectLimitOne(wrapper);
    }

    /**
     * <p>根据查询条件查询单条记录</p>
     * <p>SQL层强制返回单条记录, 仅限于MySQL</p>
     *
     * @param wrapper 查询条件
     * @return 查询结果
     */
    default T selectLimitOne(LambdaQueryWrapper<T> wrapper) {
        return getSuperMapper().selectLimitOne(wrapper);
    }

    /**
     * <p>根据实体类查询多条记录</p>
     *
     * @param entity 实体类, 非空字段为查询条件
     * @return 查询结果
     */
    default List<T> list(T entity) {
        return getSuperMapper().selectList(entity);
    }

    /**
     * <p>根据实体类分页查询多条记录</p>
     *
     * @param entity  实体类, 非空字段为查询条件
     * @param current 当前页数
     * @param size    每页大小
     * @return 分页查询结果
     */
    default Page<T> page(T entity, long current, long size) {
        return getSuperMapper().selectPage(entity, current, size);
    }

    /**
     * <p>根据查询条件分页查询多条记录</p>
     *
     * @param wrapper 查询条件
     * @param current 当前页数
     * @param size    每页大小
     * @return 分页查询结果
     */
    default Page<T> page(Wrapper<T> wrapper, long current, long size) {
        return getSuperMapper().selectPage(wrapper, current, size);
    }

    default <C> C selectLimitOneColumn(SFunction<T, C> column, T entity) {
        return getSuperMapper().selectLimitOneColumn(column, entity);
    }

    default <C> C selectLimitOneColumn(SFunction<T, C> column, LambdaQueryWrapper<T> wrapper) {
        return getSuperMapper().selectLimitOneColumn(column, wrapper);
    }

    default <C> C getOneColumn(SFunction<T, C> column, T entity) {
        return getSuperMapper().selectOneColumn(column, entity);
    }

    default <C> C getOneColumn(SFunction<T, C> column, LambdaQueryWrapper<T> wrapper) {
        return getSuperMapper().selectOneColumn(column, wrapper);
    }

    default <C> List<C> selectColumnList(SFunction<T, C> column, T entity) {
        return getSuperMapper().selectColumnList(column, entity);
    }

    default <C> List<C> selectColumnList(SFunction<T, C> column, LambdaQueryWrapper<T> wrapper) {
        return getSuperMapper().selectColumnList(column, wrapper);
    }

    /**
     * 根据字段批量更新
     *
     * @param list   实体对象集合
     * @param column 列条件
     */
    @Transactional(rollbackFor = Exception.class)
    default boolean updateBatchByColumn(Collection<T> list, SFunction<T, ?> column) {
        return updateBatchByColumn(list, column, DEFAULT_BATCH_SIZE);
    }

    /**
     * 根据字段批量更新
     *
     * @param list      实体对象集合
     * @param batchSize 更新批次数量
     */
    boolean updateBatchByColumn(Collection<T> list, SFunction<T, ?> column, int batchSize);

    SuperMapper<T> getSuperMapper();

    LambdaQueryWrapper<T> getLambdaQueryWrapper();

    LambdaQueryWrapper<T> getLambdaQueryWrapper(T entity);
}
