package com.icetech.db.config;

import com.baomidou.mybatisplus.core.parser.ISqlParserFilter;
import com.baomidou.mybatisplus.core.parser.SqlParserHelper;
import com.baomidou.mybatisplus.extension.plugins.handler.TenantLineHandler;
import com.icetech.common.content.TenantContextHolder;
import com.icetech.common.properties.TenantProperties;
import com.icetech.common.utils.StringUtils;
import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.NullValue;
import net.sf.jsqlparser.expression.StringValue;
import org.apache.ibatis.mapping.MappedStatement;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.util.CollectionUtils;

/**
 * Description 多租户自动配置
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wgy
 * @version 1.0 @Date 2023/11/2 9:49 AM
 */
@EnableConfigurationProperties(TenantProperties.class)
public class TenantAutoConfigure {
    
    @Autowired
    private TenantProperties tenantProperties;

    @Bean
    public TenantLineHandler tenantLineHandler() {
        return new TenantLineHandler() {

            @Override
            public Expression getTenantId() {
                String tenant = TenantContextHolder.getTenantId();
                if (tenant != null) {
                    return new StringValue(TenantContextHolder.getTenantId());
                }
                return new NullValue();
            }

            @Override
            public boolean ignoreTable(String tableName) {
                String tenant = TenantContextHolder.getTenantId();
                if (StringUtils.isEmpty(tenant)) {
                    return true;
                }
                if (!CollectionUtils.isEmpty(tenantProperties.getIncludeTables())) {
                    return tenantProperties.getIncludeTables().stream().noneMatch((e) -> e.equalsIgnoreCase(tableName));
                }
                return tenantProperties.getIgnoreTables().stream().anyMatch((e) -> e.equalsIgnoreCase(tableName));
            }

            /**
             * 获取租户列名
             */
            @Override
            public String getTenantIdColumn() {
                return tenantProperties.getTenantIdColumn();
            }
        };
    }

    /**
     * 过滤不需要根据租户隔离的MappedStatement
     */
    @Bean
    public ISqlParserFilter sqlParserFilter() {
        return metaObject -> {
            MappedStatement ms = SqlParserHelper.getMappedStatement(metaObject);
            return tenantProperties.getIgnoreSqlList().stream().anyMatch((e) -> e.equalsIgnoreCase(ms.getId()));
        };
    }
}