package com.icetech.common.utils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.function.Function;

public class NumberUtils extends org.apache.commons.lang3.math.NumberUtils {
    public static int toPrimitive(Integer value) {
        return toPrimitive(value, 0);
    }

    public static int toPrimitive(Integer value, int defaultValue) {
        return value == null ? defaultValue : value;
    }

    public static long toPrimitive(Long value) {
        return value == null ? 0 : value;
    }

    public static long toPrimitive(Long value, long defaultValue) {
        return value == null ? defaultValue : value;
    }

    public static float toPrimitive(Float value) {
        return value == null ? 0 : value;
    }

    public static float toPrimitive(Float value, float defaultValue) {
        return value == null ? defaultValue : value;
    }

    public static double toPrimitive(Double value) {
        return value == null ? 0 : value;
    }

    public static double toPrimitive(Double value, double defaultValue) {
        return value == null ? defaultValue : value;
    }

    public static int parseInteger(Object value) {
        return parseInteger(value, 0);
    }

    public static int parseInteger(Object value, int defaultValue) {
        return parseNumber(value, defaultValue, Number::intValue, Integer::valueOf);
    }

    public static long parseLong(Object value) {
        return parseLong(value, 0L);
    }

    public static long parseLong(Object value, long defaultValue) {
        return parseNumber(value, defaultValue, Number::longValue, Long::valueOf);
    }

    /**
     * 转换浮点型, 为空则返回 <code>0.0F</code>
     *
     * @param value
     * @return
     */
    public static float parseFloat(Object value) {
        return parseFloat(value, 0.0F);
    }

    /**
     * 转换浮点型，无需关心是否为空
     *
     * @param value
     * @param defaultValue
     * @return
     */
    public static float parseFloat(Object value, float defaultValue) {
        return parseNumber(value, defaultValue, Number::floatValue, Float::valueOf);
    }

    /**
     * 转换浮点型，无需关心是否为空
     *
     * @param value
     * @param pattern
     * @return
     */
    public static float parseFloat(Object value, String pattern) {
        DecimalFormat format = new DecimalFormat(pattern);
        return parseFloat(format.format(parseFloat(value)));
    }

    /**
     * 转换双精度型，无需关心是否为空
     *
     * @param value
     * @return
     */
    public static double parseDouble(Object value) {
        return parseDouble(value, 0.0D);
    }

    public static double parseDouble(Object value, double defaultValue) {
        return parseNumber(value, defaultValue, Number::doubleValue, Double::valueOf);
    }

    /**
     * 转换双精度型，无需关心是否为空
     *
     * @param value
     * @return pattern
     */
    public static double parseDouble(Object value, String pattern) {
        DecimalFormat format = new DecimalFormat(pattern);
        return parseDouble(format.format(parseDouble(value)));
    }

    private static <T> T parseNumber(Object value, T defaultValue, Function<Number, T> number, Function<String, T> string) {
        if (value == null) return defaultValue;
        if (value instanceof Number) return number.apply((Number) value);
        String content = value.toString().trim();
        if (content.isEmpty()) return defaultValue;
        return string.apply(content);
    }

    /**
     * 转换为BigDecimal
     *
     * @param value 待转换数字/对象
     * @return value 为 <code>null</code> 时返回 {@link BigDecimal#ZERO}, 返回其他值使用 {@link #parseDecimal(Object, BigDecimal)}
     */
    public static BigDecimal parseDecimal(Object value) {
        return parseDecimal(value, BigDecimal.ZERO);
    }

    /**
     * 转换为BigDecimal
     *
     * @param value        待转换数字/对象
     * @param defaultValue value 为 <code>null</code> 或空字符串时的默认值
     * @return
     */
    public static BigDecimal parseDecimal(Object value, BigDecimal defaultValue) {
        if (value == null) return defaultValue;

        if (value instanceof BigDecimal)
            return (BigDecimal) value;
        if (value instanceof Double)
            return BigDecimal.valueOf((Double) value);
        if (value instanceof Float)
            return new BigDecimal(Float.toString((Float) value));
        if (value instanceof Long)
            return new BigDecimal((Long) value);
        if (value instanceof Integer)
            return new BigDecimal((Integer) value);
        if (value instanceof Short)
            return new BigDecimal((Short) value);

        String middle = value.toString().trim();
        if (middle.isEmpty()) return defaultValue;
        return new BigDecimal(middle);
    }

    /**
     * 十进制加法
     *
     * @param num  被加数
     * @param nums 加数
     * @return
     */
    public static BigDecimal decimalAdd(Object num, Object... nums) {
        return decimalCalculate(BigDecimal::add, num, nums);
    }

    /**
     * 十进制减法
     *
     * @param num  被减数
     * @param nums 减数
     * @return
     */
    public static BigDecimal decimalSubtract(Object num, Object... nums) {
        return decimalCalculate(BigDecimal::subtract, num, nums);
    }

    /**
     * 十进制乘法
     *
     * @param num  被乘数
     * @param nums 乘数
     * @return
     */
    public static BigDecimal decimalMultiply(Object num, Object... nums) {
        return decimalCalculate(BigDecimal::multiply, num, nums);
    }

    /**
     * 十进制除法, 保留两位小数并截断超过的小数位({@link RoundingMode#DOWN}), 其他小数位处理方式使用 {@link #decimalDivideScale(int, RoundingMode, Object, Object...)}
     *
     * @param num  被除数
     * @param nums 除数
     * @return
     */
    public static BigDecimal decimalDivide(Object num, Object... nums) {
        return decimalDivideScale(2, RoundingMode.DOWN, num, nums);
    }

    /**
     * 十进制除法
     *
     * @param scale 小数位位数
     * @param mode  小数位截断方式
     * @param num   被除数
     * @param nums  除数
     * @return
     */
    public static BigDecimal decimalDivideScale(int scale, RoundingMode mode, Object num, Object... nums) {
        return decimalCalculate((decimal1, decimal2) -> decimal1.divide(decimal2, scale, mode), num, nums);
    }

    private static BigDecimal decimalCalculate(DecimalCalculation calculation, Object num, Object... nums) {
        BigDecimal decimal = parseDecimal(num, BigDecimal.ZERO);
        if (nums == null || nums.length == 0) return decimal;
        for (int i = 0; i < nums.length; i++) {
            BigDecimal middle = parseDecimal(nums[i], null);
            if (middle == null) continue;
            decimal = calculation.calculate(decimal, middle);
        }
        return decimal;
    }

    private interface DecimalCalculation {
        BigDecimal calculate(BigDecimal decimal1, BigDecimal decimal2);
    }

    /**
     * 转换字符型，无需关心是否为空
     *
     * @param value
     * @param pattern
     * @return
     */
    public static String parseString(Object value, String pattern) {
        DecimalFormat format = new DecimalFormat(pattern);
        return format.format(parseDouble(value));
    }

    /**
     * 数字+小写字母+大写字母, 去除 <code>olOI</code>
     */
    public static char[] RADIX_58 = "0123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnpqrstuvwxyz".toCharArray();
    /**
     * 数字+小写字母+大写字母, 去除 <code>lOI</code>
     */
    public static char[] RADIX_59 = "0123456789abcdefghijkmnopqrstuvwxyzABCDEFGHJKLMNPQRSTUVWXYZ".toCharArray();
    /**
     * 数字+小写字母+大写字母
     */
    public static char[] RADIX_62 = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ".toCharArray();

    public static String radix59(BigInteger value) {
        return radix(value, RADIX_59);
    }

    public static String radix62(BigInteger value) {
        return radix(value, RADIX_62);
    }

    public static String radix(BigInteger value, char[] chars) {
        StringBuilder builder = new StringBuilder();
        BigInteger divider = BigInteger.valueOf(chars.length);
        do {
            BigInteger[] values = value.divideAndRemainder(divider);
            builder.append(chars[values[1].intValue()]);
            value = values[0];
        } while (value.compareTo(BigInteger.ZERO) > 0);

        return builder.reverse().toString();
    }

    public static BigInteger radixOf(String value, char[] chars) {
        BigInteger result = BigInteger.ZERO;
        char[] values = value.toCharArray();
        int valueLength = values.length;
        BigInteger radix = BigInteger.valueOf(chars.length);
        for (int index = 0, r = valueLength - 1; index < valueLength; index++, r--) {
            char val = values[r];
            for (int i = 0; i < chars.length; i++) {
                if (chars[i] == val) {
                    BigInteger step = radix.pow(index).multiply(new BigInteger(String.valueOf(i)));
                    result = result.add(step);
                    break;
                }
            }
        }

        return result;
    }
}
