package com.icetech.common.utils;

import com.alibaba.fastjson.JSONObject;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Type;
import java.util.List;

/**
 * ObjectResponse 工具类
 */
@Slf4j
public class ResponseUtils {
    public static boolean isSuccess(String responseBody) {
        ObjectResponse<?> response = getObjectResponse(responseBody);
        return ObjectResponse.isSuccess(response);
    }

    public static ObjectResponse<?> getObjectResponse(String responseBody) {
        if (StringUtils.isBlank(responseBody)) {
            return null;
        }
        ObjectResponse<?> response = null;
        try {
            response = JSONObject.parseObject(responseBody, ObjectResponse.class);
            if (response == null) {
                log.warn("responseBody不能转换为ObjectResponse对象: {}", responseBody);
            }
        } catch (Exception e) {
            log.error("转换为ObjectResponse对象异常: {}", responseBody, e);
        }
        return response;
    }

    public static <T> ObjectResponse<T> getObjectResponse(String responseBody, Class<T> clazz) {
        if (StringUtils.isBlank(responseBody)) {
            return null;
        }
        ObjectResponse<T> response = null;
        try {
            Type type = ReflectUtils.getWrapperType(ObjectResponse.class, clazz);
            response = JSONObject.parseObject(responseBody, type);
            if (response == null) {
                log.warn("responseBody不能转换为ObjectResponse对象: {}", responseBody);
            }
        } catch (Exception e) {
            log.error("转换为ObjectResponse对象异常: {}", responseBody, e);
        }
        return response;
    }

    public static <T> ObjectResponse<List<T>> getListObjectResponse(String responseBody, Class<T> clazz) {
        if (StringUtils.isBlank(responseBody)) {
            return null;
        }
        ObjectResponse<List<T>> response = null;
        try {
            Type type = ReflectUtils.getLoopWrapperType(ObjectResponse.class, List.class, clazz);
            response = JSONObject.parseObject(responseBody, type);
            if (response == null) {
                log.warn("responseBody不能转换为ObjectResponse对象: {}", responseBody);
            }
        } catch (Exception e) {
            log.error("转换为ObjectResponse对象异常: {}", responseBody, e);
        }
        return response;
    }

    /**
     * 直接返回 {@link ObjectResponse} JSON串
     *
     * @return
     */
    public static String setNullMsgResponse(String code) {
        return setResponse(code, null);
    }

    /**
     * 直接返回 {@link ObjectResponse} JSON串
     *
     * @return
     */
    public static String setResponse(String code, String msg) {
        return setResponse(code, msg, null);
    }

    /**
     * 直接返回 {@link ObjectResponse} JSON串
     *
     * @return
     */
    public static String setResponse(String code, String msg, Object data) {
        return JSONObject.toJSONString(ObjectResponse.instance(code, msg, data));
    }

    /**
     * @deprecated 使用 {@link ObjectResponse#success()}
     * @param <T>
     * @return
     */
    @Deprecated
    public static <T> ObjectResponse<T> returnSuccessResponse(){
        return new ObjectResponse<T>(CodeConstants.SUCCESS, CodeConstants.getName(CodeConstants.SUCCESS));
    }
    
    /**
     * 返回正确的结果
     * @deprecated 使用 {@link ObjectResponse#success(Object)}
     * @param t
     * @param <T>
     * @return
     */
    @Deprecated
    public static <T> ObjectResponse<T> returnSuccessResponse(T t){
        if (t == null){
            return new ObjectResponse<T>(CodeConstants.SUCCESS, CodeConstants.getName(CodeConstants.SUCCESS));
        }else{
            return new ObjectResponse<T>(CodeConstants.SUCCESS, CodeConstants.getName(CodeConstants.SUCCESS), t);
        }
    }

    /**
     * 返回异常的结果
     * @deprecated 使用 {@link ObjectResponse#failed(String)}
     * @param errorCode
     * @return
     */
    @Deprecated
    public static <T> ObjectResponse<T> returnErrorResponse(String errorCode){
        return new ObjectResponse<>(errorCode, CodeConstants.getName(errorCode));
    }

    /**
     * 返回异常的结果
     * @deprecated 使用 {@link ObjectResponse#instance(String, String, Object)}
     * @param errorCode
     * @param errorMsg
     * @return
     */
    @Deprecated
    public static <T> ObjectResponse returnErrorResponse(String errorCode, String errorMsg, T t){
        return new ObjectResponse(errorCode, errorMsg, t);
    }
    /**
     * 返回异常的结果
     * @deprecated 使用 {@link ObjectResponse#failed(String, String)}
     * @param errorCode
     * @param errorMsg
     * @return
     */
    @Deprecated
    public static ObjectResponse returnErrorResponse(String errorCode, String errorMsg){
        return new ObjectResponse(errorCode, errorMsg);
    }

    /**
     * @deprecated 使用 {@link ObjectResponse#returnNotFoundIfNull(Object)}
     * @return
     */
    @Deprecated
    public static <T> ObjectResponse<T> returnNotFoundIfNull(T data) {
        if (data == null) {
            return returnErrorResponse(CodeConstants.ERROR_404);
        }

        return returnSuccessResponse(data);
    }

    /**
     * 返回结果不能失败，否则抛出异常
     * @deprecated 使用 {@link ObjectResponse#notError(Response)}
     * @param objectResponse
     */
    @Deprecated
    public static void notError(ObjectResponse objectResponse){
        if (objectResponse == null){
            throw new ResponseBodyException(CodeConstants.ERROR, CodeConstants.getName(CodeConstants.ERROR));
        }
        if (!objectResponse.getCode().equals(CodeConstants.SUCCESS)){
            throw new ResponseBodyException(objectResponse.getCode(), objectResponse.getMsg());
        }
    }

    public static void main(String[] args) {
        String json = "{\"code\":\"200\",\"msg\":\"success\",\"data\": {\"code\": \"201\",\"msg\":\"success1\",\"data\":100}}";
        System.err.println(getObjectResponse(json, ObjectResponse.class));
        json = "{\"code\":\"200\",\"msg\":\"success\",\"data\": [{\"code\": \"201\",\"msg\":\"success1\",\"data\":101},{\"code\": \"202\",\"msg\":\"success2\",\"data\":102}]}";
        System.err.println(getListObjectResponse(json, ObjectResponse.class));
    }
}
