package com.icetech.common.utils;


import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.CharsetUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ClassUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * Description:
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved.
 * @author wanggy
 */
@Slf4j
public class FileUtil extends cn.hutool.core.io.FileUtil {

    /**
     * Description: 读文件
     * Version1.0 2021-04-29 by wgy 创建
     *
     * @param filePath 文件路径
     * @return java.lang.String
     */
    public static String getReadFile(String filePath) throws IOException {
        InputStream stream = null;
        try {
            stream = ClassUtils.class.getClassLoader().getResourceAsStream(filePath);
            assert stream != null;
            BufferedReader reader = new BufferedReader(new InputStreamReader(stream, CharsetUtil.CHARSET_UTF_8));
            StringBuilder buffer = new StringBuilder();
            String text;
            while ((text = reader.readLine()) != null) {
                buffer.append(text);
            }
            return buffer.toString();
        } finally {
            if (stream != null) {
                stream.close();
            }
        }
    }

    /**
     * Description: 读文件
     * Version1.0 2021-04-29 by wgy 创建
     *
     * @param filePath 文件路径
     * @return java.lang.String
     */
    public static String getReadIoFile(String filePath) throws IOException {
        InputStream stream = null;
        try {
            stream = ClassUtils.class.getClassLoader().getResourceAsStream(filePath);
            if (stream == null) {
                log.error("文件不存在，路径：{}", filePath);
                return "";
            }
            return IoUtil.read(stream, CharsetUtil.CHARSET_UTF_8);
        } finally {
            if (stream != null) {
                stream.close();
            }
        }
    }

    /**
     * 下载文件名重新编码
     *
     * @param response     响应对象
     * @param realFileName 真实文件名
     * @return
     */
    public static void setAttachmentResponseHeader(HttpServletResponse response, String realFileName) throws UnsupportedEncodingException {
        String percentEncodedFileName = percentEncode(realFileName);

        StringBuilder contentDispositionValue = new StringBuilder();
        contentDispositionValue.append("attachment; filename=")
                .append(percentEncodedFileName)
                .append(";")
                .append("filename*=")
                .append("utf-8''")
                .append(percentEncodedFileName);
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition,download-filename");
        response.setHeader("Content-disposition", contentDispositionValue.toString());
        response.setHeader("download-filename", percentEncodedFileName);
    }

    /**
     * 百分号编码工具方法
     *
     * @param s 需要百分号编码的字符串
     * @return 百分号编码后的字符串
     */
    public static String percentEncode(String s) throws UnsupportedEncodingException {
        String encode = URLEncoder.encode(s, StandardCharsets.UTF_8.toString());
        return encode.replaceAll("\\+", "%20");
    }
}
