package com.icetech.city.road.utils;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.TimeZone;

import jakarta.annotation.Nullable;

public class DateUtils extends org.apache.commons.lang3.time.DateUtils {
    public static final String DEFAULT_DATE_PATTERN = "yyyy-MM-dd";
    public static final String SHORT_DATE_PATTERN = "yyyyMMdd";
    public static final String DEFAULT_TIME_PATTERN = "HH:mm:ss";
    public static final String SHORT_TIME_PATTERN = "HHmmss";
    public static final String DEFAULT_TIME_HM_PATTERN = "HH:mm";
    public static final String SHORT_TIME_HM_PATTERN = "HHmm";
    public static final String DEFAULT_DATETIME_PATTERN = DEFAULT_DATE_PATTERN + " " + DEFAULT_TIME_PATTERN;
    public static final String SHORT_DATETIME_PATTERN = SHORT_DATE_PATTERN + SHORT_TIME_PATTERN;
    public static final DateTimeFormatter DEFAULT_DATE_FORMAT = DateTimeFormatter.ofPattern(DEFAULT_DATE_PATTERN);
    public static final DateTimeFormatter DEFAULT_TIME_FORMAT = DateTimeFormatter.ofPattern(DEFAULT_TIME_PATTERN);
    public static final DateTimeFormatter DEFAULT_DATETIME_FORMAT = DateTimeFormatter.ofPattern(DEFAULT_DATETIME_PATTERN);
    public static final String DEFAULT_ZONE_VALUE = "+8";
    public static final ZoneId DEFAULT_ZONE_ID = ZoneId.of(DEFAULT_ZONE_VALUE);
    public static final Clock DEFAULT_CLOCK = Clock.system(DEFAULT_ZONE_ID);
    public static final ZoneOffset DEFAULT_ZONE_OFFSET = ZoneOffset.of(DEFAULT_ZONE_VALUE);
    public static final TimeZone DEFAULT_TIME_ZONE = TimeZone.getTimeZone(DEFAULT_ZONE_ID);
    public static final int MINUTE_OF_DAY = 24 * 60;
    public static final int SECOND_OF_DAY = MINUTE_OF_DAY * 60;
    public static final long MIN_MILLI_TIMESTAMP = 1000000000000L;

    public static long toEpochSecond(LocalDateTime localDateTime) {
        return localDateTime.toEpochSecond(DEFAULT_ZONE_OFFSET);
    }

    public static long toEpochMilli(LocalDateTime localDateTime) {
        return localDateTime.toInstant(DEFAULT_ZONE_OFFSET).toEpochMilli();
    }

    public static LocalDateTime toLocalDateTime(Date date) {
        return date.toInstant().atOffset(DEFAULT_ZONE_OFFSET).toLocalDateTime();
    }

    public static LocalDateTime toLocalDateTime(long milli) {
        return Instant.ofEpochMilli(milli).atZone(DEFAULT_ZONE_OFFSET).toLocalDateTime();
    }

    public static LocalDateTime toLocalDateTimeOfSecond(long second) {
        return LocalDateTime.ofEpochSecond(second, 0, DEFAULT_ZONE_OFFSET);
    }

    public static LocalDate toLocalDate(Date date) {
        return date.toInstant().atOffset(DEFAULT_ZONE_OFFSET).toLocalDate();
    }

    public static LocalTime toLocalTime(Date date) {
        return date.toInstant().atOffset(DEFAULT_ZONE_OFFSET).toLocalTime();
    }

    public static LocalTime toLocalTimeOfSecond(long second) {
        return Instant.ofEpochSecond(second).atZone(DEFAULT_ZONE_OFFSET).toLocalTime();
    }

    /**
     * 判定指定时间是否在时间段内<br />
     * 若结束时间大于开始时间, 则表示跨天
     *
     * @param time      时间
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return
     */
    public static boolean betweenOfSecond(LocalTime time, LocalTime startTime, LocalTime endTime) {
        int second = time.toSecondOfDay();
        int startSecond = startTime.toSecondOfDay();
        int endSecond = endTime.toSecondOfDay();
        return startSecond < endSecond ?
                second >= startSecond && second <= endSecond :
                (second >= startSecond && second <= SECOND_OF_DAY) || (second >= 0 && second <= endSecond);
    }

    public static boolean betweenOfMinute(LocalTime time, LocalTime startTime, LocalTime endTime) {
        int minute = time.toSecondOfDay() / 60;
        int startMinute = startTime.toSecondOfDay() / 60;
        int endMinute = endTime.toSecondOfDay() / 60;
        return startMinute < endMinute ?
                minute >= startMinute && minute <= endMinute :
                (minute >= startMinute && minute <= MINUTE_OF_DAY) || (minute >= 0 && minute <= endMinute);
    }

    public static String formatDurationOfNow(long unixTime) {
        return formatDurationOfSecond(unixTime, System.currentTimeMillis() / 1000);
    }

    public static String formatDurationOfSecond(long startUnixTime, long stopUnixTime) {
        return formatDurationOfSecond(stopUnixTime - startUnixTime);
    }

    public static String formatDurationOfSecond(long second) {
        String timeStr = "";
        long days = second / (60 * 60 * 24);//换成天
        long hours = (second - (60 * 60 * 24 * days)) / 3600;//总秒数-换算成天的秒数=剩余的秒数    剩余的秒数换算为小时
        /*分钟从0开始*/
        long minutes = (second - 60 * 60 * 24 * days - 3600 * hours) / 60;//总秒数-换算成天的秒数-换算成小时的秒数=剩余的秒数    剩余的秒数换算为分
        if (days > 0) {
            timeStr = String.format("%d天", days);
        }
        if (hours > 0) {
            timeStr += String.format("%d小时", hours);
        }
        timeStr += String.format("%d分钟", minutes);
        return timeStr;
    }

    public static String formatDurationOfSecond2(long second) {
        String timeStr = "";
        long days = second / (60 * 60 * 24);//换成天
        long hours = (second - (60 * 60 * 24 * days)) / 3600;//总秒数-换算成天的秒数=剩余的秒数    剩余的秒数换算为小时
        /*分钟从0开始*/
        long minutes = (second - 60 * 60 * 24 * days - 3600 * hours) / 60;//总秒数-换算成天的秒数-换算成小时的秒数=剩余的秒数    剩余的秒数换算为分
        long surplus = second - 60 * 60 * 24 * days - 60 * 60 * hours - 60 * minutes;
        if (days > 0) {
            timeStr = String.format("%d天", days);
        }
        if (hours > 0) {
            timeStr += String.format("%d小时", hours);
        }
        if (minutes > 0) {
            timeStr += String.format("%d分钟", minutes);
        }
        timeStr += String.format("%d秒", surplus);
        return timeStr;
    }

    public static String formatDurationOfMinute(long minute) {
        String timeStr = "";
        long days = minute / (60 * 24);//换成天
        long hours = (minute - (60 * 24 * days)) / 60;//总秒数-换算成天的秒数=剩余的秒数    剩余的秒数换算为小时
        /*分钟从0开始*/
        long minutes = minute - 60 * 24 * days - 60 * hours;//总秒数-换算成天的秒数-换算成小时的秒数=剩余的秒数    剩余的秒数换算为分
        if (days > 0) {
            timeStr = String.format("%d天", days);
        }
        if (hours > 0) {
            timeStr += String.format("%d小时", hours);
        }
        timeStr += String.format("%d分钟", minutes);
        return timeStr;
    }

    /**
     * 时间戳转换为 Date
     *
     * @param timestamp unix时间戳(毫秒/秒)
     * @return
     */
    @Nullable
    public static Date timestampToDate(@Nullable Long timestamp) {
        Date date = null;
        if (timestamp != null) {
            date = new Date(timestamp < MIN_MILLI_TIMESTAMP ? timestamp * 1000 : timestamp);
        }
        return date;
    }

    public static long currentTimeSecond() {
        return System.currentTimeMillis() / 1000;
    }

    public static int getMonthDays(int year, int month) {
        if (month == 1 || month == 3 || month == 5 || month == 7 || month == 8 || month == 10 || month == 12) return 31;
        if (month == 4 || month == 6 || month == 9 || month == 11) return 30;
        if (month == 2) return year % 4 == 0 ? 29 : 28;
        throw new IllegalArgumentException("Invalid month: " + month);
    }

    public static LocalDate atFirstDayOfMonth(LocalDate date) {
        return date.withDayOfMonth(1);
    }

    public static LocalDate atLastDayOfMonth(LocalDate date) {
        return date.withDayOfMonth(getMonthDays(date.getYear(), date.getMonthValue()));
    }

    /**
     * 获取n年前时间
     *
     * @param year int
     * @return date
     */
    public static Date getBeforeYear(int year) {
        return Date.from(LocalDateTime.now()
                .minusYears(year)
                .atZone(ZoneId.systemDefault())
                .toInstant());
    }

    /**
     * 获取日期的结束时间（23:59:59）
     */
    public static Date getEndOfDay(Date date) {
        LocalDateTime endOfDay = date.toInstant()
                .atZone(ZoneId.systemDefault())
                .toLocalDate()
                .atTime(LocalTime.MAX);
        return Date.from(endOfDay.atZone(ZoneId.systemDefault()).toInstant());
    }
    /**
     * 获取日期的开始时间（00:00:00）
     */
    public static Date getStartOfDay(Date date) {
        LocalDateTime startOfDay = date.toInstant()
                .atZone(ZoneId.systemDefault())
                .toLocalDate()
                .atStartOfDay();
        return Date.from(startOfDay.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 将 yyyyMMddHHmmss 格式的时间字符串转换为秒级时间戳
     * @param timeStr 时间字符串（如 "20210118130000"）
     * @return 秒级时间戳（Unix Timestamp）
     */
    public static long parseToTimestamp(String timeStr) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
        LocalDateTime localDateTime = LocalDateTime.parse(timeStr, formatter);
        return localDateTime.atZone(ZoneId.systemDefault()).toEpochSecond();
    }
}
