package com.icetech.city.road.component.bean;

import com.icetech.city.road.component.constant.CodeConstants;
import com.icetech.city.road.component.constant.CodeConstantsEnum;
import com.icetech.city.road.component.exception.ResponseBodyException;
import com.icetech.city.road.utils.Slf4jUtils;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;

import java.util.Collection;

@Getter
@Setter
@ToString(callSuper = true)
@NoArgsConstructor
@AllArgsConstructor
public class ObjectResponse<T> extends Response {
    protected T data;

    /**
     * traceId
     */
    protected String traceId = Slf4jUtils.getTraceId();


    public ObjectResponse(String code, String msg) {
        this(code, msg, null);
    }

    public ObjectResponse(String code, String msg, T data) {
        setCode(code);
        setMsg(msg);
        setData(data);
    }

    public static <T> ObjectResponse<T> instance(String code, T data) {
        return instance(code, CodeConstants.getName(code), data);
    }

    public static <T> ObjectResponse<T> instance(String code, String msg, T data) {
        return new ObjectResponse<>(code, msg, data);
    }

    /**
     * 从 {@link Response} 拷贝数据, 不含data
     *
     * @param response
     * @param <T>
     * @return
     */
    public static <T> ObjectResponse<T> instance(Response response) {
        return instance(response.getCode(), response.getMsg(), null);
    }

    public static <T> ObjectResponse<T> success() {
        return success(null);
    }

    public static <T> ObjectResponse<T> success(T data) {
        return instance(CodeConstants.SUCCESS, data);
    }

    public static <T> ObjectResponse<T> failed() {
        return failed(CodeConstantsEnum.ERROR);
    }

    public static <T> ObjectResponse<T> failed(ResponseBodyException exception) {
        return failed(exception.getErrCode(), exception.getMessage());
    }

    /**
     * @param response
     * @param <T>
     * @return
     * @deprecated 使用 {@link #instance(Response)}
     */
    @Deprecated
    public static <T> ObjectResponse<T> failed(Response response) {
        return failed(response.getCode(), response.getMsg());
    }

    public static <T> ObjectResponse<T> failed(CodeConstantsEnum code) {
        return failed(code.getCode(), code.getDesc());
    }

    public static <T> ObjectResponse<T> failed(String code) {
        return failed(code, CodeConstants.getName(code));
    }

    public static <T> ObjectResponse<T> failed(String code, String msg) {
        return new ObjectResponse<>(code, msg);
    }

    public static <T> ObjectResponse<T> returnNotFoundIfNull(T data) {
        if (data == null) {
            return failed(CodeConstantsEnum.ERROR_404);
        }

        return success(data);
    }

    public static <T extends Collection<?>> ObjectResponse<T> returnNotFoundIfEmpty(T data) {
        return data != null && !data.isEmpty() ? success(data) : failed(CodeConstantsEnum.ERROR_404);
    }

    public static ObjectResponse<Void> toAjax(int rows) {
        return rows > 0 ? ObjectResponse.success() : ObjectResponse.failed();
    }

    public static ObjectResponse<Void> toAjax(int rows, String code) {
        return rows > 0 ? ObjectResponse.success() : ObjectResponse.failed(code, "操作失败");
    }
}
