package com.icetech.third.service.third;

import cn.hutool.core.util.IdUtil;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.icetech.basics.domain.entity.device.HeartbeatOffline;
import com.icetech.cloudcenter.api.constants.MqConstants;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.third.ThirdParkService;
import com.icetech.cloudcenter.domain.request.p2c.HintRequest;
import com.icetech.cloudcenter.domain.request.p2c.LcdHintRequest;
import com.icetech.cloudcenter.domain.request.pnc.ShowAndSayRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateUtils;
import com.icetech.fee.domain.entity.monthcar.MonthInfo;
import com.icetech.fee.domain.entity.vip.VipInfo;
import com.icetech.order.domain.entity.BarrierGateException;
import com.icetech.order.domain.entity.OrderBack;
import com.icetech.order.domain.entity.OrderNotpay;
import com.icetech.park.domain.entity.ChannelAlarm;
import com.icetech.park.domain.entity.ShamPlate;
import com.icetech.park.domain.entity.park.ExitIdentify;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.third.anno.AsyncMethod;
import com.icetech.third.config.prop.DtnPushProp;
import com.icetech.third.dao.third.PushMessageDao;
import com.icetech.third.domain.entity.third.PushMessage;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.order.domain.entity.OrderSonInfo;
import com.icetech.basics.domain.entity.park.Opening;
import com.icetech.common.constants.RabbitConstant;
import com.icetech.common.domain.SendRequest;
import com.icetech.common.domain.mq.PushWrapper;
import com.icetech.common.utils.UUIDTools;
import com.icetech.mq.sender.RabbitSender;
import com.icetech.third.domain.enums.PushServiceEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageBuilder;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.connection.CorrelationData;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;

import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

@Slf4j
@Service
public class MqPushService {
    @Resource
    private PushMessageDao pushMessageDao;
    @Resource
    private ThirdParkService thirdParkService;
    @Resource
    private RabbitTemplate rabbitTemplate;
    @Autowired
    private RabbitSender rabbitSender;
    @Autowired
    private DtnPushProp dtnPushProp;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ObjectMapper objectMapper;
    public static final String ROUTING_KEY_EXIT_IDENTIFY = "order.exit.identify";

    /**
     * 离场识别推送
     */
    @AsyncMethod
    public void pushExitIdentify(ExitIdentify exitIdentify) {
        pushMessage(exitIdentify.getParkId(), PushMessage.MSG_TYPE_EXIT_IDENTIFY, UUIDTools.getUuid(), exitIdentify, ROUTING_KEY_EXIT_IDENTIFY, 1500);
    }

    /**
     * 入场推送
     */
    @AsyncMethod
    public void pushOrderEnter(OrderInfo orderInfo) {
        pushMessage(orderInfo.getParkId(), PushMessage.MSG_TYPE_ORDER_ENTER, String.valueOf(orderInfo.getId()), orderInfo, RabbitConstant.ROUTING_KEY_ORDER_ENTER, 3000);
        pushEventMsgToCommon(orderInfo, 1);
    }

    /**
     * 子订单入场推送
     */
    @AsyncMethod
    public void pushOrderSonEnter(OrderSonInfo orderSonInfo) {
        pushMessage(orderSonInfo.getParkId(), PushMessage.MSG_TYPE_ORDER_SON_ENTER, String.valueOf(orderSonInfo.getId()), orderSonInfo, RabbitConstant.ROUTING_KEY_ORDER_SON_ENTER, 3000);
    }

    /**
     * 离场推送
     */
    @AsyncMethod
    public void pushOrderExit(OrderInfo orderInfo) {
        pushOrderExit(orderInfo, 3000);
    }

    /**
     * 离场推送
     */
    @AsyncMethod
    public void pushOrderExit(OrderInfo orderInfo, Integer delay) {
        pushMessage(orderInfo.getParkId(), PushMessage.MSG_TYPE_ORDER_EXIT, String.valueOf(orderInfo.getId()), orderInfo, RabbitConstant.ROUTING_KEY_ORDER_EXIT, delay);
        pushEventMsgToCommon(orderInfo, 2);
    }

    /**
     * 离场推送
     */
    @AsyncMethod
    public void pushOrderSonExit(OrderSonInfo orderSonInfo) {
        pushMessage(orderSonInfo.getParkId(), PushMessage.MSG_TYPE_ORDER_SON_EXIT, String.valueOf(orderSonInfo.getId()), orderSonInfo, RabbitConstant.ROUTING_KEY_ORDER_SON_EXIT, 3000);
    }

    /**
     * 支付推送
     */
    @AsyncMethod
    public void pushOrderPay(OrderPay orderPay) {
        pushMessage(orderPay.getParkId(), PushMessage.MSG_TYPE_ORDER_PAY, String.valueOf(orderPay.getId()), orderPay, RabbitConstant.ROUTING_KEY_ORDER_PAY, 3000);
    }

    /**
     * 非延时支付推送
     */
    @AsyncMethod
    public void pushOrderPaySync(OrderPay orderPay) {
        rabbitSender.sendMessage(MqConstants.Exchange.NOTIFY_EXCHANGE, MqConstants.Routing.NOTIFY_DISCOUNT_ROUTING, orderPay);
    }

    @AsyncMethod
    public void pushDeviceStatus(ParkDevice parkDevice) {
        if (dtnPushProp.isPush(parkDevice.getParkId())
                || thirdParkService.checkPermissionService(parkDevice.getParkId(), PushServiceEnum.设备状态变化.getServiceName())) {
            pushMessage(parkDevice.getParkId(), PushMessage.MSG_TYPE_DEVICE_STATUS_UPDATE, parkDevice.getId().toString(), parkDevice, RabbitConstant.ROUTING_KEY_DEVICE_STATUS_UPDATE, null);
        }
        if (Integer.valueOf(2).equals(parkDevice.getStatus())) {
            pushMessageNotPersistent(parkDevice.getParkId(), PushMessage.MSG_TYPE_DEVICE_OFFLINE, parkDevice.getId().toString(), parkDevice, RabbitConstant.ROUTING_KEY_DEVICE_OFFLINE, null);
        }
    }

    public void pushMonthNearExpired(MonthInfo monthInfo, Integer delayMillis) {
        pushMessageNotPersistent(monthInfo.getParkId(), PushMessage.MSG_TYPE_MONTH_NEAR_EXPIRED, monthInfo.getId().toString(), monthInfo, RabbitConstant.ROUTING_KEY_MONTH_NEAR_EXPIRED, delayMillis);
    }

    public void pushVipNearExpired(VipInfo vipInfo, Integer delayMillis) {
        pushMessageNotPersistent(vipInfo.getParkId(), PushMessage.MSG_TYPE_VIP_NEAR_EXPIRED, vipInfo.getId().toString(), vipInfo, RabbitConstant.ROUTING_KEY_VIP_NEAR_EXPIRED, delayMillis);
    }

    public void pushPncCenterOffline(HeartbeatOffline offline) {
        pushMessageNotPersistent(offline.getParkId(), PushMessage.MSG_TYPE_PNC_CENTER_OFFLINE, offline.getId().toString(), offline, RabbitConstant.ROUTING_KEY_PNC_CENTER_OFFLINE, null);
    }

    public void pushPncChargeOffline(HeartbeatOffline offline) {
        pushMessageNotPersistent(offline.getParkId(), PushMessage.MSG_TYPE_PNC_CHARGE_OFFLINE, offline.getId().toString(), offline, RabbitConstant.ROUTING_KEY_PNC_CHARGE_OFFLINE, null);
    }

    public void pushZsBusyHint(Long parkId, String sn, String messageId, int delay) {
        Map<String, Object> param = new HashMap<>();
        param.put("parkId", parkId);
        param.put("sn", sn);
        param.put("messageId", messageId);
        pushMessageNotPersistent(parkId, PushMessage.MSG_TYPE_ZS_BUSY_HINT, sn, param,
                "zs.busy.hint.routing", delay);
    }

    @AsyncMethod
    public void pushErrorOpen(Opening opening) {
        if (thirdParkService.checkPermissionService(opening.getParkId(), PushServiceEnum.开关闸报警.getServiceName())) {
            pushMessage(opening.getParkId(), PushMessage.MSG_TYPE_ERROR_OPEN, String.valueOf(opening.getId()), opening, RabbitConstant.ROUTING_KEY_ERROR_OPEN, null);
        }
    }

    @AsyncMethod
    public void pushBatchDownSend(Long parkId, List<SendRequest> sendRequestList) {
        pushMessage(parkId, PushMessage.MSG_TYPE_BATCH_DOWN_SEND,
                String.valueOf(UUIDTools.getUuid()),
                sendRequestList, MqConstants.Routing.BATCH_DOWN_SEND_ROUTING, 3000);
    }

    /**
     * 车场空车位推送
     */
    @AsyncMethod
    public void pushParkFreeSpace(Long parkId) {
        pushMessageNotPersistent(parkId, PushMessage.MSG_TYPE_FREE_SPACE, String.valueOf(parkId), null,
                RabbitConstant.ROUTING_KEY_FREE_SPACE, 3000);
    }

    /**
     * 区域空车位推送
     */
    @AsyncMethod
    public void pushRegionFreeSpace(Long parkId, Long regionId) {
        pushMessageNotPersistent(parkId, PushMessage.MSG_TYPE_REGION_FREE_SPACE, String.valueOf(regionId), null,
                RabbitConstant.ROUTING_KEY_REGION_FREE_SPACE, 3000);
    }

    public void pushOrderNotPay(OrderNotpay notpay) {
        if (dtnPushProp.isPush(notpay.getParkId()))
            pushMessageNotPersistent(notpay.getParkId(), 20, String.valueOf(notpay.getId()), notpay, "order.not_pay", 3000);
    }

    public void pushShamPlate(ShamPlate shamPlate) {
        if (dtnPushProp.isPush(shamPlate.getParkId()))
            pushMessageNotPersistent(shamPlate.getParkId(), 21, String.valueOf(shamPlate.getId()), shamPlate, "order.sham_plate", 3000);
    }

    public void pushOrderBack(OrderBack orderBack) {
        if (dtnPushProp.isPush(orderBack.getParkId()))
            pushMessageNotPersistent(orderBack.getParkId(), 22, String.valueOf(orderBack.getId()), orderBack, "order.order_back", 3000);
    }

    public void pushBarrierGateException(BarrierGateException exception) {
        if (dtnPushProp.isPush(exception.getParkId()))
            pushMessageNotPersistent(exception.getParkId(), 23, String.valueOf(exception.getId()), exception, "gate.exception", 3000);
    }

    public void pushChannelAlarm(ChannelAlarm channelAlarm) {
        if (dtnPushProp.isPush(channelAlarm.getParkId()))
            pushMessageNotPersistent(channelAlarm.getParkId(), 24, String.valueOf(channelAlarm.getId()), channelAlarm, "channel.alarm", 3000);
    }

    public void pushOrderEnexShow(Long parkId, String channelCode, String deviceSn, String show) {
        if (dtnPushProp.isPush(parkId))
            pushMessageNotPersistent(parkId, 30, channelCode, new JSONObject() {{
                put("parkId", parkId);
                put("channelCode", channelCode);
                put("deviceSn", deviceSn);
                put("show", show);
            }}, "order.enex.show", 3000);
    }

    public void pushP2cLedScreen(Long parkId, String deviceSn, HintRequest request) {
        if (dtnPushProp.isPush(parkId))
            pushMessageNotPersistent(parkId, 31, deviceSn, request, "p2c.led_screen", 3000);
    }

    public void pushP2cLcdScreen(Long parkId, String deviceSn, LcdHintRequest request) {
        if (dtnPushProp.isPush(parkId))
            pushMessageNotPersistent(parkId, 32, deviceSn, request, "p2c.lcd_screen", 3000);
    }

    public void pushPncScreen(Long parkId, String channelCode, ShowAndSayRequest request) {
        if (dtnPushProp.isPush(parkId))
            pushMessageNotPersistent(parkId, 33, channelCode, request, "pnc.screen", 3000);
    }

    private void pushMessage(long parkId, int pushType, String dataId, Object content, String routingKey, Integer delay) {
        PushWrapper<Object> wrapper = new PushWrapper<>();
        wrapper.setPushType(routingKey);
        wrapper.setParkId(parkId);
        wrapper.setDataId(dataId);
        wrapper.setContent(content);
        String body = JSONObject.toJSONString(wrapper);

        String msgId = UUIDTools.getUuid();
        PushMessage record = new PushMessage();
        record.setMsgId(msgId);
        record.setParkId(parkId);
        record.setMsgType(pushType);
        record.setDataId(dataId);
        record.setContent(body);
        record.setCreateTime(new Date());
        pushMessageDao.insert(record);
        long recordId = record.getId();

        Message message = MessageBuilder
                .withBody(body.getBytes())
                .setMessageIdIfAbsent(msgId)
                .setContentType(MessageProperties.CONTENT_TYPE_JSON)
                .setHeader("count", 1)
                .setHeader("tag", routingKey)
                .build();
        if (delay != null) {
            message.getMessageProperties().setDelay(delay);
        }
        CorrelationData correlationData = null;
        if ((rabbitTemplate.isUsePublisherConnection() && rabbitTemplate.getConnectionFactory().getPublisherConnectionFactory() != null ?
                rabbitTemplate.getConnectionFactory().getPublisherConnectionFactory() : rabbitTemplate.getConnectionFactory()).isPublisherConfirms()) {
            correlationData = new CorrelationData();
            correlationData.getFuture().addCallback(new org.springframework.util.concurrent.ListenableFutureCallback<CorrelationData.Confirm>() {
                @Override
                public void onFailure(Throwable ex) {
                    PushMessage record = new PushMessage();
                    record.setId(recordId);
                    record.setPublishResult(-1);
                    record.setRemark(ex.getMessage() == null ? "发送程序出错" : ex.getMessage().substring(0, 255));
                    pushMessageDao.updateById(record);
                }

                @Override
                public void onSuccess(CorrelationData.Confirm result) {
                    PushMessage record = new PushMessage();
                    record.setId(recordId);
                    record.setPublishResult(result.isAck() ? 1 : -1);
                    record.setRemark(result.getReason());
                    pushMessageDao.updateById(record);
                }
            });
        }
        rabbitTemplate.send(RabbitConstant.EXCHANGE_ICECLOUD_EVENT, routingKey, message, correlationData);
        log.info("MQ推送完成[{}], parkId:{}, tag:{}, pushType:{}, dataId:{}", message.getMessageProperties().getMessageId(), parkId, routingKey, pushType, dataId);
    }

    /**
     * 非持久化
     */
    private void pushMessageNotPersistent(long parkId, int pushType, String dataId, Object content, String routingKey, Integer delay) {
        PushWrapper<Object> wrapper = new PushWrapper<>();
        wrapper.setPushType(routingKey);
        wrapper.setParkId(parkId);
        wrapper.setDataId(dataId);
        wrapper.setContent(content);
        String body = JSONObject.toJSONString(wrapper);

        String msgId = UUIDTools.getUuid();

        Message message = MessageBuilder
                .withBody(body.getBytes())
                .setMessageIdIfAbsent(msgId)
                .setContentType(MessageProperties.CONTENT_TYPE_JSON)
                .setHeader("count", 1)
                .setHeader("tag", routingKey)
                .build();
        if (delay != null) {
            message.getMessageProperties().setDelay(delay);
        }
        CorrelationData correlationData = null;
        if ((rabbitTemplate.isUsePublisherConnection() && rabbitTemplate.getConnectionFactory().getPublisherConnectionFactory() != null ?
                rabbitTemplate.getConnectionFactory().getPublisherConnectionFactory() : rabbitTemplate.getConnectionFactory()).isPublisherConfirms()) {
            correlationData = new CorrelationData();
            correlationData.getFuture().addCallback(new org.springframework.util.concurrent.ListenableFutureCallback<CorrelationData.Confirm>() {
                @Override
                public void onFailure(Throwable ex) {
                    log.warn("MQ推送失败[{}], parkId:{}, tag:{}, pushType:{}, dataId:{}", message.getMessageProperties().getMessageId(), parkId, routingKey, pushType, dataId);
                }

                @Override
                public void onSuccess(CorrelationData.Confirm result) {

                }
            });
        }
        rabbitTemplate.send(RabbitConstant.EXCHANGE_ICECLOUD_EVENT, routingKey, message, correlationData);
        log.info("MQ推送完成[{}], parkId:{}, tag:{}, pushType:{}, dataId:{}", message.getMessageProperties().getMessageId(), parkId, routingKey, pushType, dataId);
    }

    /**
     * 出入场推送到公共服务mq
     *
     * @param orderInfo 订单信息
     * @param eventType 1-入场 2-离场
     */
    private void pushEventMsgToCommon(OrderInfo orderInfo, int eventType) {
        try {
            ObjectResponse<Park> parkResp = parkService.findByParkId(orderInfo.getParkId());
            if (!ObjectResponse.isSuccess(parkResp)) {
                log.info("获取车场信息失败！：{}", parkResp);
                return;
            }
            Park park = parkResp.getData();
            if (Objects.isNull(park)) {
                log.info("未获取到车场信息");
                return;
            }
            Map<String, Object> body = new HashMap<>(8); // 主体
            Map<String, Object> content = new HashMap<>(8); // 业务内容
            // 填充业务内容
            content.put("carStatus", eventType);
            content.put("plateNum", orderInfo.getPlateNum());
            if (eventType == 1) {
                content.put("statusDate", DateUtils.toLocalDateTimeOfSecond(orderInfo.getEnterTime())
                        .format(DateUtils.DEFAULT_DATETIME_FORMAT));
            } else {
                content.put("statusDate", DateUtils.toLocalDateTimeOfSecond(orderInfo.getExitTime())
                        .format(DateUtils.DEFAULT_DATETIME_FORMAT));
            }
            content.put("parkCode", park.getParkCode());
            // 填充消息主体
            body.put("messageId", IdUtil.fastSimpleUUID()); // 消息唯一标识
            body.put("serviceName", "specialMessage"); // 服务名/消息类型标识
            body.put("timestamp", System.currentTimeMillis()); // 时间戳
            body.put("bizContent", content); // 嵌套业务内容
            sendMessage("common.server.event", "common.server.event", objectMapper.writeValueAsString(body));
        } catch (Exception e) {
            log.error("业务发生异常，消息推送Mq失败：", e);
        }
    }

    /**
     * 发送消息基础方法
     *
     * @param exchange    交换机名称
     * @param routingKey  路由键
     * @param message     消息内容
     * @param delayMillis 延迟时间(毫秒)，null表示不延迟
     */
    private void sendMessage(String exchange, String routingKey, String message, Long delayMillis) {
        // 参数校验
        Assert.hasText(exchange, "交换机名称不能为空");
        Assert.hasText(message, "消息内容不能为空");
        String messageId = UUID.randomUUID().toString();
        try {
            MessageProperties properties = new MessageProperties();
            properties.setMessageId(messageId);
            properties.setContentEncoding(StandardCharsets.UTF_8.name());
            // 设置延迟时间
            if (delayMillis != null && delayMillis > 0) {
                properties.setHeader("x-delay", delayMillis);
                log.debug("准备发送延迟消息，消息ID: {}, 延迟时间: {}ms", messageId, delayMillis);
            }
            Message msg = new Message(message.getBytes(StandardCharsets.UTF_8), properties);
            // 发送消息
            rabbitTemplate.convertAndSend(exchange, routingKey, msg);
            log.info("消息发送成功，消息ID: {}, 交换机: {}, 路由键: {}", messageId, exchange, routingKey);
        } catch (Exception e) {
            log.error("消息发送失败，交换机: {}, 路由键: {}, 错误原因: {}", exchange, routingKey, e.getMessage(), e);
        }
    }

    /**
     * 发送普通消息
     *
     * @param exchange   交换机名称
     * @param routingKey 路由键
     * @param message    消息内容
     */
    public void sendMessage(String exchange, String routingKey, String message) {
        log.debug("开始发送普通消息，交换机: {}, 路由键: {}", exchange, routingKey);
        sendMessage(exchange, routingKey, message, null);
    }

}
