package com.icetech.third.aop;

import java.util.concurrent.ThreadPoolExecutor;

import com.icetech.third.anno.AsyncMethod;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

@Slf4j
@Aspect
@Order(Integer.MIN_VALUE)
@Component
public class AsyncMethodAop {
    @Autowired
    private ThreadPoolExecutor asyncMethodExecutor;

    @Pointcut("@annotation(com.icetech.third.anno.AsyncMethod)")
    public void pointcut() {
    }

    @Around(value = "pointcut()")
    public Object async(ProceedingJoinPoint point) throws Throwable {
        Signature signature = point.getSignature();
        if (signature instanceof MethodSignature) {
            MethodSignature methodSignature = (MethodSignature) signature;
            AsyncMethod asyncMethod = methodSignature.getMethod().getAnnotation(AsyncMethod.class);
            if (asyncMethod != null && asyncMethod.async()) {
                log.info("异步执行[{}.{}]开始", point.getTarget().getClass().getName(), methodSignature.getMethod().getName());
                asyncMethodExecutor.execute(() -> {
                    try {
                        Object result = point.proceed(point.getArgs());
                        log.info("异步执行[{}.{}]成功: {}", point.getTarget().getClass().getName(), methodSignature.getMethod().getName(), result);
                    } catch (Throwable e) {
                        log.error("异步执行[{}.{}]出错", point.getTarget().getClass().getName(), methodSignature.getMethod().getName(), e);
                    }
                });
            } else {
                return point.proceed(point.getArgs());
            }
        } else {
            throw new UnsupportedOperationException();
        }

        return null;
    }
}
