package com.icetech.third.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundSetOperations;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;

import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Resource;

@Slf4j
@Component
public class RedisUtils {
    @Resource(name = "stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;
    @Resource
    private StringRedisTemplate stringRedisTemplate;
    @Value("${spring.redis.host}")
    private String redisHost;
    @Value("${spring.redis.port}")
    private Integer redisPort;
    @Value("${spring.redis.database}")
    private Integer redisDatabase;
    @Value("${spring.redis.password}")
    private String redisPassword;

    public StringRedisTemplate getStringRedisTemplate() {
        return stringRedisTemplate;
    }

    /**
     * 验证有效性
     *
     * @return
     */
    public boolean isValidity() {
        try {
            set("test", "test", 1L);
            return true;
        } catch (Exception e) {
            log.error("redis连接错误: {}", e.getMessage(), e);
            return false;
        }
    }

    /**
     * 批量删除对应的value
     *
     * @param keys
     */
    public void remove(final String... keys) {
        for (String key : keys) {
            remove(key);
        }
    }

    /**
     * 批量删除key
     *
     * @param pattern 模式
     */
    public void removePattern(final String pattern) {
        redisTemplate.delete(redisTemplate.keys(pattern));
    }

    /**
     * 获取所有匹配的key
     * @param pattern 模式
     * @return
     */
    public Set<String> keys(final String pattern) {
         return redisTemplate.keys(pattern);
    }

    /**
     * 删除对应的value
     *
     * @param key
     */
    public Boolean remove(String key) {
        return redisTemplate.delete(key);
    }

    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    public boolean exists(String key) {
        return Boolean.TRUE.equals(redisTemplate.hasKey(key));
    }

    /**
     * 读取缓存
     *
     * @param key
     * @return
     */
    public <T> T get(String key, Class<T> clazz) {
        String cache = redisTemplate.opsForValue().get(key);
        if (StringUtils.isBlank(cache)) {
            return null;
        }

        return JSON.parseObject(cache, clazz);
    }

    public <T> T get(String key, TypeReference<T> type) {
        String cache = redisTemplate.opsForValue().get(key);
        if (StringUtils.isBlank(cache)) {
            return null;
        }

        return JSON.parseObject(cache, type);
    }

    /**
     * 读取缓存
     *
     * @param key key
     * @return String
     */
    public String get(String key) {
        return redisTemplate.opsForValue().get(key);
    }

    /**
     * @param key     redis key
     * @param typeSub 泛型
     * @param <T>
     * @return
     */
    public <T> T get(String key, Type typeSub) {
        String cache = redisTemplate.opsForValue().get(key);
        if (StringUtils.isBlank(cache)) {
            return null;
        }
        return JSON.parseObject(cache, typeSub);
    }

    public <T> List<T> getListValue(String key, Class<T> clazz) {
        String cache = redisTemplate.opsForValue().get(key);
        if (StringUtils.isBlank(cache)) {
            return null;
        }

        return JSON.parseArray(cache, clazz);
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @return
     */
    public boolean set(String key, Object value) {
        String cache = JSON.toJSONString(value);
        redisTemplate.opsForValue().set(key, cache);
        return true;
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @param expireSecond 单位秒
     * @return
     */
    public boolean set(String key, Object value, long expireSecond) {
        String cache = JSON.toJSONString(value);
        redisTemplate.opsForValue().set(key, cache, expireSecond, TimeUnit.SECONDS);
        return true;
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @param expireSecond 单位秒
     * @return
     */
    public boolean set(String key, Object value, long expireSecond, TimeUnit timeUnit) {
        String cache = JSON.toJSONString(value);
        redisTemplate.opsForValue().set(key, cache, expireSecond, timeUnit);
        return true;
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @param milliSeconds 毫秒
     * @return
     */
    public boolean setExpireMilliSeconds(String key, Object value, Long milliSeconds) {
        String cache = JSON.toJSONString(value);
        redisTemplate.opsForValue().set(key, cache, milliSeconds, TimeUnit.MILLISECONDS);
        return true;
    }

    public boolean tryGetDistributedLock(String key, String requestId) {
        return tryGetDistributedLock(key, requestId, 3L);
    }

    /**
     * 尝试获取分布式锁
     *
     * @param key       锁
     * @param requestId 请求标识
     * @return 是否获取成功
     */
    public boolean tryGetDistributedLock(String key, String requestId, Long timeout) {
        String lockKey = "LOCK_" + key;
        for (int i = 0; i < 30; i++) {
            Boolean locked = redisTemplate.opsForValue().setIfAbsent(lockKey, requestId, timeout, TimeUnit.SECONDS);
            if (locked != null && locked) {
                log.info("第{}次成功获取到了锁{}", i + 1, lockKey);
                return true;
            }
            // 尝试等待3秒来获取锁
            try {
                Thread.sleep(100L);
            } catch (InterruptedException e) {
                log.warn("Redis锁获取等待失败", e);
            }
        }

        log.info("未成功获取到锁{}", lockKey);
        return false;
    }

    /**
     * 尝试获取分布式锁
     *
     * @param key      锁
     * @param value    value
     * @param timeout  锁的时长
     * @param waitTime 等待获取锁的时间，单位毫秒
     * @return 是否获取成功
     */
    public boolean tryGetDistributedLock(String key, String value, Long timeout, Long waitTime) {
        String lockKey = "lock:" + key;
        long now = System.currentTimeMillis();
        while (now <= now + waitTime) {
            Boolean locked = redisTemplate.opsForValue().setIfAbsent(lockKey, value, timeout, TimeUnit.SECONDS);
            if (locked != null && locked) {
                return true;
            }
            // 尝试等待来获取锁
            try {
                Thread.sleep(100L);
            } catch (InterruptedException e) {
                log.warn("Redis锁获取等待失败", e);
            }
            now = System.currentTimeMillis();
        }
        log.info("未成功获取到锁{}", lockKey);
        return false;
    }

    /**
     * 尝试获取分布式锁，并加锁
     *
     * @param key   锁
     * @param value 请求标识
     * @return 是否获取成功
     */
    public boolean tryLock(String key, String value, long timeout) {
        String lockKey = "lock:" + key;
        try {
            Boolean result = stringRedisTemplate.opsForValue().setIfAbsent(lockKey, value, timeout, TimeUnit.MILLISECONDS);
            log.info("[Redis锁]获取锁[{}]结果: {}", lockKey, result);
            return result != null && result;
        } catch (Exception e) {
            log.info("[Redis锁获取失败]获取锁[{}]结果:", lockKey, e);
            return false;
        }

    }

    /**
     * 尝试获取分布式锁
     *
     * @param key 锁
     * @return 是否获取成功
     */
    public String getLockValue(String key) {
        String lockKey = "lock:" + key;
        return stringRedisTemplate.opsForValue().get(lockKey);
    }

    /**
     * 更新分布式锁值
     *
     * @param key   锁
     * @param value 请求标识
     * @return 是否获取成功
     */
    public boolean updateLockValue(String key, String value) {
        String lockKey = "lock:" + key;
        Boolean result = stringRedisTemplate.opsForValue().setIfPresent(lockKey, value);
        log.info("[Redis锁]更新锁[{}]存储值为[{}]: {}", lockKey, value, result);
        return result != null && result;
    }

    /**
     * 释放分布式锁
     *
     * @param key 锁
     */
    public void releaseLock(String key) {
        String lockKey = "lock:" + key;
        stringRedisTemplate.delete(lockKey);
    }

    /**
     * 释放分布式锁
     *
     * @param key       锁
     * @param requestId 请求标识
     * @return 是否释放成功
     */
    public boolean releaseDistributedLock(String key, String requestId) {
        String lockKey = "LOCK_" + key;
        String val = stringRedisTemplate.opsForValue().get(lockKey);
        if (val != null && val.equals(requestId)) {
            log.info("成功释放了锁 {}，requestId：{}", lockKey, requestId);
            remove(lockKey);
            return true;
        } else {
            log.error("重要提示：未成功释放锁 {}，requestId：{}", lockKey, requestId);
        }
        return false;
    }
    /** -------------------hash相关操作------------------------- */

    /**
     * 获取存储在哈希表中指定字段的值
     *
     * @param key
     * @param field
     * @return
     */
    public <T> T hGet(String key, String field, Class<T> clazz) {
        Object cacheObj = redisTemplate.opsForHash().get(key, field);
        if (cacheObj == null) {
            return null;
        }
        return JSON.parseObject(cacheObj.toString(), clazz);
    }

    /**
     * 获取所有给定字段的值
     *
     * @param key
     * @return
     */
    public <T> Map<String, T> hGetAll(String key, Class<T> clazz) {
        Map<Object, Object> cacheMap = redisTemplate.opsForHash().entries(key);
        if (MapUtils.isEmpty(cacheMap)) {
            return Collections.emptyMap();
        }
        try {
            return cacheMap.entrySet().stream().collect(Collectors.toMap(
                    entry -> entry.getKey().toString(),
                    entry -> entry.getValue() == null ? null : JSON.parseObject(entry.getValue().toString(), clazz),
                    (old, newer) -> newer, LinkedHashMap::new));
        } catch (Exception e) {
            log.warn("Redis Map反序列化失败|{}|{}", key, clazz.getName(), e);
            return Collections.emptyMap();
        }
    }

    /**
     * 获取所有给定字段的值
     *
     * @param key
     * @param clazz
     * @param fields
     * @return
     */
    public <T> List<T> hMultiGet(String key, Class<T> clazz, String... fields) {
        List<Object> cacheList = redisTemplate.opsForHash().multiGet(key, Arrays.asList(fields));
        if (CollectionUtils.isEmpty(cacheList)) {
            return Collections.emptyList();
        }
        return cacheList.stream().map(value -> JSON.parseObject(value.toString(), clazz)).collect(Collectors.toList());
    }

    public void hPut(String key, String hashKey, Object value) {
        redisTemplate.opsForHash().put(key, hashKey, JSON.toJSONString(value));
    }

    public void hPut(String key, String hashKey, Object value, long timeout) {
        redisTemplate.opsForHash().put(key, hashKey, JSON.toJSONString(value));
        redisTemplate.expire(key, timeout, TimeUnit.SECONDS);
    }

    public void hMultiPut(String key, Map<String, Object> fieldValueMap) {
        Jedis jedisIns = JedisUtils.getJedisIns(redisHost, redisPort, redisPassword, redisDatabase);
        Map<String, String> cacheMap = fieldValueMap.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> JSON.toJSONString(entry.getValue())));
        jedisIns.hmset(key, cacheMap);
    }

    public void hPutAll(String key, Map<String, Object> maps) {
        Map<String, String> cacheMap = maps.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> JSON.toJSONString(entry.getValue())));
        redisTemplate.opsForHash().putAll(key, cacheMap);
    }

    /**
     * 仅当hashKey不存在时才设置
     *
     * @param key
     * @param hashKey
     * @param value
     * @return
     */
    public Boolean hPutIfAbsent(String key, String hashKey, Object value) {
        return redisTemplate.opsForHash().putIfAbsent(key, hashKey, JSON.toJSONString(value));
    }

    /**
     * 删除一个或多个哈希表字段
     *
     * @param key
     * @param fields
     * @return
     */
    public Long hDelete(String key, String... fields) {
        return redisTemplate.opsForHash().delete(key, (Object[]) fields);
    }

    /**
     * 查看哈希表 key 中，指定的字段是否存在
     *
     * @param key
     * @param field
     * @return
     */
    public boolean hExists(String key, String field) {
        return redisTemplate.opsForHash().hasKey(key, field);
    }

    /**
     * 为哈希表 key 中的指定字段的整数值加上增量 increment
     *
     * @param key
     * @param field
     * @param increment
     * @return
     */
    public Long hIncrBy(String key, String field, long increment) {
        return redisTemplate.opsForHash().increment(key, field, increment);
    }

    /**
     * 为哈希表 key 中的指定字段的整数值加上增量 increment
     *
     * @param key
     * @param field
     * @param delta
     * @return
     */
    public Double hIncrByFloat(String key, String field, double delta) {
        return redisTemplate.opsForHash().increment(key, field, delta);
    }

    /**
     * 获取所有哈希表中的字段
     *
     * @param key
     * @return
     */
    public Set<String> hKeys(String key) {
        Set<Object> list = redisTemplate.opsForHash().keys(key);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptySet();
        }
        return list.stream().map(Object::toString).collect(Collectors.toSet());
    }

    /**
     * 获取哈希表中字段的数量
     *
     * @param key
     * @return
     */
    public Long hSize(String key) {
        return redisTemplate.opsForHash().size(key);
    }

    /**
     * 获取哈希表中所有值
     *
     * @param key
     * @return
     */
    public <T> List<T> hValues(String key, Class<T> clazz) {
        List<Object> cacheValues = redisTemplate.opsForHash().values(key);
        if (CollectionUtils.isEmpty(cacheValues)) {
            return Collections.emptyList();
        }
        return cacheValues.stream().map(value -> JSON.parseObject(value.toString(), clazz)).collect(Collectors.toList());
    }

    /**
     * 迭代哈希表中的键值对
     *
     * @param key
     * @param options
     * @return
     */
    public Cursor<Map.Entry<Object, Object>> hScan(String key, ScanOptions options) {
        // TODO 序列化反序列化
        return redisTemplate.opsForHash().scan(key, options);
    }

    /** ------------------------list相关操作---------------------------- */

    /**
     * 通过索引获取列表中的元素
     *
     * @param key
     * @param index
     * @return
     */
    public <T> T lIndex(String key, long index, Class<T> clazz) {
        String cacheObj = redisTemplate.opsForList().index(key, index);
        if (StringUtils.isBlank(cacheObj)) {
            return null;
        }
        return JSON.parseObject(cacheObj, clazz);
    }

    /**
     * 获取列表指定范围内的元素
     *
     * @param key
     * @param start 开始位置, 0是开始位置
     * @param end   结束位置, -1返回所有
     * @return
     */
    public <T> List<T> lRange(String key, long start, long end, Class<T> clazz) {
        List<String> cacheList = redisTemplate.opsForList().range(key, start, end);
        if (CollectionUtils.isEmpty(cacheList)) {
            return Collections.emptyList();
        }
        return cacheList.stream().map(value -> JSON.parseObject(value, clazz)).collect(Collectors.toList());
    }

    /**
     * 存储在list头部
     *
     * @param key
     * @param value
     * @return
     */
    public Long lLeftPush(String key, Object value) {
        return redisTemplate.opsForList().leftPush(key, JSON.toJSONString(value));
    }

    /**
     * @param key
     * @param value
     * @return
     */
    public Long lLeftPushAll(String key, Object... value) {
        List<String> values = Stream.of(value).map(JSON::toJSONString).collect(Collectors.toList());
        return redisTemplate.opsForList().leftPushAll(key, values);
    }

    /**
     * @param key
     * @param value
     * @return
     */
    public Long lLeftPushAll(String key, Collection<Object> value) {
        List<String> values = value.stream().map(JSON::toJSONString).collect(Collectors.toList());
        return redisTemplate.opsForList().leftPushAll(key, values);
    }

    /**
     * 当list存在的时候才加入
     *
     * @param key
     * @param value
     * @return
     */
    public Long lLeftPushIfPresent(String key, Object value) {
        return redisTemplate.opsForList().leftPushIfPresent(key, JSON.toJSONString(value));
    }

    /**
     * 如果pivot存在,再pivot前面添加
     *
     * @param key
     * @param pivot
     * @param value
     * @return
     */
    public Long lLeftPush(String key, String pivot, Object value) {
        return redisTemplate.opsForList().leftPush(key, pivot, JSON.toJSONString(value));
    }

    /**
     * @param key
     * @param value
     * @return
     */
    public Long lRightPush(String key, Object value) {
        return redisTemplate.opsForList().rightPush(key, JSON.toJSONString(value));
    }

    /**
     * @param key
     * @param value
     * @return
     */
    public Long lRightPushAll(String key, Object... value) {
        List<String> values = Stream.of(value).map(JSON::toJSONString).collect(Collectors.toList());
        return redisTemplate.opsForList().rightPushAll(key, values);
    }

    /**
     * @param key
     * @param value
     * @return
     */
    public Long lRightPushAll(String key, Collection<Object> value) {
        List<String> values = value.stream().map(JSON::toJSONString).collect(Collectors.toList());
        return redisTemplate.opsForList().rightPushAll(key, values);
    }

    /**
     * 为已存在的列表添加值
     *
     * @param key
     * @param value
     * @return
     */
    public Long lRightPushIfPresent(String key, Object value) {
        return redisTemplate.opsForList().rightPushIfPresent(key, JSON.toJSONString(value));
    }

    /**
     * 在pivot元素的右边添加值
     *
     * @param key
     * @param pivot
     * @param value
     * @return
     */
    public Long lRightPush(String key, String pivot, Object value) {
        return redisTemplate.opsForList().rightPush(key, pivot, JSON.toJSONString(value));
    }

    /**
     * 通过索引设置列表元素的值
     *
     * @param key
     * @param index 位置
     * @param value
     */
    public void lSet(String key, long index, Object value) {
        redisTemplate.opsForList().set(key, index, JSON.toJSONString(value));
    }

    /**
     * 移出并获取列表的第一个元素
     *
     * @param key
     * @return 删除的元素
     */
    public <T> T lLeftPop(String key, Class<T> clazz) {
        String cache = redisTemplate.opsForList().leftPop(key);
        if (cache == null) {
            return null;
        }
        return JSON.parseObject(cache, clazz);
    }

    /**
     * 移出并获取列表的第一个元素， 如果列表没有元素会阻塞列表直到等待超时或发现可弹出元素为止
     *
     * @param key
     * @param timeout 等待时间
     * @param unit    时间单位
     * @return
     */
    public <T> T lBLeftPop(String key, long timeout, TimeUnit unit, Class<T> clazz) {
        String cache = redisTemplate.opsForList().leftPop(key, timeout, unit);
        if (cache == null) {
            return null;
        }
        return JSON.parseObject(cache, clazz);
    }

    /**
     * 移除并获取列表最后一个元素
     *
     * @param key
     * @return 删除的元素
     */
    public <T> T lRightPop(String key, Class<T> clazz) {
        String cache = redisTemplate.opsForList().rightPop(key);
        if (cache == null) {
            return null;
        }
        return JSON.parseObject(cache, clazz);
    }

    /**
     * 移出并获取列表的最后一个元素， 如果列表没有元素会阻塞列表直到等待超时或发现可弹出元素为止
     *
     * @param key
     * @param timeout 等待时间
     * @param unit    时间单位
     * @return
     */
    public <T> T lBRightPop(String key, long timeout, TimeUnit unit, Class<T> clazz) {
        String cache = redisTemplate.opsForList().rightPop(key, timeout, unit);
        if (cache == null) {
            return null;
        }
        return JSON.parseObject(cache, clazz);
    }

    /**
     * 移除列表的最后一个元素，并将该元素添加到另一个列表并返回
     *
     * @param sourceKey
     * @param destinationKey
     * @return
     */
    public <T> T lRightPopAndLeftPush(String sourceKey, String destinationKey, Class<T> clazz) {
        String cache = redisTemplate.opsForList().rightPopAndLeftPush(sourceKey, destinationKey);
        if (cache == null) {
            return null;
        }
        return JSON.parseObject(cache, clazz);
    }

    /**
     * 从列表中弹出一个值，将弹出的元素插入到另外一个列表中并返回它； 如果列表没有元素会阻塞列表直到等待超时或发现可弹出元素为止
     *
     * @param sourceKey
     * @param destinationKey
     * @param timeout
     * @param unit
     * @return
     */
    public <T> T lBRightPopAndLeftPush(String sourceKey, String destinationKey,
                                       long timeout, TimeUnit unit, Class<T> clazz) {
        String cache = redisTemplate.opsForList().rightPopAndLeftPush(sourceKey,
                destinationKey, timeout, unit);
        if (cache == null) {
            return null;
        }
        return JSON.parseObject(cache, clazz);
    }

    /**
     * 删除集合中值等于value得元素
     *
     * @param key
     * @param index index=0, 删除所有值等于value的元素; index>0, 从头部开始删除第一个值等于value的元素;
     *              index<0, 从尾部开始删除第一个值等于value的元素;
     * @param value
     * @return
     */
    public Long lRemove(String key, long index, Object value) {
        return redisTemplate.opsForList().remove(key, index, JSON.toJSONString(value));
    }

    public Long lRemoveOriginVal(String key, long index, Object value) {
        return redisTemplate.opsForList().remove(key, index, value.toString());
    }

    /**
     * 裁剪list
     *
     * @param key
     * @param start
     * @param end
     */
    public void lTrim(String key, long start, long end) {
        redisTemplate.opsForList().trim(key, start, end);
    }

    /**
     * 获取列表长度
     *
     * @param key
     * @return
     */
    public Long lLen(String key) {
        return redisTemplate.opsForList().size(key);
    }

    /**
     * 设置过期时间
     *
     * @param key
     * @param expireTime 过期时间(秒)
     */
    public void expire(String key, long expireTime) {
        redisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
    }

    /**
     * redis消息发布
     *
     * @param channel
     * @param content
     */
    public void convertAndSend(String channel, String content) {
        redisTemplate.convertAndSend(channel, content);
    }

    /**
     * 写入缓存
     *
     * @param key
     * @param value
     * @return
     */
    public boolean setIfAbsent(String key, Object value, long timeout) {
        String cache = JSON.toJSONString(value);
        Boolean aBoolean = redisTemplate.opsForValue().setIfAbsent(key, cache, timeout, TimeUnit.SECONDS);
        return aBoolean != null && aBoolean;
    }

    public boolean setNoProfile(String key, Object value, long expireSecond) {
        String cache = JSON.toJSONString(value);
        redisTemplate.opsForValue().set(key, cache, expireSecond, TimeUnit.SECONDS);
        return true;
    }

}
