package com.icetech.park.service.work.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.google.common.collect.Lists;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.common.domain.Page;
import com.icetech.common.utils.StringUtils;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.park.dao.work.WorkOrderMapper;
import com.icetech.park.domain.dto.QueryWorkOrderDTO;
import com.icetech.park.domain.entity.WorkOrder;
import com.icetech.park.domain.enums.OrderStatus;
import com.icetech.park.domain.vo.WorkOrderCountVo;
import com.icetech.park.domain.vo.WorkOrderPageVo;
import com.icetech.park.service.work.WorkOrderService;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 工单表 服务实现类
 *
 * Copyright (c) Department of Research and Development/Beijing
 * All Rights Reserved
 *
 * @author wanggang
 * @version 1.0 @date 2023-11-24
 */
@Service
@RequiredArgsConstructor
public class WorkOrderServiceImpl extends BaseServiceImpl<WorkOrderMapper, WorkOrder> implements WorkOrderService {

    private final ParkService parkService;
    
    /**
     * Description: 根据ID查询 工单表对象信息 <br>
     * Version1.0 2023-11-24 by wanggang创建
     *
     * @param id 对象id
     * @return WorkOrder
     */
    @Override
    public WorkOrder getWorkOrderById (Long id) {
        return getById(id);
    }

    /**
     * Description: 新增  工单表对象信息 <br>
     * Version1.0 2023-11-24 by wanggang创建
     *
     * @param entity 对象信息
     * @return Boolean 返回新增后主键
     */
    @Override
    public Boolean addWorkOrder (WorkOrder entity) {
        return save(entity);
    }

    /**
     * Description: 修改 工单表对象信息 <br>
     * Version1.0 2023-11-24 by wanggang创建
     *
     * @param entity 对象信息
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean modifyWorkOrder (WorkOrder entity) {
        return updateById(entity);
    }

    /**
     * Description: 删除  工单表对象信息 <br>
     * Version1.0 2023-11-24 by wanggang创建
     *
     * @param id 对象id
     * @return Boolean 返回更新后的结果
     */
    @Override
    public Boolean removeWorkOrderById (Long id) {
        return removeById(id);
    }

    @Override
    public Page<WorkOrderPageVo> page(QueryWorkOrderDTO dto, Long userId, Integer isCs) {
        List<Long> parkList = parkService.getParkListByUserId(userId.intValue());
        if (CollectionUtils.isEmpty(parkList)) {
            return Page.instance(0, 0, Lists.newArrayList());
        }
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<WorkOrder> page =
                page(new com.baomidou.mybatisplus.extension.plugins.pagination.Page<>(dto.getPageNo(), dto.getPageSize()), getQueryWrapper(dto, userId, isCs, parkList));
        if (Objects.nonNull(page)) {
            List<WorkOrderPageVo> list = page.getRecords().stream().map(t-> {
                WorkOrderPageVo workOrderPageVo = new WorkOrderPageVo();
                BeanUtils.copyProperties(t, workOrderPageVo);
                if (Objects.nonNull(t.getParkId())) {
                    workOrderPageVo.setParkName(parkService.findByParkId(t.getParkId()).getData().getParkName());
                }
                return workOrderPageVo;
            }).collect(Collectors.toList());
            return new Page<>(page.getTotal(), (int) page.getPages(), list);
        }
        return null;
    }

    @Override
    public List<WorkOrderCountVo> orderCount(QueryWorkOrderDTO dto, Long userId, Integer isCs) {
        List<Long> parkList = parkService.getParkListByUserId(userId.intValue());
        if (CollectionUtils.isEmpty(parkList)) {
            return Lists.newArrayList();
        }
        List<WorkOrderCountVo> list = Lists.newArrayList();
        for (OrderStatus status : OrderStatus.values()) {
            if (status == OrderStatus.RETURNED) {
                continue;
            }
            dto.setStatus(status.getValue());
            LambdaQueryWrapper<WorkOrder> queryWrapper = getQueryWrapper(dto, userId, isCs, parkService.getParkListByUserId(userId.intValue()));
            list.add(WorkOrderCountVo.builder().orderStatus(status.getValue()).orderCount(count(queryWrapper)).build());
        }
        dto.setStatus(null);
        LambdaQueryWrapper<WorkOrder> queryWrapper = getQueryWrapper(dto, userId, isCs, parkService.getParkListByUserId(userId.intValue()));
        list.add(WorkOrderCountVo.builder().orderStatus(-1).orderCount(count(queryWrapper)).build());
        return list;
    }
    
    private LambdaQueryWrapper<WorkOrder> getQueryWrapper(QueryWorkOrderDTO dto, Long userId, Integer isCs, List<Long> parkList) {
        LambdaQueryWrapper<WorkOrder> queryWrapper = new LambdaQueryWrapper<>();
        if (!parkList.contains(-1L)) {
            queryWrapper.in(WorkOrder::getParkId, parkList);
        }
        // 待处理
        if (Objects.nonNull(dto.getStatus()) && dto.getStatus() == OrderStatus.PENDING.getValue()) {
            if (StringUtils.getInt(isCs) == 0) {
                queryWrapper.in(WorkOrder::getStatus, Lists.newArrayList(OrderStatus.PENDING.getValue(), OrderStatus.RETURNED.getValue()));
            }
            else {
                queryWrapper.eq(Objects.nonNull(dto.getStatus()), WorkOrder::getStatus, dto.getStatus());
            }
        }
        // 已完成
        else if (Objects.nonNull(dto.getStatus()) && dto.getStatus() == OrderStatus.COMPLETED.getValue() ) {
            if (StringUtils.getInt(isCs) == 1) {
                queryWrapper.in(WorkOrder::getStatus, Lists.newArrayList(OrderStatus.COMPLETED.getValue(), OrderStatus.RETURNED.getValue()));
            }
            else {
                queryWrapper.eq(Objects.nonNull(dto.getStatus()), WorkOrder::getStatus, dto.getStatus());
            }
        }
        else {
            queryWrapper.eq(Objects.nonNull(dto.getStatus()), WorkOrder::getStatus, dto.getStatus());
        }
        queryWrapper.eq(Objects.nonNull(dto.getParkId()), WorkOrder::getParkId, dto.getParkId());
        queryWrapper.like(StringUtils.isNotEmpty(dto.getCode()), WorkOrder::getCode, dto.getCode());
        if (Objects.nonNull(dto.getView())) {
            if (dto.getView() == 1) {
                queryWrapper.eq(WorkOrder::getUserId, userId);
            } else {
                queryWrapper.eq(WorkOrder::getCsUserId, userId);
            }
        }
        queryWrapper.between(StringUtils.isNotEmpty(dto.getStartTime()) && StringUtils.isNotEmpty(dto.getEndTime()), WorkOrder::getCreateTime, dto.getStartTime(),
                dto.getEndTime());
        if (CollectionUtils.isNotEmpty(dto.getPriorityTypes())) {
            queryWrapper.in(WorkOrder::getPriority, dto.getPriorityTypes().stream().map(QueryWorkOrderDTO.PriorityType::getPriority).collect(Collectors.toList()));
            // 将dto.getPriorityTypes() 中的所有 type 转换成一个 list
            List<Integer> allTypes = dto.getPriorityTypes().stream()
                    .map(QueryWorkOrderDTO.PriorityType::getTypes)
                    .filter(CollectionUtils::isNotEmpty)
                    .flatMap(List::stream)
                    .collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(allTypes)) {
                queryWrapper.in(WorkOrder::getType, allTypes);
            }
        }
        queryWrapper.orderByDesc(WorkOrder::getId);
        return queryWrapper;
    }
}
