package com.icetech.park.service.sms;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import com.github.pagehelper.Page;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.PageResult;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.DataCollectionEnum;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.db.mybatis.base.service.impl.BaseServiceImpl;
import com.icetech.db.utils.PageHelperUtils;
import com.icetech.park.dao.sms.SmsAlarmConfigDao;
import com.icetech.park.domain.constant.sms.SmsAlarmOperRecordConsts;
import com.icetech.park.domain.dto.sms.SmsAlarmConfigDto;
import com.icetech.park.domain.entity.sms.SmsAlarmConfig;
import com.icetech.park.domain.entity.sms.SmsAlarmOperRecord;
import com.icetech.park.domain.entity.sms.SmsAlarmOrder;
import com.icetech.park.domain.request.sms.SmsAlarmConfigParam;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Slf4j
@Service
public class SmsAlarmConfigService extends BaseServiceImpl<SmsAlarmConfigDao, SmsAlarmConfig> {
    public static final String LOCK_KEY = "lock:sms_alarm:park:";
    @Resource
    private SmsAlarmTemplateService smsAlarmTemplateService;
    @Resource
    private SmsAlarmOperRecordService smsAlarmOperRecordService;
    @Resource
    private ParkService parkService;
    @Resource
    private RedissonClient redissonClient;

    public PageResult<SmsAlarmConfigDto> pageDto(SmsAlarmConfigParam param) {
        Page<SmsAlarmConfigDto> page = PageHelperUtils.page(param.getPageIndex(), param.getPageSize(),
                () -> getBaseMapper().getDtoList(param));
        return PageResult.fromPages(page);
    }

    public SmsAlarmConfig getByParkId(Long parkId) {
        return lambdaQuery().eq(SmsAlarmConfig::getParkId, parkId).one();
    }

    public SmsAlarmConfigDto getDtoByParkId(Long parkId) {
        return getBaseMapper().getDtoByParkId(parkId);
    }

    public boolean saveOrUpdate(SmsAlarmOrder order) {
        String lockKey = LOCK_KEY + order.getParkId();
        RLock lock = redissonClient.getLock(lockKey);
        try {
            lock.tryLock(30, 30, TimeUnit.SECONDS);
            SmsAlarmConfig exists = getByParkId(order.getParkId());
            if (exists == null) {
                SmsAlarmConfig config = new SmsAlarmConfig()
                        .setParkId(order.getParkId())
                        .setTotalCount(order.getSmsCount())
                        .setCurrCount(order.getSmsCount());
                boolean result = save(config);
                smsAlarmTemplateService.initTemplate(order);
                return result;
            } else {
                return changeCount(exists.getParkId(), order.getSmsCount());
            }
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        } finally {
            lock.unlock();
        }
    }

    @Transactional
    public boolean updateEnableFlagByParkId(SmsAlarmConfigParam config) {
        SmsAlarmConfig old = getByParkId(config.getParkId());
        if (old == null) throw new ResponseBodyException(CodeConstants.ERROR_404, "车场配置不存在");
        if (config.getDeviceEnableFlag() != null) { // 修改设备开关时确认端云/端网云以修改不同字段
            ObjectResponse<ParkConfig> parkConfigResp = parkService.getParkConfig(config.getParkId());
            ObjectResponse.notError(parkConfigResp);
            ParkConfig parkConfig = parkConfigResp.getData();
            if (DataCollectionEnum.端网云.getType().equals(parkConfig.getDataCollection())) {
                // 如果为端网云则表示为端网云相关提醒开关
                config.setPncCenterEnableFlag(config.getDeviceEnableFlag())
                        .setPncChargeEnableFlag(config.getDeviceEnableFlag());
            } else {
                // 清除端网云开关配置
                config.setPncCenterEnableFlag(null)
                        .setPncChargeEnableFlag(null);
            }
        }
        List<SmsAlarmOperRecord> records = new ArrayList<>(5);
        SmsAlarmConfig updated = new SmsAlarmConfig().setId(old.getId());
        if (config.getDeviceEnableFlag() != null && !config.getDeviceEnableFlag().equals(old.getDeviceEnableFlag())) {
            updated.setDeviceEnableFlag(config.getDeviceEnableFlag());
            records.add(new SmsAlarmOperRecord().setParkId(config.getParkId())
                    .setModuleType(SmsAlarmOperRecordConsts.MODULE_TYPE_DEVICE)
                    .setOperType(SmsAlarmOperRecordConsts.OPER_TYPE_STATUS)
                    .setOperVal(config.getDeviceEnableFlag())
                    .setOperTime(LocalDateTime.now())
                    .setOperUser(config.getUpdateUser()));
        }
        if (config.getMonthEnableFlag() != null && !config.getMonthEnableFlag().equals(old.getMonthEnableFlag())) {
            updated.setMonthEnableFlag(config.getMonthEnableFlag());
            records.add(new SmsAlarmOperRecord().setParkId(config.getParkId())
                    .setModuleType(SmsAlarmOperRecordConsts.MODULE_TYPE_MONTH)
                    .setOperType(SmsAlarmOperRecordConsts.OPER_TYPE_STATUS)
                    .setOperVal(config.getMonthEnableFlag())
                    .setOperTime(LocalDateTime.now())
                    .setOperUser(config.getUpdateUser()));
        }
        if (config.getVipEnableFlag() != null && !config.getVipEnableFlag().equals(old.getVipEnableFlag())) {
            updated.setVipEnableFlag(config.getVipEnableFlag());
            records.add(new SmsAlarmOperRecord().setParkId(config.getParkId())
                    .setModuleType(SmsAlarmOperRecordConsts.MODULE_TYPE_VIP)
                    .setOperType(SmsAlarmOperRecordConsts.OPER_TYPE_STATUS)
                    .setOperVal(config.getVipEnableFlag())
                    .setOperTime(LocalDateTime.now())
                    .setOperUser(config.getUpdateUser()));
        }
        if (config.getPncCenterEnableFlag() != null && !config.getPncCenterEnableFlag().equals(old.getPncCenterEnableFlag())) {
            updated.setPncCenterEnableFlag(config.getPncCenterEnableFlag());
            records.add(new SmsAlarmOperRecord().setParkId(config.getParkId())
                    .setModuleType(SmsAlarmOperRecordConsts.MODULE_TYPE_PNC_CENTER)
                    .setOperType(SmsAlarmOperRecordConsts.OPER_TYPE_STATUS)
                    .setOperVal(config.getPncCenterEnableFlag())
                    .setOperTime(LocalDateTime.now())
                    .setOperUser(config.getUpdateUser()));
        }
        if (config.getPncChargeEnableFlag() != null && !config.getPncChargeEnableFlag().equals(old.getPncChargeEnableFlag())) {
            updated.setPncChargeEnableFlag(config.getPncChargeEnableFlag());
            records.add(new SmsAlarmOperRecord().setParkId(config.getParkId())
                    .setModuleType(SmsAlarmOperRecordConsts.MODULE_TYPE_PNC_CHARGE)
                    .setOperType(SmsAlarmOperRecordConsts.OPER_TYPE_STATUS)
                    .setOperVal(config.getPncChargeEnableFlag())
                    .setOperTime(LocalDateTime.now())
                    .setOperUser(config.getUpdateUser()));
        }
        if (records.isEmpty()) return true;
        boolean result = updateById(updated);
        smsAlarmOperRecordService.saveBatch(records);
        return result;
    }

    public boolean changeCount(long parkId, int count) {
        return getBaseMapper().changeCount(parkId, count) > 0;
    }
}
