package com.icetech.park.service.report.pnc.impl;

import com.icetech.common.constants.CodeConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.StringUtils;
import com.icetech.cloudcenter.domain.request.p2r.PrintInfoRequest;
import com.icetech.cloudcenter.domain.request.pnc.DataCenterBaseRequest;
import com.icetech.cloudcenter.domain.request.pnc.RobotPrintInfoRequest;
import com.icetech.cloudcenter.domain.enumeration.P2rCmdEnum;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.down.p2r.impl.PrintInfoServiceImpl;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.handle.P2rMsgHandle;
import com.icetech.park.service.report.pnc.ReportService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 机器人打印机信息实现 <br>
 *     接口功能：当需要机器人打印小票时，第三方平台需要调用此接口，例如：打印红包找零的小票；
 */
@Service
@Slf4j
public class RobotPrintInfoServiceImpl extends AbstractService implements ReportService {

    /** 下发机器人打印实现类 */
    @Autowired
    private PrintInfoServiceImpl printInfoService;

    /** 通道缓存管理 */
    @Autowired
    private CacheHandle cacheHandle;

    /** 向机器人下发消息 */
    @Autowired
    private P2rMsgHandle p2rMsgHandle;

    @Override
    public ObjectResponse report(DataCenterBaseRequest dataCenterBaseRequest, Long parkId) {
        // 从请求参数中获取约定的bizContent参数
        RobotPrintInfoRequest robotSayShowRequest = JsonUtils.convert2bean(dataCenterBaseRequest.getBizContent(), RobotPrintInfoRequest.class);

        // 必填项参数校验
        verifyParams(robotSayShowRequest);

        // 获取管道机器人序列号
        String serialNumber = cacheHandle.getChannelRobot(dataCenterBaseRequest.getParkCode(), robotSayShowRequest.getChannelId());
        if (serialNumber != null) {
            // 封装下发机器人打印机信息参数
            PrintInfoRequest printInfoRequest = new PrintInfoRequest();
            printInfoRequest.setOrderNum(robotSayShowRequest.getOrderId()); // 订单号码
            printInfoRequest.setHeadMsg(robotSayShowRequest.getHeadMsg()); // 小票头部信息
            printInfoRequest.setBottomMsg(robotSayShowRequest.getBottomMsg()); // 小票底部提示信息
            printInfoRequest.setQrCodeUrl(robotSayShowRequest.getQrCodeUrl()); // 中间的二维码URL
            // 下发机器人执行打印功能
            String messageId = p2rMsgHandle.send(parkId, serialNumber, P2rCmdEnum.打印机信息下发.getCmdType(), printInfoRequest);
            if (StringUtils.isNotBlank(messageId)) {
                return ObjectResponse.success();
            }
        }

        log.info("[端网云]机器人打印机信息失败，未获取到通道机器人序列号，parkCode[{}]，channelId[{}]",
                dataCenterBaseRequest.getParkCode(), robotSayShowRequest.getChannelId());
        return ObjectResponse.failed(CodeConstants.ERROR_3003, "机器人设备不在线");
    }
}
