package com.icetech.park.service.report.pnc.impl;

import com.alibaba.fastjson.JSON;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.request.pnc.DataCenterBaseRequest;
import com.icetech.cloudcenter.domain.request.pnc.QueryMoreMonthCarRequest;
import com.icetech.cloudcenter.domain.response.pnc.QueryMoreMonthCarResponse;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.park.domain.entity.more.MoreMonthInfo;
import com.icetech.park.domain.entity.more.MoreMonthPark;
import com.icetech.park.domain.entity.more.MoreMonthPlate;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.more.MoreMonthInfoService;
import com.icetech.park.service.more.MoreMonthParkService;
import com.icetech.park.service.more.MoreMonthPlateService;
import com.icetech.park.service.report.pnc.ReportService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author wanggang
 * @create 2024-05-09 17:45
 */
@Service
public class QueryMoreMonthCarServiceImpl extends AbstractService implements ReportService {

    @Autowired
    private MoreMonthInfoService moreMonthInfoService;

    @Autowired
    private MoreMonthParkService moreMonthParkService;

    @Autowired
    private MoreMonthPlateService moreMonthPlateService;

    @Autowired
    private OrderService orderService;

    @Autowired
    private ParkService parkService;

    @Override
    public ObjectResponse report(DataCenterBaseRequest dataCenterBaseRequest, Long parkId) {
        QueryMoreMonthCarRequest request = JSON.parseObject(JSON.toJSONString(dataCenterBaseRequest.getBizContent()), QueryMoreMonthCarRequest.class);
        //参数校验
        verifyParams(request);
        Long moreMonthId = request.getMoreMonthId();
        // 1.获取多车场月卡信息
        MoreMonthInfo moreMonthInfo = moreMonthInfoService.getById(moreMonthId);
        if (moreMonthInfo == null) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "多车场月卡信息不存在");
        }
        if (moreMonthInfo.getCardStatus() != null && moreMonthInfo.getCardStatus() == 3) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "多车场月卡信息不存在");
        }
        // 2.获取多车场月卡关联车场信息
        List<MoreMonthPark> monthParks = moreMonthParkService.getMonthParkByMoreMonthId(moreMonthId);
        if (CollectionUtils.isEmpty(monthParks)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "多车场月卡没有关联车场");
        }
        // 3.获取多车场月卡关联车牌信息
        List<MoreMonthPlate> moreMonthPlates = moreMonthPlateService.getMoreMonthPlateByMoreMonthId(moreMonthId);
        if (CollectionUtils.isEmpty(moreMonthPlates)) {
            return ObjectResponse.failed(CodeConstants.ERROR_404, "多车场月卡没有关联车牌号");
        }
        List<QueryMoreMonthCarResponse> response = monthParks.stream()
                .map(monthPark -> {
                    Long monthParkParkId = monthPark.getParkId();
                    ObjectResponse<Park> parkObj = parkService.findByParkId(monthParkParkId);
                    String parkName = ObjectResponse.isSuccess(parkObj) ? parkObj.getData().getParkName() : "";
                    return moreMonthPlates.stream()
                            .map(MoreMonthPlate::getPlateNum)
                            .map(plateNum -> OrderInfo.builder()
                                    .parkId(monthParkParkId)
                                    .plateNum(plateNum)
                                    .build())
                            .map(orderInfo -> orderService.findByOrderInfo(orderInfo))
                            .filter(ObjectResponse::isSuccess)
                            .map(ObjectResponse::getData)
                            .filter(this::isValidOrder)
                            .map(orderResp -> QueryMoreMonthCarResponse.builder()
                                    .parkId(monthParkParkId)
                                    .parkName(parkName)
                                    .enterTime(orderResp.getEnterTime())
                                    .exitTime(orderResp.getExitTime())
                                    .type(orderResp.getType())
                                    .plateNum(orderResp.getPlateNum())
                                    .status(orderResp.getServiceStatus() == 1 ? 1 : 2)
                                    .build())
                            .collect(Collectors.toList());
                })
                .flatMap(List::stream)
                .collect(Collectors.toList());
        return ObjectResponse.success(response);
    }

    public boolean isValidOrder(OrderInfo orderResp) {
        // 排除 getServiceStatus() 为 3 的订单
        if (OrderStatusConstants.CANCEL == orderResp.getServiceStatus() ) {
            return false;
        }
        // 排除 getServiceStatus() 为 4 且出入场时间相同的订单
        if (OrderStatusConstants.EXCEPTION == orderResp.getServiceStatus() && orderResp.getEnterTime().equals(orderResp.getExitTime())) {
            return false;
        }
        return true;
    }
}
