package com.icetech.park.service.report.pnc.impl;

import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.api.third.ThirdParkService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.cloudcenter.domain.request.pnc.AutopayRequest;
import com.icetech.cloudcenter.domain.request.pnc.DataCenterBaseRequest;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.JsonUtils;
import com.icetech.common.utils.MoneyTool;
import com.icetech.common.utils.StringUtils;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.report.pnc.ReportService;
import com.icetech.partner.api.ThirdAutoPayService;
import com.icetech.partner.api.request.ThirdAutoPayRequest;
import com.icetech.paycenter.api.IPayCenterService;
import com.icetech.paycenter.domain.request.autopay.ExitpayRequest;
import com.icetech.third.domain.entity.third.ThirdInfo;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service("autopayServiceImpl")
@Slf4j
public class PncAutopayServiceImpl extends AbstractService implements ReportService {

    @Autowired
    private IPayCenterService payCenterService; // 注入支付中心服务

    @Autowired
    private OrderService orderService; // 注入订单服务

    @Autowired
    private OrderPayService orderPayService; // 注入订单支付服务

    @Autowired
    private ParkService parkService; // 注入车场服务

    @Autowired
    private ThirdParkService thirdParkService; // 注入第三方车场服务

    @Autowired
    private ThirdAutoPayService thirdAutoPayService; // 注入第三方无感支付服务
    @Autowired
    private RedisUtils redisUtil; // 注入Redis工具类

    @Override
    public ObjectResponse report(DataCenterBaseRequest dataCenterBaseRequest, Long parkId) { // 重写报告方法

        AutopayRequest autopayRequest = JsonUtils.convert2bean(dataCenterBaseRequest.getBizContent(), AutopayRequest.class); // 将请求内容转换为无感支付请求对象

        // 参数校验
        verifyParams(autopayRequest); // 验证参数
        String orderId = autopayRequest.getOrderId(); // 获取订单ID
        if (orderId == null) { // 如果订单ID为空
            throw new ResponseBodyException(CodeConstants.ERROR_400, "检验参数未通过"); // 抛出异常
        }
        OrderInfo orderInfo = new OrderInfo(); // 创建订单信息对象
        orderInfo.setParkId(parkId); // 设置车场ID
        orderInfo.setLocalOrderNum(orderId); // 设置本地订单号
        ObjectResponse<OrderInfo> byOrderInfo = orderService.findByOrderInfo(orderInfo); // 根据订单信息查询订单
        ObjectResponse.notError(byOrderInfo, "本地订单号无效"); // 检查查询结果是否有误

        OrderInfo orderInfoData = byOrderInfo.getData(); // 获取查询到的订单数据
        autopayRequest.setOrderNum(orderInfoData.getOrderNum()); // 设置无感支付请求的订单号
        ThirdAutoPayRequest thirdAutoPayRequest = new ThirdAutoPayRequest(); // 创建第三方无感支付请求对象
        BeanUtils.copyProperties(autopayRequest, thirdAutoPayRequest); // 复制属性
        thirdAutoPayRequest.setParkId(parkId); // 设置车场ID
        thirdAutoPayRequest.setParkCode(dataCenterBaseRequest.getParkCode()); // 设置车场编码
        thirdAutoPayRequest.setEnterTime(orderInfoData.getEnterTime()); // 设置入场时间
        thirdAutoPayRequest.setType(orderInfoData.getType()); // 设置订单类型
        BigDecimal totalAmount = BigDecimal.ZERO;
        if (StringUtils.isNotBlank(autopayRequest.getPaidPrice())) {
            totalAmount = totalAmount.add(new BigDecimal(autopayRequest.getPaidPrice()));
        }
        if (StringUtils.isNotBlank(autopayRequest.getTotalPrice())) {
            totalAmount = totalAmount.add(new BigDecimal(autopayRequest.getTotalPrice()));
        }
        thirdAutoPayRequest.setTotalAmount(totalAmount.toString());
        ObjectResponse<Map<String, Object>> response = report(dataCenterBaseRequest.getParkCode(), parkId, thirdAutoPayRequest);
        if (ObjectResponse.isSuccess(response)) { // 如果响应成功
            Map<String, Object> data = response.getData(); // 获取响应数据
            if (Boolean.TRUE.equals(data.get("isAfterPay"))) { // 如果是后支付
                OrderInfo orderInfoUpdate = new OrderInfo(); // 创建订单更新对象
                orderInfoUpdate.setOrderNum(orderInfoData.getOrderNum()); // 设置订单号
                orderInfoUpdate.setTotalPrice(autopayRequest.getTotalPrice()); // 设置总价格
                orderInfoUpdate.setPaidPrice(autopayRequest.getPaidPrice()); // 设置已支付价格
                orderInfoUpdate.setDiscountPrice(autopayRequest.getDiscountPrice()); // 设置折扣价格
                orderInfoUpdate.setCityAutopay(1); // 设置城市无感支付标志
                orderInfoUpdate.setNeedPrice(autopayRequest.getUnpayPrice()); // 设置未支付价格
                orderService.updateOrderInfo(orderInfoUpdate); // 更新订单信息
            }
        }
        return response; // 返回响应
    }

    public ObjectResponse<Map<String, Object>> report(String parkCode, Long parkId, ThirdAutoPayRequest autoPayRequest) { // 重载报告方法
        log.info("[请求第三方无感支付] autoPayRequest[{}]", autoPayRequest); // 记录日志
        ObjectResponse<Object> response = null; // 声明响应对象
        try {
            if (autoPayRequest.getChannelId() == null && autoPayRequest.getChannelCode() != null) { // 如果通道ID为空且通道编码不为空
                ObjectResponse<ParkInoutdevice> objectResponse = parkService.getInOutDeviceByCode(autoPayRequest.getParkId(), autoPayRequest.getChannelCode()); // 根据通道编码获取进出设备
                ObjectResponse.notError(objectResponse); // 检查查询结果是否有误
                ParkInoutdevice inoutdevice = objectResponse.getData(); // 获取设备数据
                autoPayRequest.setChannelId(inoutdevice.getId()); // 设置通道ID
                autoPayRequest.setChannelName(inoutdevice.getInandoutName()); // 设置通道名称
            }
            // 根据车场查询第三方无感支付配置
            List<ThirdInfo> thirdInfos = thirdParkService.selectNoSenConfigs(parkId);
            if (CollectionUtils.isNotEmpty(thirdInfos)) {
                for (ThirdInfo thirdInfo : thirdInfos) {
                    // 调用第三方无感支付服务
                    ObjectResponse<Object> objectObjectResponse = thirdAutoPayService.autoPay(autoPayRequest, thirdInfo);
                    response = objectObjectResponse;
                    if (ObjectResponse.isSuccess(objectObjectResponse)) {
                        break;
                    }
                }
            }
        } catch (Exception e) { // 捕获异常
            log.error("[请求第三方无感支付] 请求参数[{}], e=", autoPayRequest, e); // 记录错误日志
        }
        if (ObjectResponse.isSuccess(response)) { // 如果响应成功
            if (response.getData() != null) { // 如果响应数据不为空
                Map<String, Object> retMap = JsonUtils.parseMap(JsonUtils.toString(response.getData())); // 解析响应数据
                retMap.put("payWay", 4); // 设置支付方式
                // 移除etc扣费锁
                redisUtil.remove(RedisConstants.ETC_PAY_LOCK_KEY + autoPayRequest.getOrderNum()); // 移除ETC支付锁
                return ObjectResponse.success(retMap); // 返回成功响应
            }
        }
        /*
         * 开始处理业务
         */
        ObjectResponse<ParkConfig> objectResponse = parkService.getParkConfig(parkId); // 获取车场配置
        ObjectResponse.notError(objectResponse); // 检查查询结果是否有误
        ParkConfig data = objectResponse.getData(); // 获取车场配置数据
        if (data == null || data.getIsNosenpayment() != 1) { // 如果配置为空或未开启无感支付功能
            log.info("<离场扣款接口> 车场未开启无感支付功能，参数：{}", autoPayRequest); // 记录日志
            return ObjectResponse.failed(CodeConstants.ERROR_2002, CodeConstants.getName(CodeConstants.ERROR_2002)); // 返回失败响应
        }
        ExitpayRequest exitpayRequest = getExitPayRequest(parkCode, autoPayRequest); // 获取离场支付请求
        // 设置无感支付的类型
        exitpayRequest.setNoSenseType(data.getNoSenPayment()); // 设置无感支付类型
        log.info("准备请求扣费，参数[{}]", exitpayRequest); // 记录日志
        ObjectResponse<Map<String, Object>> objectResponse1 = null; // 声明响应对象
        try {
            objectResponse1 = payCenterService.autopay(exitpayRequest); // 调用支付中心的无感支付方法
            log.info("无感支付响应[{}]", objectResponse1); // 记录响应日志
        } catch (Exception e) { // 捕获异常
            log.error("[请求paycenter无感支付] 请求参数[{}], e=", autoPayRequest, e); // 记录错误日志
        }
        if (objectResponse1 != null && objectResponse1.getCode().equals(CodeConstants.SUCCESS)) { // 如果响应成功
            autoPayRequest.setPaidPrice(autoPayRequest.getUnpayPrice()); // 设置已支付价格
            savePayRecord(autoPayRequest, parkId, exitpayRequest.getTradeNo()); // 保存支付记录
            Map<String, Object> retMap = new HashMap<>(); // 创建返回数据Map

            retMap.put("tradeNo", exitpayRequest.getTradeNo()); // 设置交易号
            retMap.put("payWay", 4); // 设置支付方式
            return ObjectResponse.success(retMap); // 返回成功响应
        }
        return objectResponse1; // 返回响应
    }

    /**
     * 保存交易记录
     * @param autoPayRequest 参数
     * @param tradeNo 交易流水号
     */
    private void savePayRecord(ThirdAutoPayRequest autoPayRequest, Long parkId, String tradeNo) { // 保存支付记录方法

        OrderPay orderPay = new OrderPay(); // 创建订单支付对象
        orderPay.setOrderNum(autoPayRequest.getOrderNum()); // 设置订单号
        orderPay.setPayStatus(PayStatusConstants.PAID); // 设置支付状态为已支付
        orderPay.setParkId(parkId); // 设置车场ID
        orderPay.setDiscountPrice(autoPayRequest.getDiscountPrice()); // 设置折扣价格
        orderPay.setPaidPrice(autoPayRequest.getPaidPrice()); // 设置已支付价格
        // 无感支付
        orderPay.setPayChannel(7); // 设置支付渠道为无感支付
        orderPay.setPayTerminal("银联无感支付"); // 设置支付终端为银联无感支付
        orderPay.setPayTime(autoPayRequest.getPayTime()); // 设置支付时间
        orderPay.setOrderTime(autoPayRequest.getPayTime()); // 设置订单时间
        // 银联
        orderPay.setPayWay(4); // 设置支付方式为银联
        orderPay.setTotalPrice(autoPayRequest.getTotalPrice()); // 设置总价格
        orderPay.setTradeNo(tradeNo); // 设置交易号
        orderPay.setIsSync(0); // 设置同步状态
        orderPayService.addOrderPay(orderPay); // 添加订单支付记录
    }

    private ExitpayRequest getExitPayRequest(String parkCode, ThirdAutoPayRequest autopayRequest) { // 获取离场支付请求方法
        ExitpayRequest exitpayRequest = new ExitpayRequest(); // 创建离场支付请求对象
        exitpayRequest.setParkCode(parkCode); // 设置车场编码

        String tradeNo = CodeTools.GenerateTradeNo(); // 生成交易号
        exitpayRequest.setTradeNo(tradeNo); // 设置交易号

        ObjectResponse<OrderInfo> objectResponse = orderService.findByOrderNum(autopayRequest.getOrderNum()); // 根据订单号查询订单信息
        ObjectResponse.notError(objectResponse); // 检查查询结果是否有误
        OrderInfo orderInfo = objectResponse.getData(); // 获取订单信息
        exitpayRequest.setEnterTime(DateTools.secondTostring(orderInfo.getEnterTime().intValue())); // 设置入场时间
        exitpayRequest.setUnpayPrice(MoneyTool.fromYuanToFen(autopayRequest.getUnpayPrice())); // 设置未支付价格
        exitpayRequest.setTotalPrice(MoneyTool.fromYuanToFen(autopayRequest.getTotalPrice())); // 设置总价格
        exitpayRequest.setExitTime(DateTools.secondTostring(autopayRequest.getPayTime().intValue())); // 设置离场时间
        exitpayRequest.setPlateNum(autopayRequest.getPlateNum()); // 设置车牌号
        exitpayRequest.setOrderNum(orderInfo.getOrderNum()); // 设置订单号
        return exitpayRequest; // 返回离场支付请求对象
    }

}
