package com.icetech.park.service.report.pnc.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkChargeconfig;
import com.icetech.park.domain.entity.park.ParkGuide;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.basics.domain.entity.park.ParkRegion;
import com.icetech.basics.domain.entity.RegionChargeconfig;
import com.icetech.park.service.report.pnc.ReportService;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.request.pnc.DataCenterBaseRequest;
import com.icetech.cloudcenter.domain.request.pnc.ParkingFacilitiesUpRequest;
import com.icetech.cloudcenter.domain.request.pnc.ParkingFacilitiesUpRequest.ChannelInfo;
import com.icetech.cloudcenter.domain.request.pnc.ParkingFacilitiesUpRequest.DeviceInfo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.charge.ChargeServiceImpl;
import com.icetech.basics.service.device.impl.ParkDeviceDaoImpl;
import com.icetech.park.service.park.impl.ParkGuideServiceImpl;
import com.icetech.basics.service.park.impl.ParkInoutdeviceServiceImpl;
import com.icetech.park.service.park.impl.ParkRegionServiceImpl;
import com.icetech.basics.service.park.impl.ParkServiceImpl;
import com.icetech.park.service.park.impl.RegionChargeconfigServiceImpl;
import com.icetech.third.service.third.SendInfoServiceImpl;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.CodeConstantsEnum;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.utils.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 车场区域、通道和设备上报
 */
@Slf4j
@Service
public class ParkingFacilitiesUpServiceImpl extends AbstractService implements ReportService {
    @Autowired
    private ParkServiceImpl parkService;
    @Autowired
    private ParkRegionServiceImpl parkRegionService;
    @Autowired
    private RegionChargeconfigServiceImpl regionChargeConfigService;
    @Autowired
    private ParkInoutdeviceServiceImpl parkInoutDeviceService;
    @Autowired
    private ParkDeviceDaoImpl parkDeviceService;
    @Autowired
    private ChargeServiceImpl chargeService;
    @Autowired
    private ParkGuideServiceImpl parkGuideService;
    @Autowired
    private SendInfoServiceImpl sendInfoService;
    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Override
    @Transactional
    public ObjectResponse report(DataCenterBaseRequest baseRequest, Long parkId) throws Exception {
        if (baseRequest.getBizContent() == null) return ObjectResponse.failed(CodeConstantsEnum.ERROR_400);
        List<ParkingFacilitiesUpRequest> requests = JSON.parseArray(JSON.toJSONString(baseRequest.getBizContent()), ParkingFacilitiesUpRequest.class);
        if (CollectionUtils.isEmpty(requests)) return ObjectResponse.failed(CodeConstantsEnum.ERROR_400);
        Map<String, ParkingFacilitiesUpRequest> regionInfoMap = new HashMap<>(requests.size()); // regionCode : ParkingFacilitiesUpRequest
        Map<String, ChannelInfo> channelInfoMap = new HashMap<>(requests.size() * 2);
        Map<String, DeviceInfo> deviceInfoMap = new HashMap<>(requests.size() * 2 * 2);
        for (ParkingFacilitiesUpRequest request : requests) {
            regionInfoMap.put(request.getRegionCode(), request);
            if (CollectionUtils.isEmpty(request.getWayInfo())) continue;

            for (ChannelInfo channelInfo : request.getWayInfo()) {
                channelInfo.setRegionCode(request.getRegionCode());
                channelInfoMap.put(channelInfo.getInandoutCode(), channelInfo);

                if (CollectionUtils.isEmpty(channelInfo.getDevices())) continue;

                for (DeviceInfo device : channelInfo.getDevices()) {
                    device.setInandoutCode(channelInfo.getInandoutCode());
                    deviceInfoMap.put(device.getDeviceCode(), device);
                }
            }
        }

        Set<String> cacheKeys = new HashSet<>();
        Map<String, ParkRegion> regionMap = handleRegion(parkId, regionInfoMap);
        handleRegionCharge(parkId, regionInfoMap, regionMap);
        cacheKeys.add(RedisKeyConstants.KEY_PREFIX_CHARGE_REGION_PARK + parkId);
        Map<String, ParkInoutdevice> channelMap = handleChannel(parkId, channelInfoMap, regionMap, cacheKeys);
        handleDevice(parkId, deviceInfoMap, channelMap, cacheKeys);

        ParkGuide guide = new ParkGuide();
        guide.setParkId(parkId);
        guide.setStep3(1);
        guide.setStep4(1);
        parkGuideService.updateByParkId(guide);

        List<SendInfo> sendInfos = new LinkedList<>();
        regionMap.values().stream().map(region -> SendInfo.buildUnneeded(parkId, region.getId(), DownServiceEnum.区域信息和计费规则关联关系.getServiceType(), "本地上报")).collect(Collectors.toCollection(() -> sendInfos));
        channelMap.values().stream().map(channel -> SendInfo.buildUnneeded(parkId, channel.getId(), DownServiceEnum.通道信息.getServiceType(), "本地上报")).collect(Collectors.toCollection(() -> sendInfos));
        sendInfoService.saveBatch(sendInfos);

        stringRedisTemplate.delete(cacheKeys);
        return ObjectResponse.success();
    }

    private Map<String, ParkRegion> handleRegion(long parkId, Map<String, ParkingFacilitiesUpRequest> regionInfoMap) {
        List<ParkRegion> parkRegions = parkRegionService.list(Wrappers.lambdaQuery(ParkRegion.class).eq(ParkRegion::getParkId, parkId).eq(ParkRegion::getStatus, 0));
        Map<String, ParkRegion> existsRegionMap = parkRegions.stream().collect(Collectors.toMap(ParkRegion::getRegionCode, Function.identity()));
        List<ParkRegion> outerRegions = new LinkedList<>(); // 外区域列表
        List<ParkRegion> innerRegions = new LinkedList<>(); // 内区域列表
        List<ParkRegion> updateRegions = new LinkedList<>();    // 更新区域列表
        Map<String, ParkRegion> newRegionMap = new HashMap<>(); // 修改后的全部区域列表
        for (Map.Entry<String, ParkingFacilitiesUpRequest> regionInfoEntry : regionInfoMap.entrySet()) {
            ParkingFacilitiesUpRequest regionInfo = regionInfoEntry.getValue();
            if (regionInfo.getRegionType() == 1 && StringUtils.isBlank(regionInfo.getParentCode())) {
                throw new ResponseBodyException(CodeConstants.ERROR_400, "区域[" + regionInfo.getRegionCode() + "]父区域编码不存在");
            }
            ParkRegion existsRegion = existsRegionMap.remove(regionInfoEntry.getKey()); // 从已存在的区域列表内移除本次请求的区域信息, 剩余未移除的即要删除的区域
            ParkRegion newRegion = buildRegion(parkId, regionInfo);
            if (existsRegion == null) {
                if (regionInfo.getRegionType() == 0) {
                    outerRegions.add(newRegion);
                } else {
                    innerRegions.add(newRegion);
                }
            } else {
                newRegion.setId(existsRegion.getId());
                updateRegions.add(newRegion);
            }
            newRegionMap.put(regionInfo.getRegionCode(), newRegion);
        }

        if (!outerRegions.isEmpty()) {
            parkRegionService.saveBatch(outerRegions);  // 先存储新增的外区域信息，获得保存后的外区域ID
        }
        if (!innerRegions.isEmpty()) {
            for (ParkRegion innerRegion : innerRegions) {   // 循环内区域列表，将保存后的外区域ID填充到内区域的父区域ID字段
                ParkRegion outerRegion = newRegionMap.get(innerRegion.getParentCode());
                if (outerRegion == null) {
                    throw new ResponseBodyException(CodeConstants.ERROR_400, "区域[" + innerRegion.getRegionCode() + "]对应父区域[" + innerRegion.getParentCode() + "]不存在");
                }
                innerRegion.setFatherRelationId(outerRegion.getId());
            }
            parkRegionService.saveBatch(innerRegions);
        }
        if (!updateRegions.isEmpty()) {
            for (ParkRegion updateRegion : updateRegions) { // 循环更新区域列表，将保存后的外区域ID填充到更新区域的父区域ID字段
                if (updateRegion.getRegionType() == 2) {
                    ParkRegion region = newRegionMap.get(updateRegion.getParentCode());
                    if (region == null) {
                        throw new ResponseBodyException(CodeConstants.ERROR_400, "区域[" + updateRegion.getRegionCode() + "]对应父区域[" + updateRegion.getParentCode() + "]不存在");
                    }
                    updateRegion.setFatherRelationId(region.getId());
                }
            }
            parkRegionService.updateBatchById(updateRegions);
        }
        if (!existsRegionMap.isEmpty()) {
            List<Long> regionIds = existsRegionMap.values().stream().map(ParkRegion::getId).collect(Collectors.toList());
            log.info("删除区域信息|{}", regionIds);
            parkRegionService.removeByIds(regionIds);
        }
        return newRegionMap;
    }

    private void handleRegionCharge(long parkId, Map<String, ParkingFacilitiesUpRequest> regionInfoMap, Map<String, ParkRegion> regionMap) {
        ObjectResponse<List<ParkChargeconfig>> parkChargeConfigResp = chargeService.getConfigs(parkId);
        Map<String, ParkChargeconfig> parkChargeConfigMap = parkChargeConfigResp.getData() == null ? Collections.emptyMap() :
                parkChargeConfigResp.getData().stream().collect(Collectors.toMap(ParkChargeconfig::getBilltypecode, Function.identity()));
        List<RegionChargeconfig> chargeConfigs = new LinkedList<>();
        for (Map.Entry<String, ParkingFacilitiesUpRequest> regionInfoEntry : regionInfoMap.entrySet()) {
            ParkingFacilitiesUpRequest regionInfo = regionInfoEntry.getValue();
            ParkRegion region = regionMap.get(regionInfo.getRegionCode());
            int regionId = region.getId().intValue();
            if (StringUtils.isNotBlank(regionInfo.getDefaultBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 0, parkChargeConfigMap.get(regionInfo.getDefaultBill()));
                chargeConfig.setDefaultCharge(1);
                chargeConfigs.add(chargeConfig);
            }
            if (StringUtils.isNotBlank(regionInfo.getBlueCarBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 3, parkChargeConfigMap.get(regionInfo.getBlueCarBill()));
                chargeConfigs.add(chargeConfig);
            }
            if (StringUtils.isNotBlank(regionInfo.getYellowCarBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 4, parkChargeConfigMap.get(regionInfo.getYellowCarBill()));
                chargeConfigs.add(chargeConfig);
            }
            if (StringUtils.isNotBlank(regionInfo.getFixedCarBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 5, parkChargeConfigMap.get(regionInfo.getFixedCarBill()));
                chargeConfigs.add(chargeConfig);
            }
            if (StringUtils.isNotBlank(regionInfo.getStoredCardBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 6, parkChargeConfigMap.get(regionInfo.getStoredCardBill()));
                chargeConfigs.add(chargeConfig);
            }
            if (StringUtils.isNotBlank(regionInfo.getNewEnergyCarBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 1, parkChargeConfigMap.get(regionInfo.getNewEnergyCarBill()));
                chargeConfigs.add(chargeConfig);
            }
            if (StringUtils.isNotBlank(regionInfo.getNewEnergyPureEleCarBill())) {
                RegionChargeconfig chargeConfig = buildChargeConfig(regionId, 7, parkChargeConfigMap.get(regionInfo.getNewEnergyPureEleCarBill()));
                chargeConfigs.add(chargeConfig);
            }
        }

        regionChargeConfigService.remove(Wrappers.lambdaQuery(RegionChargeconfig.class).eq(RegionChargeconfig::getParkId, parkId));
        if (!chargeConfigs.isEmpty()) {
            regionChargeConfigService.saveBatch(chargeConfigs);
        }
    }

    private Map<String, ParkInoutdevice> handleChannel(long parkId, Map<String, ChannelInfo> channelInfoMap, Map<String, ParkRegion> regionMap, Set<String> cacheKeys) {
        List<ParkInoutdevice> existsChannels = parkInoutDeviceService.list(Wrappers.lambdaQuery(ParkInoutdevice.class)
                .eq(ParkInoutdevice::getParkId, parkId).eq(ParkInoutdevice::getStatus, 0));
        Map<String, ParkInoutdevice> existsChannelMap = existsChannels.stream().collect(Collectors.toMap(ParkInoutdevice::getInandoutCode, Function.identity()));
        List<ParkInoutdevice> addChannels = new LinkedList<>();
        List<ParkInoutdevice> updateChannels = new LinkedList<>();
        Map<String, ParkInoutdevice> newChannelMap = new HashMap<>(); // 修改后的全部区域列表
        for (Map.Entry<String, ChannelInfo> channelInfoEntry : channelInfoMap.entrySet()) {
            ChannelInfo channelInfo = channelInfoEntry.getValue();
            ParkRegion region = regionMap.get(channelInfo.getRegionCode());
            if (region == null) {
                throw new ResponseBodyException(CodeConstants.ERROR_400, "通道[" + channelInfo.getInandoutCode() + "]对应区域[" + channelInfo.getRegionCode() + "]不存在");
            }
            ParkInoutdevice existsChannel = existsChannelMap.remove(channelInfoEntry.getKey());
            ParkInoutdevice newChannel = buildChannel(parkId, channelInfo);
            newChannel.setRegionId(region.getId());
            if (existsChannel == null) {
                addChannels.add(newChannel);
            } else {
                newChannel.setId(existsChannel.getId());
                updateChannels.add(newChannel);
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL + existsChannel.getId());
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL_CODE + existsChannel.getInandoutCode());
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_GRAY + RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL + existsChannel.getId());
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_GRAY + RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL_CODE + existsChannel.getInandoutCode());
            }
            newChannelMap.put(channelInfo.getInandoutCode(), newChannel);
        }

        if (!addChannels.isEmpty()) {
            parkInoutDeviceService.saveBatch(addChannels);
        }
        if (!updateChannels.isEmpty()) {
            parkInoutDeviceService.updateBatchById(updateChannels);
        }
        if (!existsChannelMap.isEmpty()) {
            List<Long> channelIds = existsChannelMap.values().stream().map(channel -> {
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL + channel.getId());
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL_CODE + channel.getInandoutCode());
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_GRAY + RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL + channel.getId());
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_GRAY + RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL_CODE + channel.getInandoutCode());
                return channel.getId();
            }).collect(Collectors.toList());
            log.info("删除通道信息|{}", channelIds);
            parkInoutDeviceService.removeByIds(channelIds);
        }

        return newChannelMap;
    }

    private Map<String, ParkDevice> handleDevice(long parkId, Map<String, DeviceInfo> deviceInfoMap, Map<String, ParkInoutdevice> channelMap, Set<String> cacheKeys) {
        List<ParkDevice> devices = parkDeviceService.list(Wrappers.lambdaQuery(ParkDevice.class)
                .eq(ParkDevice::getParkId, parkId).eq(ParkDevice::getDelFlag, 0));
        Map<String, ParkDevice> existsDeviceMap = devices.stream().collect(Collectors.toMap(ParkDevice::getDeviceNo, Function.identity()));
        List<ParkDevice> addDevices = new LinkedList<>();
        List<ParkDevice> updateDevices = new LinkedList<>();
        Map<String, ParkDevice> newDeviceMap = new HashMap<>();
        for (Map.Entry<String, DeviceInfo> deviceInfoEntry : deviceInfoMap.entrySet()) {
            DeviceInfo deviceInfo = deviceInfoEntry.getValue();
            ParkInoutdevice channel = channelMap.get(deviceInfo.getInandoutCode());
            if (channel == null) {
                throw new ResponseBodyException(CodeConstants.ERROR_400, "设备[" + deviceInfo.getDeviceCode() + "]对应通道[" + deviceInfo.getInandoutCode() + "]不存在");
            }
            ParkDevice existsDevice = existsDeviceMap.get(deviceInfoEntry.getKey());
            ParkDevice newDevice = buildDevice(parkId, deviceInfo);
            newDevice.setChannelId(channel.getId().intValue());
            if (existsDevice == null) {
                addDevices.add(newDevice);
            } else {
                newDevice.setId(existsDevice.getId());
                updateDevices.add(newDevice);
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_PARK_DEVICE_SERIAL_NUMBER + existsDevice.getSerialNumber());
            }
        }

        if (!addDevices.isEmpty()) {
            parkDeviceService.saveBatch(addDevices);
        }
        if (!updateDevices.isEmpty()) {
            parkDeviceService.updateBatchById(updateDevices);
        }
        if (!existsDeviceMap.isEmpty()) {
            List<Integer> deviceIds = existsDeviceMap.values().stream().map(device -> {
                cacheKeys.add(RedisKeyConstants.KEY_PREFIX_PARK_DEVICE_SERIAL_NUMBER + device.getSerialNumber());
                return device.getId();
            }).collect(Collectors.toList());
            log.info("删除设备信息|{}", deviceIds);
            parkDeviceService.removeByIds(deviceIds);
        }

        return newDeviceMap;
    }

    private ParkRegion buildRegion(long parkId, ParkingFacilitiesUpRequest request) {
        ParkRegion region = new ParkRegion();
//        region.setId(request.getId());
        region.setRegionCode(request.getRegionCode());
        region.setRegionName(request.getRegionName());
        region.setParkId(parkId);
        region.setRegionType(request.getRegionType() == 0 ? 1 : 2);
        if (request.getRegionType() == 0) {
            region.setFatherRelationId(0L);
        } else {
            region.setParentCode(request.getParentCode());
        }
        region.setRegionPark(request.getRegionTotalPark());
        region.setFreePark(request.getFreeSpace());
//        region.setCreateTime(request.getCreateTime());
//        region.setAdder(request.getAdder());
//        region.setUpdateTime(request.getUpdateTime());
//        region.setUpdateUser(request.getUpdateUser());
//        region.setRemark(request.getRemark());
        region.setStatus(0);
//        region.setPositionType(request.getPositionType());
        region.setDefaultName(request.getRegionType());
        region.setIsNewEnergyDiffBill(request.getIsNewEnergyDiffBill());

        region.setIsFullForbidenter(request.getIsFullForbidEnter());
        region.setIsFullForbidMonthEnter(request.getIsFullForbidMonthEnter());
        region.setIsFullForbidVIPEnter(request.getIsFullForbidVIPEnter());
        region.setIsFullForbidReservEnter(request.getIsFullForbidReservEnter());
        region.setIsFullForbidStoreEnter(request.getIsFullForbidStoreEnter());
        region.setVipCarTypeId(request.getVipCarType());
        region.setIsFullForbidblackenter(request.getIsFullForbidBlackEnter());
        region.setFullEmptynum(request.getFullEmptyNum());
        return region;
    }

    private RegionChargeconfig buildChargeConfig(int regionId, int plateType, ParkChargeconfig parkChargeConfig) {
        if (parkChargeConfig == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_404, "未知计费规则");
        }
        RegionChargeconfig config = new RegionChargeconfig();
        config.setParkId(parkChargeConfig.getParkId());
        config.setRegionId(regionId);
        config.setBilltypecode(parkChargeConfig.getBilltypecode());
        config.setBilltype(parkChargeConfig.getBilltype());
        config.setDefaultCharge(0);
        config.setLicensePlateType(plateType);
        config.setStatus(0);
        return config;
    }

    private ParkInoutdevice buildChannel(long parkId, ChannelInfo info) {
        ParkInoutdevice channel = new ParkInoutdevice();
        channel.setParkId(parkId);
        channel.setInandoutCode(info.getInandoutCode());
//        channel.setThirdInandoutCode(info.getThirdInandoutCode());
        channel.setInandoutName(info.getInandoutName());
        channel.setInandoutType(info.getInandoutType());
//        channel.setImage(info.getImage());
        channel.setStatus(0);
//        channel.setCreateTime(info.getCreateTime());
//        channel.setAdder(info.getAdder());
//        channel.setUpdateTime(info.getUpdateTime());
//        channel.setUpdateUser(info.getUpdateUser());
        channel.setIsAllowTempcarrun(info.getIsAllowTempcarrun());
        channel.setIsAllowNocardrun(info.getIsAllowNocardrun());
        channel.setIsAllowYellowcarrun(info.getIsAllowYellowcarrun());
        channel.setIsOpenVaguetype(info.getIsOpenVaguetype());
//        channel.setCacheExpires(info.getCacheExpires());
        channel.setVaguetype(info.getVagueType());
        channel.setIsOpenQrcodetype(info.getIsOpenQrcodetype());
        channel.setIsAllowNewenergycarrun(info.getIsAllowNewenergycarrun());
        channel.setIsAllowVisitCar(info.getIsAllowVisitCar());
        channel.setIsAllowMonthCar(info.getIsAllowMonthCar());
        channel.setIsAllowOvertimeCar(info.getIsOverTimeMonth());
        channel.setOverTimeDay(info.getOverTimeDay());
        channel.setOverTimeMonthlyCard(info.getOverTimeMonthlyCard());
        channel.setIsAllowBackCar(info.getIsAllowBackCar());
        channel.setIsAllowVipCar(info.getIsVipCar());
//        channel.setVipCarTypeId(info.getVipCarTypeId());
        channel.setIsAllowAbcar(info.getIsAllowABcar());
        channel.setIsFee(info.getIsCharge());
        channel.setIsMaster(info.getIsMaster());
//        channel.setRegionId(info.getRegionId());
        channel.setIsAllowBluerun(info.getIsAllowBluerun());
        channel.setIsAllowStoredCar(info.getIsAllowStoredCar());
        return channel;
    }

    private ParkDevice buildDevice(long parkId, DeviceInfo info) {
        ParkDevice device = new ParkDevice();
        device.setParkId(parkId);
        device.setDeviceNo(info.getDeviceCode());
        device.setType(info.getType());
//        device.setChannelId(info.getChannelId());
//        device.setModelType(info.getModelType());
        device.setSerialNumber(info.getSerialNumber());
        device.setIp(info.getIp());
        device.setPort(info.getPort() == null ? null : info.getPort().toString());
//        device.setCallMethods(info.getCallMethods());
        device.setStatus(0);
        device.setDelFlag(0);
//        device.setCreateTime(info.getCreateTime());
//        device.setUpdateTime(info.getUpdateTime());
//        device.setAdder(info.getAdder());
//        device.setEndUpdatetime(info.getEndUpdatetime());
//        device.setLinetype(info.getLinetype());
//        device.setSerialaddr(info.getSerialaddr());
//        device.setBaudrate(info.getBaudrate());
//        device.setVideoUrl(info.getVideoUrl());
//        device.setThirdCode(info.getThirdCode());
        if (info.getDualCamera() == null || info.getDualCamera() == 0) {
            device.setIsDualcamera(0);
            device.setIsMaster(1);
        } else if (info.getIsMaster() == null) {
            throw new ResponseBodyException(CodeConstants.ERROR_400, "缺少是否为主相机");
        } else {
            device.setIsDualcamera(info.getDualCamera());
            device.setIsMaster(info.getIsMaster());
        }
//        device.setVerCode(info.getVerCode());
//        device.setEzopenId(info.getEzopenId());
//        device.setProtocolVer(info.getProtocolVer());
//        device.setProtocolType(info.getProtocolType());
//        device.setProductModel(info.getProductModel());
//        device.setFirmwareVersion(info.getFirmwareVersion());
//        device.setIsAutoswitchCheck(info.getIsAutoswitchCheck());
//        device.setVoicevendorType(info.getVoicevendorType());
//        device.setConnectionMode(info.getConnectionMode());
//        device.setWirelessCarrier(info.getWirelessCarrier());
//        device.setNetwork(info.getNetwork());
        return device;
    }
}
