package com.icetech.park.service.report.p2c.impl;

import com.icetech.cloudcenter.api.month.MonthCarService;
import com.icetech.cloudcenter.api.month.VipCarService;
import com.icetech.cloudcenter.api.third.SendInfoService;
import com.icetech.basics.constants.TextConstant;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.cloudcenter.domain.request.p2c.TaskProgressRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.cloudcenter.domain.constants.BatchSendConstants;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.dao.BatchsendTaskDao;
import com.icetech.park.dao.BatchsendTaskSubDao;
import com.icetech.park.domain.entity.BatchsendTask;
import com.icetech.park.domain.entity.BatchsendTaskSub;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.BatchDownUtils;
import com.icetech.park.service.report.CallService;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

@Service("p2cTaskProgressServiceImpl")
@Slf4j
public class TaskProgressServiceImpl extends AbstractService implements CallService<TaskProgressRequest, Void> {

    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private BatchsendTaskDao batchsendTaskDao;
    @Autowired
    private BatchsendTaskSubDao batchsendTaskSubDao;
    @Autowired
    private SendInfoService sendInfoService;
    @Autowired
    private MonthCarService monthCarService;
    @Autowired
    private VipCarService vipCarService;

    @Override
    public P2cBaseResponse<Void> execute(TokenDeviceVo deviceToken, P2cBaseRequest<TaskProgressRequest> baseRequest) {
        TaskProgressRequest taskProgressRequest = baseRequest.getBizContent();

        //参数校验
        verifyParams(taskProgressRequest);

        String subTaskId = taskProgressRequest.getSubTaskId();
        String subTaskKey = RedisConstants.BATCH_DOWN_SUBTASK_HASH + subTaskId;
        Integer totalNum = redisUtils.hGet(subTaskKey, "totalNum", Integer.class);
        Integer successNum = redisUtils.hGet(subTaskKey, "successNum", Integer.class);
        Integer failNum = redisUtils.hGet(subTaskKey, "failNum", Integer.class);
        Integer currDataType = redisUtils.hGet(subTaskKey, "serviceType", Integer.class);
        Integer thisNum = taskProgressRequest.getThisNum();
        Integer thisFailNum = taskProgressRequest.getThisFailNum();
        Integer doneNum = taskProgressRequest.getDoneNum();
        int successNumNew = successNum + (thisNum - thisFailNum);
        int failNumNew = failNum + taskProgressRequest.getThisFailNum();
        if (successNumNew > totalNum){
            successNumNew = totalNum;
            failNumNew = 0;
        }else{
            //相机未上报的失败条数，不想保存在失败条数里，所以如果失败条数+成功条数大于总条数 有可能是重发的情况,失败条数=总条数-成功条数
            if (failNumNew + successNumNew > totalNum){
                failNumNew = totalNum - successNumNew;
            }
        }
        redisUtils.hPut(subTaskKey, "successNum", successNumNew);
        redisUtils.hPut(subTaskKey, "failNum", failNumNew);
        String reason = redisUtils.hGet(subTaskKey, "reason", String.class);
        String failIds = redisUtils.hGet(subTaskKey, "failIds", String.class);
        String taskId = redisUtils.hGet(subTaskKey, "taskId", String.class);
        String thisFailReason = taskProgressRequest.getThisFailReason();
        if (StringUtils.isNotBlank(thisFailReason)){
            redisUtils.hPut(subTaskKey, "reason", BatchDownUtils.convertMsgAndGetNewMsg(reason, thisFailReason));
        }

        Integer cmdType = BatchSendConstants.convert2BatchBizOpenType(currDataType);
        String dataListKey = RedisConstants.BATCH_DOWN_DATA_LIST + subTaskId + ":" + cmdType;

        //更新处理成功的记录状态
        int startIndex = doneNum - thisNum;
        int endIndex = doneNum - 1;
        List<Long> idLongList = redisUtils.lRange(dataListKey, startIndex, endIndex, Long.class);
        if (BatchSendConstants.MC_BIZ == currDataType){
            ObjectResponse<List<Long>> recordIdList = monthCarService.getRecordIdListByIds(idLongList);
            if (ObjectResponse.isSuccess(recordIdList)){
                idLongList = recordIdList.getData();
            }
        }else if (BatchSendConstants.VIP_BIZ == currDataType){
            ObjectResponse<List<Long>> recordIdList = vipCarService.getRecordIdListByIds(idLongList);
            if (ObjectResponse.isSuccess(recordIdList)){
                idLongList = recordIdList.getData();
            }
        }
        sendInfoService.updateSuccessByServiceIds(idLongList, cmdType, TextConstant.getSuccessDefaultMessage(TextConstant.ONE));
        log.info("[批量下发业务处理进度接口]业务类型[{}],id集合[{}]", cmdType, idLongList);

        if (thisFailNum > 0){
            int failStartIndex = doneNum - thisFailNum;
            int failEndIndex = doneNum - 1;
            List<Long> failIdList = redisUtils.lRange(dataListKey, failStartIndex, failEndIndex, Long.class);
            String ids = failIdList.stream().map(String::valueOf).collect(Collectors.joining(","));
            if (StringUtils.isNotBlank(ids)){
                //限制不能重复累加id
                if (StringUtils.isNotBlank(failIds)){
                    if (failIds.split(",").length + failIdList.size() <= totalNum){
                        redisUtils.hPut(subTaskKey, "failIds", failIds + "," + ids);
                    }
                }else{
                    redisUtils.hPut(subTaskKey, "failIds", ids);
                }
            }
        }

        if (totalNum.equals(successNumNew) || taskProgressRequest.getIsEnd() == 1){
            log.info("[批量下发业务处理进度接口]子任务处理结束[{}]", subTaskId);
            redisUtils.hPut(subTaskKey, "status", 3);
            if (failNumNew == 0){
                redisUtils.hDelete(subTaskKey, "reason");
            }

            BatchsendTaskSub batchsendTaskSub = new BatchsendTaskSub();
            batchsendTaskSub.setId(Integer.parseInt(subTaskId));
            batchsendTaskSub.setStatus(3);
            batchsendTaskSub.setSuccessNum(successNumNew);
            batchsendTaskSub.setFailNum(failNumNew);
            batchsendTaskSub.setTotalNum(totalNum);
            batchsendTaskSubDao.updateById(batchsendTaskSub);

            //处理其他子任务
            dealOtherSubTask(deviceToken.getParkId(), currDataType, taskId, dataListKey);
        }else{
            //处理中
            redisUtils.hPut(subTaskKey, "status", 2);
            redisUtils.hDelete(subTaskKey, "reason");
        }
        return P2cBaseResponse.success(baseRequest);
    }

    /**
     * 判断其他任务是否成功
     * @param parkId
     * @param currDataType
     * @param taskId
     * @param dataListKey
     */
    private void dealOtherSubTask(Long parkId, Integer currDataType, String taskId, String dataListKey) {

        //判断其他任务是否成功
        String taskIdKey = RedisConstants.BATCH_DOWN_TASK_LIST + taskId;
        List<Integer> subTaskList = redisUtils.lRange(taskIdKey, 0, redisUtils.lLen(taskIdKey), Integer.class);
        if (subTaskList != null && subTaskList.size() > 0){
            boolean isAllDataTypeOK = true;
            boolean isAllOk = true;
            for (int i = 0; i < subTaskList.size(); i++){
                Integer subT = subTaskList.get(i);
                Integer status = redisUtils.hGet(RedisConstants.BATCH_DOWN_SUBTASK_HASH + subT, "status", Integer.class);
                Integer dataType = redisUtils.hGet(RedisConstants.BATCH_DOWN_SUBTASK_HASH + subT, "serviceType", Integer.class);

                if (status != 3){
                    isAllOk = false;
                    if (currDataType.equals(dataType)){
                        isAllDataTypeOK = false;
                    }
                    break;
                }
            }
            if (isAllOk){
                BatchsendTask batchsendTask = new BatchsendTask();
                batchsendTask.setTaskId(taskId);
                batchsendTask = batchsendTaskDao.selectOneByEntity(batchsendTask);
                batchsendTask.setStatus(3);
                batchsendTaskDao.updateById(batchsendTask);

                redisUtils.expire(dataListKey, 120);
                log.info("[批量下发业务处理进度接口]所有下发业务都已结束[{}]", taskId);
            }
            if (isAllDataTypeOK){
                //关闭批量处理的状态
                redisUtils.remove(RedisConstants.BATCH_DOWN_OPEN_FLAG + parkId + ":" + BatchSendConstants.convert2BatchBizOpenType(currDataType));
                log.info("[批量下发业务处理进度接口]所有下发业务都已结束[{}],dataType[{}]", taskId, currDataType);
            }
        }
    }
}
