package com.icetech.park.service.report.p2c.impl;

import java.util.concurrent.ThreadPoolExecutor;

import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.request.p2c.RegisterDeviceRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.response.p2c.RegisterDeviceResponse;
import com.icetech.cloudcenter.domain.vo.p2c.ParkConnectedDeviceVo;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.handle.DeviceUpgradeHandle;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.exception.ResponseBodyException;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.UUIDTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service("p2cRegisterDeviceServiceImpl")
@Slf4j
public class RegisterDeviceServiceImpl extends AbstractService {

    @Autowired
    private ParkService parkService;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private DeviceUpgradeHandle deviceUpgradeHandle;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;

    /**
     * 初始化最大时间，超过这个时间会下发基础配置信息
     */
    private static Long INIT_TIME = 30L;

    public P2cBaseResponse<RegisterDeviceResponse> execute(RegisterDeviceRequest registerDeviceRequest){

        /**
         * 返回参数封装
         */
        P2cBaseResponse<RegisterDeviceResponse> p2CBaseResponse = new P2cBaseResponse<>();

        p2CBaseResponse.setCmd(registerDeviceRequest.getCmd() + AbstractService.CMD_SUFFIX);
        p2CBaseResponse.setMessageId(registerDeviceRequest.getMessageId());
        try {
            //参数校验
            verifyParams(registerDeviceRequest);
        } catch (ResponseBodyException e){
            p2CBaseResponse.setCode(CodeEnum.缺失参数.getCode());
            p2CBaseResponse.setMsg(CodeEnum.缺失参数.getMsg());
            return p2CBaseResponse;
        }

        // 此车场编号不能直接用,根据serialNumber查询设备信息
        String serialNumber = registerDeviceRequest.getDeviceNo();
        ObjectResponse<ParkDevice> parkDeviceResp = parkDeviceService.getDeviceBySerialNumber(serialNumber);
        ParkDevice parkDevice = parkDeviceResp.getData();

        if (parkDevice == null || parkDevice.getType() != 1) {
            p2CBaseResponse.setCode(CodeEnum.成功.getCode());
            p2CBaseResponse.setMsg(CodeEnum.成功.getMsg());
            RegisterDeviceResponse response = new RegisterDeviceResponse();
            response.setToken("TEMPTOKEN");
            response.setCloudTime(DateTools.unixTimestamp());
            response.setParkCode("P999");
            response.setEnexType(0);
            p2CBaseResponse.setData(response);
            return p2CBaseResponse;
        }
        // 根据设备信息中的channelId查询通道信息
        ObjectResponse<ParkInoutdevice> parkInoutdeviceObjectResponse = parkService.getInoutDeviceById(parkDevice.getChannelId().longValue());
        if (!ObjectResponse.isSuccess(parkInoutdeviceObjectResponse)) {
            log.info("[设备注册接口] 根据channelId[{}],未查询到通道信息,serialNumber[{}]", parkDevice.getChannelId(), serialNumber);
            p2CBaseResponse.setCode(CodeEnum.非法参数.getCode());
            p2CBaseResponse.setMsg(CodeEnum.非法参数.getMsg());
            return p2CBaseResponse;
        }
        ParkInoutdevice parkInoutdevice = parkInoutdeviceObjectResponse.getData();
        Integer inandoutType = parkInoutdevice.getInandoutType();

        long parkId = parkDevice.getParkId();
        ObjectResponse<Park> byParkCode = parkService.findByParkId(parkId);
        ObjectResponse.notError(byParkCode);
        Park park = byParkCode.getData();
        String parkCode = park.getParkCode();
        // 首次连接或超过时限后再下发基础配置信息
        if (parkDevice.getEndUpdatetime() == null || (System.currentTimeMillis() - parkDevice.getEndUpdatetime().getTime()) / 1000 / 60 > INIT_TIME) {
            redisUtils.set(RedisConstants.DEVICE_INIT + serialNumber, 1, 60L);
        }
        TokenDeviceVo deviceInfo = cacheHandle.getDeviceInfo(serialNumber);
        String token = null;
        if (deviceInfo != null && deviceInfo.getToken() != null) {
            token = deviceInfo.getToken();
            //有缓存时，直接返回
            log.info("deviceInfo[{}]", deviceInfo);
            Integer protocolType = registerDeviceRequest.getSource();
            String protocolVer = registerDeviceRequest.getVersion();
            String productModel = registerDeviceRequest.getProductModel();
            String firmwareVersion = registerDeviceRequest.getFirmwareVersion();

            if (protocolVer.equals(parkDevice.getProtocolVer())
                    && protocolType.equals(parkDevice.getProtocolType())
                    && (productModel == null || productModel.equals(parkDevice.getProductModel()))
                    && (firmwareVersion == null || firmwareVersion.equals(parkDevice.getFirmwareVersion())
                    && deviceInfo.getInandoutType().equals(inandoutType)
                    && deviceInfo.getParkCode().equals(parkCode))
            ) {
                log.info("[设备注册接口] 从缓存中返回parkCode[{}], serialNumber[{}]", parkCode, serialNumber);
                RegisterDeviceResponse response = new RegisterDeviceResponse();
                response.setToken(token);
                response.setCloudTime(DateTools.unixTimestamp());
                response.setParkCode(parkCode);
                response.setEnexType(inandoutType);
                p2CBaseResponse.setData(response);

                p2CBaseResponse.setCode(CodeEnum.成功.getCode());
                p2CBaseResponse.setMsg(CodeEnum.成功.getMsg());
                return p2CBaseResponse;
            }
        }
        if (token == null) {
            token = UUIDTools.getUuid();
        }
        ParkConnectedDeviceVo parkConnectedDeviceVo = new ParkConnectedDeviceVo();
        parkConnectedDeviceVo.setDeviceNo(serialNumber);
        parkConnectedDeviceVo.setInandoutCode(parkInoutdevice.getInandoutCode());
        parkConnectedDeviceVo.setRegionId(parkInoutdevice.getRegionId());
        parkConnectedDeviceVo.setInandoutType(inandoutType);
        parkConnectedDeviceVo.setId(parkInoutdevice.getId());
        //默认为主相机，如果是双摄像机，并且配置为从相机，则设备为false
        if (parkDevice.getIsDualcamera() != null && parkDevice.getIsDualcamera() == 1
                && parkDevice.getIsMaster() != null && parkDevice.getIsMaster() == 0) {
            parkConnectedDeviceVo.setMaster(false);
        } else {
            parkConnectedDeviceVo.setMaster(true);
        }
        //更新已连接设备列表
        cacheHandle.addParkConnectList(parkCode, parkConnectedDeviceVo);

        TokenDeviceVo tdv = new TokenDeviceVo();
        // 车场属性
        tdv.setParkId(park.getId());
        tdv.setParkCode(parkCode);
        // 设备属性
        tdv.setDeviceNo(serialNumber);
        tdv.setMaster(parkConnectedDeviceVo.isMaster());
        tdv.setVersion(registerDeviceRequest.getVersion());
        tdv.setSource(registerDeviceRequest.getSource());
        // 通道数据
        tdv.setId(parkInoutdevice.getId());
        tdv.setInandoutName(parkInoutdevice.getInandoutName());
        tdv.setInandoutCode(parkInoutdevice.getInandoutCode());
        tdv.setRegionId(parkInoutdevice.getRegionId());
        tdv.setInandoutType(parkInoutdevice.getInandoutType());
        tdv.setToken(token);
        //存放token对应的相机信息
        cacheHandle.cacheDeviceInfo(serialNumber, tdv);
        //存放平台设备编号和相机序列号的映射，如果是双摄相机模式，只存主相机序列号，如果主相机不在线，从相机在线，则存从相机序列号
        cacheHandle.setChannelSn(parkCode, parkInoutdevice.getInandoutCode(), serialNumber, parkConnectedDeviceVo.isMaster());
        //设备属性发生变化时处理
        propertyModify(registerDeviceRequest, serialNumber, parkDevice);

        RegisterDeviceResponse response = new RegisterDeviceResponse();
        response.setToken(token);
        response.setCloudTime(DateTools.unixTimestamp());
        response.setParkCode(park.getParkCode());
        response.setEnexType(inandoutType);
        p2CBaseResponse.setData(response);

        p2CBaseResponse.setCode(CodeEnum.成功.getCode());
        p2CBaseResponse.setMsg(CodeEnum.成功.getMsg());
        return p2CBaseResponse;
    }

    /**
     * 相机属性发生变化时处理
     * @param registerDeviceRequest
     * @param serialNumber
     * @param parkDevice
     */
    private void propertyModify(RegisterDeviceRequest registerDeviceRequest, String serialNumber, ParkDevice parkDevice) {
        asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
            //根据条件更新协议版本号和协议类型
            Integer protocolType = registerDeviceRequest.getSource();
            String protocolVer = registerDeviceRequest.getVersion();
            String productModel = registerDeviceRequest.getProductModel();
            String firmwareVersion = registerDeviceRequest.getFirmwareVersion();

            if (firmwareVersion != null) {
                deviceUpgradeHandle.versionModify(serialNumber, firmwareVersion);
            }
            if (!protocolVer.equals(parkDevice.getProtocolVer())
                    || !protocolType.equals(parkDevice.getProtocolType())
                    || (productModel != null && !productModel.equals(parkDevice.getProductModel()))
                    || (firmwareVersion != null && !firmwareVersion.equals(parkDevice.getFirmwareVersion()))
            ) {
                parkDevice.setProtocolVer(protocolVer);
                parkDevice.setProtocolType(protocolType);
                parkDevice.setProductModel(productModel);
                parkDevice.setFirmwareVersion(firmwareVersion);
                parkDeviceService.updateDevice(parkDevice);
                log.info("[设备注册接口] 更新版本号和协议类型完成, sn[{}], protocolType[{}], protocolVer[{}], productModel[{}], firmwareVersion[{}]",
                        serialNumber, protocolType, protocolVer, productModel, firmwareVersion);
            }
        }));
    }
}
