package com.icetech.park.service.report.p2c.impl;

import com.fasterxml.jackson.core.type.TypeReference;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.basics.config.GrayProperties;
import com.icetech.basics.dao.park.ParkInoutdeviceDao;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.park.component.AsyncNotifyClient;
import com.icetech.park.domain.dto.LedConfigDTO;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.basics.domain.entity.park.ParkInoutdevice;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.request.p2c.PropertyInfoRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.domain.entity.park.ParkRecovery;
import com.icetech.park.domain.vo.ParkRecoveryVo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.handle.DeviceUpgradeHandle;
import com.icetech.park.service.park.ParkRecoveryService;
import com.icetech.park.service.report.CallService;
import com.icetech.third.utils.JsonUtils;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.constants.RedisKeyConstants;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.NumberUtils;
import com.icetech.common.utils.SpringUtils;
import com.icetech.redis.handle.RedisHandle;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ThreadPoolExecutor;

@Service("p2cPropertyInfoServiceImpl")
@Slf4j
@RefreshScope
public class PropertyInfoServiceImpl extends AbstractService implements CallService<PropertyInfoRequest, Map<String, Object>> {
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private DeviceUpgradeHandle deviceUpgradeHandle;
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private ParkService parkService;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;
    @Autowired
    private AsyncNotifyClient asyncNotifyClient;

    @Value("${model.config}")
    private String modelConfig;

    @Autowired
    private ParkInoutdeviceDao parkInoutdeviceDao;

    @Resource
    private RedisHandle redisHandle;

    @Autowired
    private GrayProperties grayProperties;

    @Autowired
    private ParkRecoveryService parkRecoveryService;


    @Override
    public P2cBaseResponse<Map<String, Object>> execute(TokenDeviceVo deviceInfo, P2cBaseRequest<PropertyInfoRequest> baseRequest) {
        PropertyInfoRequest propertyInfoRequest = baseRequest.getBizContent();

        //参数校验
        verifyParams(propertyInfoRequest);

        //版本号不存在
        String reportVersion = propertyInfoRequest.getVersion().trim();
        if (P2cVersionEnum.getIndex(reportVersion) == 0){
            return P2cBaseResponse.instance(baseRequest, CodeEnum.非法参数.getCode(), "版本号不存在");
        }
        Map<String, Object> dataMap = new HashMap<>();
        dataMap.put("enexType", deviceInfo.getInandoutType());
        dataMap.put("parkCode", deviceInfo.getParkCode());
        dataMap.put("protocolType", deviceInfo.getSource());
        ObjectResponse<ParkConfig> parkConfigObjectResponse = parkService.getParkConfig(deviceInfo.getParkId());
        ObjectResponse.notError(parkConfigObjectResponse);
        ParkConfig parkConfig = parkConfigObjectResponse.getData();
        dataMap.put("retentionAlarm", 1);
        dataMap.put("retentionTimeThreshold",
                NumberUtils.toPrimitive(parkConfig.getCarenexTimelong()) * 60 < 30 ? 60 : parkConfig.getCarenexTimelong() * 60);
        dataMap.put("retentionAlarmMethod", NumberUtils.toPrimitive(parkConfig.getRetentionAlarmMethod(), 1 ));
        dataMap.put("retentionIntervalTime", NumberUtils.toPrimitive(parkConfig.getRetentionIntervalTime() , 600));
        if (P2cVersionEnum.getIndex(deviceInfo.getVersion()) >= P2cVersionEnum.版本18.getIndex()) {
            List<ParkRecoveryVo> recoverys = parkRecoveryService.getOpenParkRecoveryByParkIds(Collections.singletonList(deviceInfo.getParkId()));
            if (CollectionUtils.isNotEmpty(recoverys)) {
                dataMap.put("enableShamPlate", parkConfig.getEnableShamPlate());
                dataMap.put("shamPlateMethod", parkConfig.getShamPlateMethod());
            } else {
                if (propertyInfoRequest.getEnableShamPlate() != null) {
                    dataMap.put("enableShamPlate", propertyInfoRequest.getEnableShamPlate());
                    dataMap.put("shamPlateMethod", propertyInfoRequest.getShamPlateMethod());
                }
            }
        }
        //设备属性发生变化时处理
        propertyModify(deviceInfo, propertyInfoRequest, baseRequest.getMessageId());

        return P2cBaseResponse.success(baseRequest, dataMap);

    }

    /**
     * 相机属性发生变化时处理
     * @param deviceInfo
     * @param propertyInfoRequest
     */
    private void propertyModify(TokenDeviceVo deviceInfo, PropertyInfoRequest propertyInfoRequest, String messageId) {
        asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
            String serialNumber = deviceInfo.getDeviceNo();
            Integer protocolType = deviceInfo.getSource();
            //如果和缓存的版本号不一致时，更新缓存版本号
            if (!propertyInfoRequest.getVersion().equals(deviceInfo.getVersion())
                    || (propertyInfoRequest.getNewBluetooth() != null && !propertyInfoRequest.getNewBluetooth().equals(deviceInfo.getNewBluetooth()))) {
                deviceInfo.setVersion(propertyInfoRequest.getVersion());
                deviceInfo.setNewBluetooth(propertyInfoRequest.getNewBluetooth());
                cacheHandle.updateDeviceInfo(deviceInfo.getDeviceNo(), deviceInfo);
            }
            String param = redisUtils.get(RedisConstants.PROPERTY_GET_FLAG_PROFILE + deviceInfo.getDeviceNo(), String.class);
            if (param != null){
                String[] params = param.split("/");
                redisUtils.set(RedisConstants.PROPERTY_GET_INFO_PROFILE + deviceInfo.getDeviceNo(), propertyInfoRequest);
                asyncNotifyClient.callBack(params[0], params[1], SpringUtils.getActiveProfile(), propertyInfoRequest);
            }

            ObjectResponse<ParkDevice> deviceBySerialNumber = parkDeviceService.getDeviceBySerialNumber(serialNumber);
            ParkDevice parkDevice = deviceBySerialNumber.getData();
            //根据条件更新协议版本号和协议类型
            String protocolVer = propertyInfoRequest.getVersion();
            String productModel = propertyInfoRequest.getProductModel();
            String firmwareVersion = propertyInfoRequest.getFirmwareVersion();
            Integer connectionMode = propertyInfoRequest.getConnectionMode();
            Integer wirelessCarrier = propertyInfoRequest.getWirelessCarrier();
            String network = propertyInfoRequest.getNetwork();
            Integer newBluetooth = propertyInfoRequest.getNewBluetooth();

            if (firmwareVersion != null){
                deviceUpgradeHandle.versionModify(serialNumber, firmwareVersion);
            }
            if (!protocolVer.equals(parkDevice.getProtocolVer())
                    || !protocolType.equals(parkDevice.getProtocolType())
                    || (productModel != null && !productModel.equals(parkDevice.getProductModel()))
                    || (firmwareVersion != null && !firmwareVersion.equals(parkDevice.getFirmwareVersion()))
                    || (connectionMode != null && !connectionMode.equals(parkDevice.getCallMethods()))
                    || (wirelessCarrier != null && !wirelessCarrier.equals(parkDevice.getWirelessCarrier()))
                    || (network != null && !network.equals(parkDevice.getNetwork()))
                    || (newBluetooth != null && !newBluetooth.equals(parkDevice.getNewBluetooth()))
            ) {
                parkDevice.setProtocolVer(protocolVer);
                parkDevice.setProtocolType(protocolType);
                parkDevice.setProductModel(productModel);
                parkDevice.setFirmwareVersion(firmwareVersion);
                parkDevice.setCallMethods(connectionMode);
                parkDevice.setWirelessCarrier(wirelessCarrier);
                parkDevice.setNetwork(network);
                parkDevice.setNewBluetooth(newBluetooth);
                parkDeviceService.updateDevice(parkDevice);
                log.info("[设备属性上报接口] 更新版本号和协议类型完成, sn[{}], protocolType[{}], protocolVer[{}], productModel[{}], firmwareVersion[{}]",
                        serialNumber, protocolType, protocolVer, productModel, firmwareVersion);
            }
            // 根据相机型号设置控制卡类型
            if (StringUtils.isNotBlank(modelConfig)) {
                List<LedConfigDTO> ledConfigDTOS = JsonUtils.fromJson(modelConfig, new TypeReference<List<LedConfigDTO>>() {
                });
                if (CollectionUtils.isNotEmpty(ledConfigDTOS)) {
                    Optional<LedConfigDTO> ledConfigOptional = ledConfigDTOS.stream().
                            filter(a -> StringUtils.isNotBlank(a.getModel()) && a.getModel().equals(productModel))
                            .findFirst();
                    if (ledConfigOptional.isPresent()) {
                        LedConfigDTO ledConfigDTO = ledConfigOptional.get();
                        ObjectResponse<ParkInoutdevice> channelDevice = parkService.getInoutDeviceById(Long.valueOf(parkDevice.getChannelId()));
                        // 修改元数据
                        if (ObjectResponse.isSuccess(channelDevice)) {
                            ParkInoutdevice parkInoutdevice = channelDevice.getData();
                            // 不一致时修改
                            if (!parkInoutdevice.getLedcardType().equals(ledConfigDTO.getValue()) ||
                                    !parkInoutdevice.getTtsType().equals(ledConfigDTO.getTtsType())) {
                                log.info("[设备属性上报接口]控制卡类型和tts有其中一不一致");
                                ParkInoutdevice update = new ParkInoutdevice();
                                update.setId(parkInoutdevice.getId());
                                update.setLedcardType(ledConfigDTO.getValue());
                                update.setTtsType(ledConfigDTO.getTtsType());
                                parkInoutdeviceDao.updateById(update);
                                // 删除缓冲
                                String deleteKey = RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL + parkDevice.getChannelId();
                                String deleteCode = RedisKeyConstants.KEY_PREFIX_PARK_CHANNEL_CODE  + parkInoutdevice.getInandoutCode();
                                if (grayProperties.isCacheEnable()) {
                                    deleteKey = RedisKeyConstants.KEY_PREFIX_GRAY + deleteKey;
                                    deleteCode = RedisKeyConstants.KEY_PREFIX_GRAY + deleteCode;
                                }
                                redisHandle.delete(deleteKey);
                                redisHandle.delete(deleteCode);
                            }
                        }
                    }
                }
            }
        }));
    }
}
