package com.icetech.park.service.report.p2c.impl;

import java.util.concurrent.ThreadPoolExecutor;

import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.domain.constants.RedisConstants;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.park.domain.request.HeartbeatRequest;
import com.icetech.cloudcenter.domain.response.p2c.HeartbeatResponse;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.report.CallService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.thread.ThreadUtils;
import com.icetech.common.utils.DateTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Slf4j
@Service("p2cHeartbeatServiceImpl")
public class HeartbeatServiceImpl extends AbstractService implements CallService<HeartbeatRequest, HeartbeatResponse> {
    @Autowired
    private CacheHandle cacheHandle;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private ThreadPoolExecutor asyncExecutor;

    @Override
    public P2cBaseResponse<HeartbeatResponse> execute(TokenDeviceVo deviceToken,
                                                      P2cBaseRequest<HeartbeatRequest> baseRequest) {
        String serialNumber = deviceToken.getDeviceNo();
        String version = deviceToken.getVersion();

        HeartbeatRequest heartbeatRequest = baseRequest.getBizContent();
        //临时统一与相机的时间
        if (P2cVersionEnum.版本1.getVersion().equals(version)){
            Long deviceTime = heartbeatRequest.getDeviceTime();
            heartbeatRequest.setDeviceTime(DateTools.getUnixTime(deviceTime));
        }
        //参数校验
        verifyParams(heartbeatRequest);

        //版本号不存在
        String reportVersion = heartbeatRequest.getVersion();
        if (reportVersion != null && P2cVersionEnum.getIndex(reportVersion) == 0){
            return P2cBaseResponse.instance(baseRequest,
                    CodeEnum.非法参数.getCode(), "版本号不存在");
        }
        //更新心跳时间
        redisUtils.hPut(RedisConstants.P2C_HEARTBEAT_TIME, serialNumber,
                DateTools.unixTimestamp());
        //异步处理
        asyncHandle(deviceToken, serialNumber, heartbeatRequest.getVersion());
        // 返回结果
        HeartbeatResponse heartbeatResponse = new HeartbeatResponse();
        heartbeatResponse.setCloudTime(DateTools.unixTimestamp());
        return P2cBaseResponse.success(baseRequest, heartbeatResponse);
    }

    private void asyncHandle(TokenDeviceVo deviceToken, String serialNumber, String reportVersion) {
        asyncExecutor.execute(ThreadUtils.wrapTrace(() -> {
            String parkCode = deviceToken.getParkCode();
            String channelCode = deviceToken.getInandoutCode();
            String version = deviceToken.getVersion();

            boolean isUpdate = false;
            if (reportVersion != null && !reportVersion.equals(version)) {
                isUpdate = true;
                deviceToken.setVersion(reportVersion);
                //更新数据库
                ObjectResponse<ParkDevice> deviceBySerialNumber =
                        parkDeviceService.getDeviceBySerialNumber(serialNumber);
                if (ObjectResponse.isSuccess(deviceBySerialNumber)
                        && Integer.valueOf(1).equals(deviceBySerialNumber.getData().getType())) {
                    ParkDevice parkDevice = deviceBySerialNumber.getData();
                    parkDevice.setProtocolVer(reportVersion);
                    parkDeviceService.updateDevice(parkDevice);
                }
            }
            if (isUpdate) {
                cacheHandle.cacheDeviceInfo(serialNumber, deviceToken);
            } else {
                //刷新redis有效期
                cacheHandle.expireDeviceInfo(serialNumber);
            }
            if (!redisUtils.exists(RedisConstants.SERIAL_NUMBER_PROFILE + parkCode + "_" + channelCode)) {
                cacheHandle.setChannelSn(parkCode, channelCode, serialNumber, true);
            } else {
                cacheHandle.expireChannelSn(parkCode, channelCode);
            }
        }));
    }
}
