package com.icetech.park.service.report.p2c.impl;

import com.icetech.cloudcenter.api.order.CarOrderEnterService;
import com.icetech.cloudcenter.api.order.CarOrderExitService;
import com.icetech.cloudcenter.api.order.OrderPayService;
import com.icetech.cloudcenter.api.order.OrderService;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.DataCommonConstants;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderPay;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.p2c.CompleteOrderRequest;
import com.icetech.cloudcenter.domain.response.QueryOrderFeeResponse;
import com.icetech.cloudcenter.domain.response.p2c.CarEnterResult;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.park.service.AbstractService;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.report.CallService;
import com.icetech.common.constants.CodeConstants;
import com.icetech.common.constants.OrderCarInfoConstant;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.constants.PayStatusConstants;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.domain.response.PageQuery;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

@Service("p2cCompleteOrderServiceImpl")
@Slf4j
public class CompleteOrderServiceImpl extends AbstractService implements CallService<List<CompleteOrderRequest>, Object> {
    @Autowired
    private CarOrderExitService carOrderExitService;
    @Autowired
    private OrderService orderService;
    @Autowired
    private CarOrderEnterService carOrderEnterService;
    @Autowired
    private OrderPayService orderPayService;
    @Autowired
    private ParkService parkService;
    @Autowired
    private CacheHandle cacheHandle;

    @Override
    public P2cBaseResponse<Object> execute(TokenDeviceVo deviceToken, P2cBaseRequest<List<CompleteOrderRequest>> baseRequest) {
        List<CompleteOrderRequest> list = baseRequest.getBizContent();
        if (list == null || list.isEmpty()){
            return P2cBaseResponse.instance(baseRequest, CodeEnum.缺失参数.getCode());
        }else{
            for (int i = 0; i < list.size(); i++) {
                CompleteOrderRequest completeOrderRequest = list.get(i);
                P2cBaseResponse<?> p2cBaseResponse = execute(deviceToken, baseRequest, completeOrderRequest);
                if (!p2cBaseResponse.getCode().equals(CodeEnum.成功.getCode())) {
                    log.info("第[{}]个元素的参数处理失败，参数[{}}, 处理结果[{}]", i, completeOrderRequest, p2cBaseResponse);
                }
            }
            return P2cBaseResponse.success(baseRequest);
        }

    }
    public P2cBaseResponse<?> execute(TokenDeviceVo deviceToken, P2cBaseRequest<List<CompleteOrderRequest>> baseRequest, CompleteOrderRequest completeOrderRequest) {
        Long parkId = deviceToken.getParkId();
        String parkCode = deviceToken.getParkCode();
        String inandoutCode = deviceToken.getInandoutCode();
        String inandoutName = deviceToken.getInandoutName();
        //参数校验
        verifyParams(completeOrderRequest);
        //扩展参数设置
        completeOrderRequest.setParkId(parkId);
        completeOrderRequest.setParkCode(parkCode);
        completeOrderRequest.setInandoutCode(inandoutCode);
        completeOrderRequest.setInandoutName(inandoutName);
        //不以上报的订单号为准
        completeOrderRequest.setOrderNum(null);

        /*
         * 开始处理业务
         */
        String plateNum = completeOrderRequest.getPlateNum();
        Long enterTime = completeOrderRequest.getEnterTime();

        if (enterTime == null) {
            /*
             * 没有进场记录的情况
             */
            completeOrderRequest.setEnterTime(completeOrderRequest.getExitTime());
            return saveEnex(baseRequest, parkCode, inandoutName, inandoutCode, completeOrderRequest);
        } else {
            /*
             * 有完整进出记录的情况
             */
            OrderInfo orderInfo;
            if (!DataCommonConstants.isNoPlate(plateNum)) {
                /*
                 * 入场后脱机情况
                 */
                OrderInfo orderInfo1 = new OrderInfo();
                orderInfo1.setPlateNum(plateNum);
                orderInfo1.setParkId(parkId);
                //判断车辆是否在场内
                ObjectResponse<OrderInfo> inPark = orderService.findByOrderInfo(orderInfo1);
                orderInfo = inPark.getData();
                if (orderInfo != null && OrderStatusConstants.IN_PARK == orderInfo.getServiceStatus()) {
                    //以最新的在场记录的订单号为准
                    completeOrderRequest.setOrderNum(orderInfo.getOrderNum());
                    completeOrderRequest.setType(orderInfo.getType());
                } else {
                    //如果已经离场，直接返回
                    if (orderInfo != null
                            && (OrderStatusConstants.LEAVED_PARK == orderInfo.getServiceStatus()
                                || OrderStatusConstants.EXCEPTION == orderInfo.getServiceStatus())
                            //脱机上报的入场时间早于最近的订单入场时间，则直接返回（识别无权限，摇控器开闸时，平台记录的入场时间是开闸的时间）
                            && orderInfo.getEnterTime().compareTo(enterTime) >= 0) {
                        log.info("[端云-脱机完整订单补报] 车辆已经离场[{}], orderNum[{}]", plateNum, orderInfo.getOrderNum());
                        return P2cBaseResponse.success(baseRequest);
                    }
                    /*
                     * 按模糊规则查询场内车辆
                     */
                    inPark = orderService.fuzzyPlate(parkId, inandoutCode, plateNum);
                    if (inPark != null && inPark.getCode().equals(CodeConstants.SUCCESS)) {
                        orderInfo = inPark.getData();
                        log.info("[端云-脱机完整订单补报] 离场车牌[{}], 模糊匹配到车牌[{}]", plateNum, orderInfo.getPlateNum());
                        //以最新的在场记录的订单号为准
                        completeOrderRequest.setOrderNum(orderInfo.getOrderNum());
                        plateNum = orderInfo.getPlateNum();
                        completeOrderRequest.setPlateNum(plateNum);
                        completeOrderRequest.setType(orderInfo.getType());
                    }
                }
                if (orderInfo == null || !orderInfo.getEnterTime().equals(enterTime)) {
                    // 未匹配到平台记录
                    return saveEnex(baseRequest, parkCode, inandoutName, inandoutCode, completeOrderRequest);
                } else {
                    // 处理交易记录
                    dealPayRecord(inandoutName, inandoutCode, completeOrderRequest, orderInfo.getOrderNum(), orderInfo);
                    // 匹配到了平台的在场记录
                    completeOrderRequest.setOrderNum(orderInfo.getOrderNum());
                    ObjectResponse<?> objectResponse = normalExit(completeOrderRequest, parkCode);

                    if (ObjectResponse.isSuccess(objectResponse)){
                        return P2cBaseResponse.success(baseRequest);
                    }else{
                        return P2cBaseResponse.instance(baseRequest, Integer.parseInt(objectResponse.getCode()));
                    }
                }
            }else{
                /*
                 * 不是有效车牌号的情况
                 */
                return saveEnex(baseRequest, parkCode, inandoutName, inandoutCode, completeOrderRequest);
            }
        }
    }
    private void dealPayRecord(String inandoutName, String inandoutCode, CompleteOrderRequest completeOrderRequest, String orderNum, OrderInfo orderInfo) {
        Integer paidAmountFen = completeOrderRequest.getPaidAmountFen();
        Integer discountAmountFen = completeOrderRequest.getDiscountAmountFen();

        ObjectResponse<ParkConfig> parkConfigObjectResponse = parkService.getParkConfig(orderInfo.getParkId());
        orderInfo.setExitTime(completeOrderRequest.getExitTime());
        ObjectResponse<QueryOrderFeeResponse> objectResponse = orderService.p2cQueryFee(orderInfo, parkConfigObjectResponse.getData());
        //平台计费待缴费
        double platform_unpayPrice = 0;
        //平台计费本次优惠
        double platform_discountPrice = 0;
        QueryOrderFeeResponse orderFeeResponse = null;
        if (ObjectResponse.isSuccess(objectResponse)){
            orderFeeResponse = objectResponse.getData();
            platform_unpayPrice = Double.parseDouble(orderFeeResponse.getUnpayPrice());
            platform_discountPrice = Double.parseDouble(orderFeeResponse.getDiscountPrice());
        }
        if (paidAmountFen != null && discountAmountFen != null
                && paidAmountFen + discountAmountFen > 0){
            PageQuery pageQuery = new PageQuery();
            OrderPay orderPayPara = new OrderPay();
            orderPayPara.setParkId(completeOrderRequest.getParkId());
            orderPayPara.setOrderNum(orderNum);
            orderPayPara.setPayStatus(PayStatusConstants.PAID);
            pageQuery.setParam(orderPayPara);
            ObjectResponse<List<OrderPay>> orderPayListObjectResponse = orderPayService.findList(pageQuery);
            /**
             * 判断是否已经有脱机下的支付记录，如果已经支付，则不处理相机上报的支付金额
             */
            if (ObjectResponse.isSuccess(orderPayListObjectResponse)) {
                List<OrderPay> orderPayList = orderPayListObjectResponse.getData();
                for (OrderPay orderPay : orderPayList) {
                    if (orderPay.getPayTerminal() != null) {
                        log.info("[端云-脱机完整订单补报] 已经有支付记录，无需处理相机上报交易，订单号[{}]", orderNum);
                        return;
                    }
                }
            }
            //创建交易记录对象
            OrderPay orderPay = new OrderPay();
            orderPay.setPayStatus(PayStatusConstants.PAID);
            orderPay.setOrderNum(orderNum);
            orderPay.setChannelId(inandoutCode);
            orderPay.setPayWay(1);
            orderPay.setTradeNo(CodeTools.GenerateTradeNo());
            orderPay.setPayTime(completeOrderRequest.getExitTime());
            orderPay.setOrderTime(completeOrderRequest.getExitTime());
            orderPay.setTotalPrice(String.valueOf((paidAmountFen + discountAmountFen)/100.0));
            orderPay.setPaidPrice(String.valueOf(paidAmountFen/100.0));
            orderPay.setDiscountPrice(String.valueOf(discountAmountFen/100.0));
            orderPay.setPayTerminal(inandoutName);
            orderPay.setPayChannel(10);//遥控器
            orderPay.setIsSync(0);
            //保存
            ObjectResponse orderPayResponse = orderPayService.addOrderPay(orderPay);
            if (ObjectResponse.isSuccess(orderPayResponse)){
                log.info("[端云-脱机完整订单补报]保存交易记录成功,参数[{}],响应[{}]",orderPay, JsonUtils.toString(orderPayResponse));
            }else{
                log.info("[端云-脱机完整订单补报]保存交易记录失败,参数[{}],响应[{}]",orderPay, JsonUtils.toString(orderPayResponse));
            }
        }else if (paidAmountFen != null && discountAmountFen != null){
            //前提是相机上报的实收+优惠为0
            if (platform_unpayPrice + platform_discountPrice > 0){
                if (platform_unpayPrice == 0){
                    //创建交易记录对象
                    OrderPay orderPay = new OrderPay();
                    orderPay.setPayStatus(PayStatusConstants.PAID);
                    orderPay.setOrderNum(orderNum);
                    orderPay.setChannelId(inandoutCode);
                    orderPay.setPayWay(1);
                    orderPay.setTradeNo(CodeTools.GenerateTradeNo());
                    orderPay.setPayTime(completeOrderRequest.getExitTime());
                    orderPay.setOrderTime(completeOrderRequest.getExitTime());
                    orderPay.setTotalPrice(String.valueOf((platform_unpayPrice + platform_discountPrice)));
                    orderPay.setPaidPrice("0.00");
                    orderPay.setDiscountPrice(String.valueOf(platform_discountPrice));
                    orderPay.setPayTerminal(inandoutName);
                    orderPay.setPayChannel(10);//遥控器
                    orderPay.setIsSync(0);
                    //保存
                    ObjectResponse orderPayResponse = orderPayService.addOrderPay(orderPay);
                    if (ObjectResponse.isSuccess(orderPayResponse)){
                        log.info("[端云-脱机完整订单补报]保存交易记录成功,参数[{}],响应[{}]",orderPay, JsonUtils.toString(orderPayResponse));
                    }else{
                        log.info("[端云-脱机完整订单补报]保存交易记录失败,参数[{}],响应[{}]",orderPay, JsonUtils.toString(orderPayResponse));
                    }
                }else{
                    completeOrderRequest.setException(true);
                    completeOrderRequest.setTotalAmount(orderFeeResponse.getTotalAmount());
                    completeOrderRequest.setPaidAmount(orderFeeResponse.getPaidAmount());
                    completeOrderRequest.setDiscountAmount(orderFeeResponse.getDiscountAmount());
                }
            }
        }
    }

    private P2cBaseResponse saveEnex(P2cBaseRequest p2CBaseRequest, String parkCode, String inandoutName, String inandoutCode, CompleteOrderRequest completeOrderRequest) {
        //匹配到入场，但不是同一次停车时，清除订单号
        completeOrderRequest.setOrderNum(null);
        String orderNum = normalEnter(completeOrderRequest);
        if (orderNum == null){
            return P2cBaseResponse.instance(p2CBaseRequest, CodeEnum.服务器异常.getCode());
        }
        completeOrderRequest.setOrderNum(orderNum);

        ObjectResponse<OrderInfo> orderInfoObjectResponse = orderService.findByOrderNum(orderNum);
        ObjectResponse.notError(orderInfoObjectResponse);
        //处理交易记录
        dealPayRecord(inandoutName, inandoutCode, completeOrderRequest, orderNum, orderInfoObjectResponse.getData());
        ObjectResponse objectResponse = normalExit(completeOrderRequest, parkCode);
        if (ObjectResponse.isSuccess(objectResponse)){
            return P2cBaseResponse.success(p2CBaseRequest);
        }else{
            return P2cBaseResponse.instance(p2CBaseRequest, Integer.parseInt(objectResponse.getCode()));
        }
    }

    private String normalEnter(CompleteOrderRequest completeOrderRequest) {
        CarEnterRequest carEnterRequest = new CarEnterRequest();
        BeanUtils.copyProperties(completeOrderRequest, carEnterRequest);
        carEnterRequest.setEnterWay(OrderCarInfoConstant.IN_OUT_WAY_PLATE_NUM);
        ObjectResponse<CarEnterResult> enterObjectResponse = carOrderEnterService.enter(carEnterRequest);
        if (ObjectResponse.isSuccess(enterObjectResponse)){
            return enterObjectResponse.getData().getOrderNum();
        }else{
            log.info("[端云-脱机完整订单补报] 入场失败, 车牌号[{}], 原因[{}]", completeOrderRequest.getPlateNum(), enterObjectResponse.getMsg());
            return null;
        }
    }

    public ObjectResponse normalExit(CompleteOrderRequest completeOrderRequest, String parkCode) {
        CarExitRequest carExitRequest = new CarExitRequest();
        BeanUtils.copyProperties(completeOrderRequest, carExitRequest);
        carExitRequest.setMaxImage(completeOrderRequest.getExitMaxImage());
        carExitRequest.setSmallImage(completeOrderRequest.getExitSmallImage());
        log.info("[端云-脱机完整订单补报] 准备请求cloudcenter离场服务, 参数[{}]", carExitRequest);
        ObjectResponse<Map<String, Object>> objectResponse;
        if (completeOrderRequest.isException()){
            //非正常出场
            objectResponse = carOrderExitService.exceptionExit(carExitRequest, 4);
        }else{
            objectResponse = carOrderExitService.exit(carExitRequest);
        }

        if (ObjectResponse.isSuccess(objectResponse)) {
            //清除缓存中的上次异常记录
            cacheHandle.removeExit(parkCode, completeOrderRequest.getInandoutCode());
            cacheHandle.removeChannelFee(parkCode, completeOrderRequest.getInandoutCode());
        }
        return objectResponse;

    }


}
