package com.icetech.park.service.report.p2c.impl;

import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.enumeration.TriggerTypeEnum;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnexResponse;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.impl.DualCameraV2ServiceImpl;
import com.icetech.park.service.report.CallService;
import com.icetech.park.service.report.p2c.impl.exit.CarExitHandler;
import com.icetech.third.service.third.MqPushService;
import com.icetech.third.utils.RedisUtils;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.utils.DateTools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

@Service("p2cCarExitServiceImpl")
@Slf4j
public class CarExitServiceImpl extends AbstractService implements CallService<CarExitRequest, CarEnexResponse> {
    private static final String REDIS_KEY_LOCK_EXIT_PREFIX = "camera:report:exit:";
    private static final long REDIS_KEY_LOCK_TIMEOUT = 10000;
    @Autowired
    private CarExitHandler carExitHandler;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private DualCameraV2ServiceImpl dualCameraService;

    @Override
    public P2cBaseResponse<CarEnexResponse> execute(TokenDeviceVo deviceToken, P2cBaseRequest<CarExitRequest> baseRequest) {
        CarExitRequest exitRequest = baseRequest.getBizContent();
        String lockKey = REDIS_KEY_LOCK_EXIT_PREFIX + deviceToken.getInandoutCode() + ":"
                + exitRequest.getPlateNum() + ":" + exitRequest.getExitTime();

        boolean lock = redisUtils.tryLock(lockKey, baseRequest.getMessageId(), REDIS_KEY_LOCK_TIMEOUT);
        if (!lock) {
            String oldMsgId = redisUtils.getLockValue(lockKey);
            redisUtils.updateLockValue(lockKey, baseRequest.getMessageId());
            P2cBaseResponse p2cBaseResponse = new P2cBaseResponse();
            p2cBaseResponse.setMessageId(oldMsgId);
            p2cBaseResponse.setCmd(baseRequest.getCmd() + AbstractService.CMD_SUFFIX);
            p2cBaseResponse.setCode(CodeEnum.请求重复.getCode());
            p2cBaseResponse.setMsg("正在处理中");
            return p2cBaseResponse;
        }

        try {
            //临时统一与相机的时间
            if (P2cVersionEnum.版本1.getVersion().equals(deviceToken.getVersion())) {
                Long exitTime = exitRequest.getExitTime();
                exitRequest.setExitTime(DateTools.getUnixTime(exitTime));
            }
            //参数校验
            verifyParams(exitRequest);

            //扩展参数设置
            exitRequest.setParkId(deviceToken.getParkId());
            exitRequest.setParkCode(deviceToken.getParkCode());
            exitRequest.setInandoutCode(deviceToken.getInandoutCode());
            exitRequest.setInandoutName(deviceToken.getInandoutName());
            exitRequest.setOriginalExitTime(exitRequest.getExitTime());
            // 记录上报信息
            carExitHandler.addExitRecord(exitRequest);

            //不以上报的订单号为准
            exitRequest.setOrderNum(null);
            P2cBaseResponse<CarEnexResponse> response = carExitHandler.execute(deviceToken, baseRequest);
            String messageId = redisUtils.getLockValue(lockKey);
            if (messageId != null) {
                response.setMessageId(messageId);
            }
            dualCameraHandle(deviceToken, exitRequest, response.getData());
            pushDigitalTwin(deviceToken, exitRequest, response.getData());
            return response;
        } finally {
            redisUtils.releaseLock(lockKey);
        }
    }

    /**
     * 双摄业务处理
     * @param deviceToken
     * @param exitRequest
     * @param carEnexResponse
     */
    private void dualCameraHandle(TokenDeviceVo deviceToken, CarExitRequest exitRequest, CarEnexResponse carEnexResponse) {
        if (TriggerTypeEnum.软触发.getVal().equals(exitRequest.getTriggerType())
                || Integer.valueOf(2).equals(exitRequest.getProperty())) {
            return;
        }
        //异步将结果转发当前通道的其他相机
        dualCameraService.afterHandle(carEnexResponse, deviceToken.getParkId(), deviceToken.getParkCode(),
                deviceToken.getInandoutCode(), deviceToken.getInandoutType(),
                exitRequest.getPlateNum(), exitRequest.getOpenFlag(), deviceToken.getDeviceNo());
    }

    @Autowired
    private MqPushService mqPushService;
    private void pushDigitalTwin(TokenDeviceVo deviceToken, CarExitRequest request, CarEnexResponse response) {
        try {
            if (response == null || StringUtils.isBlank(response.getShow())) return;
            mqPushService.pushOrderEnexShow(deviceToken.getParkId(), deviceToken.getInandoutCode(), deviceToken.getDeviceNo(), response.getShow());
        } catch (Exception e) {
            log.warn("推送数据孪生出错", e);
        }
    }
}
