package com.icetech.park.service.report.p2c.impl;

import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.P2cVersionEnum;
import com.icetech.cloudcenter.domain.enumeration.TriggerTypeEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.response.p2c.CarEnexResponse;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.utils.StringUtils;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.impl.DualCameraV2ServiceImpl;
import com.icetech.park.service.report.CallService;
import com.icetech.park.service.report.p2c.impl.enter.CarEnterHandler;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.utils.CodeTools;
import com.icetech.common.utils.DateTools;
import com.icetech.third.service.third.MqPushService;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

@Service("p2cCarEnterServiceImpl")
@Slf4j
public class CarEnterServiceImpl extends AbstractService implements CallService<CarEnterRequest, CarEnexResponse> {
    @Autowired
    private CarEnterHandler carEnterHandler;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private DualCameraV2ServiceImpl dualCameraService;
    private static final String REDIS_KEY_LOCK_ENTER_PREFIX = "camera:report:enter:";
    private static final long REDIS_KEY_LOCK_TIMEOUT = 10000;

    @Override
    public P2cBaseResponse<CarEnexResponse> execute(TokenDeviceVo deviceToken, P2cBaseRequest<CarEnterRequest> baseRequest) {
        CarEnterRequest enterRequest = baseRequest.getBizContent();
        String lockKey = REDIS_KEY_LOCK_ENTER_PREFIX + deviceToken.getInandoutCode() + ":"
                + enterRequest.getPlateNum() + ":" + enterRequest.getEnterTime();

        boolean lock = redisUtils.tryLock(lockKey, baseRequest.getMessageId(), REDIS_KEY_LOCK_TIMEOUT);
        if (!lock) {
            String oldMsgId = redisUtils.getLockValue(lockKey);
            redisUtils.updateLockValue(lockKey, baseRequest.getMessageId());
            P2cBaseResponse<CarEnexResponse> p2cBaseResponse = new P2cBaseResponse<>();
            p2cBaseResponse.setMessageId(oldMsgId);
            p2cBaseResponse.setCmd(baseRequest.getCmd() + CMD_SUFFIX);
            p2cBaseResponse.setCode(CodeEnum.请求重复.getCode());
            p2cBaseResponse.setMsg("正在处理中");
            return p2cBaseResponse;
        }
        try {
            String version = deviceToken.getVersion();
            String serialNumber = deviceToken.getDeviceNo();

            //临时统一与相机的时间
            if (P2cVersionEnum.版本1.getVersion().equals(version)) {
                Long enterTime = enterRequest.getEnterTime();
                enterRequest.setEnterTime(DateTools.getUnixTime(enterTime));
            }
            //参数校验
            verifyParams(enterRequest);
            //扩展参数设置
            enterRequest.setParkId(deviceToken.getParkId());
            enterRequest.setParkCode(deviceToken.getParkCode());
            enterRequest.setInandoutCode(deviceToken.getInandoutCode());
            enterRequest.setInandoutName(deviceToken.getInandoutName());
            // 预生成订单号供后续流程使用
            enterRequest.setOrderNum(CodeTools.GenerateOrderNum());
            // 记录上报信息
            carEnterHandler.addEnterRecord(enterRequest);
            CarEnexResponse carEnexResponse = carEnterHandler.execute(enterRequest, serialNumber, version);
            dualCameraHandle(deviceToken, enterRequest, carEnexResponse);
            pushDigitalTwin(deviceToken, enterRequest, carEnexResponse);
            return P2cBaseResponse.success(baseRequest, carEnexResponse);
        } finally {
            redisUtils.releaseLock(lockKey);
        }
    }

    private void dualCameraHandle(TokenDeviceVo deviceToken, CarEnterRequest enterRequest, CarEnexResponse carEnexResponse) {
        if (TriggerTypeEnum.软触发.getVal().equals(enterRequest.getTriggerType())
                || Integer.valueOf(2).equals(enterRequest.getProperty())) {
            return;
        }
        //异步将结果转发给主相机处理，并返回响应给从相机的结果
        dualCameraService.afterHandle(carEnexResponse, deviceToken.getParkId(), deviceToken.getParkCode(),
                deviceToken.getInandoutCode(), deviceToken.getInandoutType(),
                enterRequest.getPlateNum(), enterRequest.getOpenFlag(), deviceToken.getDeviceNo());
    }

    @Autowired
    private MqPushService mqPushService;
    private void pushDigitalTwin(TokenDeviceVo deviceToken, CarEnterRequest request, CarEnexResponse response) {
        try {
            if (response == null || StringUtils.isBlank(response.getShow())) return;
            mqPushService.pushOrderEnexShow(deviceToken.getParkId(), deviceToken.getInandoutCode(), deviceToken.getDeviceNo(), response.getShow());
        } catch (Exception e) {
            log.warn("推送数据孪生出错", e);
        }
    }
}
