package com.icetech.park.service.report.mor.impl;

import cn.hutool.core.thread.ThreadUtil;
import com.icetech.basics.domain.entity.device.ParkDevice;
import com.icetech.cloudcenter.api.park.ParkDeviceService;
import com.icetech.cloudcenter.domain.enumeration.CodeEnum;
import com.icetech.cloudcenter.domain.enumeration.MorVersionEnum;
import com.icetech.cloudcenter.domain.request.mor.MorPropertyInfoRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.handle.DeviceUpgradeHandle;
import com.icetech.park.service.handle.MorCacheHandle;
import com.icetech.park.service.report.CallService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
@RefreshScope
@Component
@Slf4j
public class MorPropertyInfoServiceImpl extends AbstractService implements CallService<MorPropertyInfoRequest, Map<String, Object>> {

    @Autowired
    private ParkDeviceService parkDeviceService;
    @Autowired
    private DeviceUpgradeHandle deviceUpgradeHandle;
    @Autowired
    private MorCacheHandle cacheHandle;
    @Value("${mor.videoKbps:1024}")
    private Integer videoKbps;

    @Override
    public P2cBaseResponse<Map<String, Object>> execute(TokenDeviceVo deviceInfo, P2cBaseRequest<MorPropertyInfoRequest> baseRequest) {
        MorPropertyInfoRequest propertyInfoRequest = baseRequest.getBizContent();

        //参数校验
        verifyParams(propertyInfoRequest);

        //版本号不存在
        String reportVersion = propertyInfoRequest.getVersion().trim();
        if (MorVersionEnum.getIndex(reportVersion) == 0){
            return P2cBaseResponse.instance(baseRequest, CodeEnum.非法参数.getCode(), "版本号不存在");
        }

        Map<String, Object> dataMap = new HashMap<>();
        dataMap.put("videoKbps", videoKbps);
        //dataMap.put("videoDpi", );
        //设备属性发生变化时处理
        propertyModify(propertyInfoRequest, deviceInfo.getDeviceNo(), deviceInfo.getSource());

        return P2cBaseResponse.success(baseRequest, dataMap);

    }

    /**
     * 相机属性发生变化时处理
     * @param propertyInfoRequest 属性参数
     * @param serialNumber sn
     */
    private void propertyModify(MorPropertyInfoRequest propertyInfoRequest, String serialNumber, Integer protocolType) {
        ThreadUtil.execute(() -> {
            ObjectResponse<ParkDevice> deviceBySerialNumber = parkDeviceService.getDeviceBySerialNumber(serialNumber);
            ParkDevice parkDevice = deviceBySerialNumber.getData();
            //根据条件更新协议版本号和协议类型
            String protocolVer = propertyInfoRequest.getVersion();
            String productModel = propertyInfoRequest.getProductModel();
            String firmwareVersion = propertyInfoRequest.getFirmwareVersion();
            Integer connectionMode = propertyInfoRequest.getConnectionMode();
            Integer wirelessCarrier = propertyInfoRequest.getWirelessCarrier();
            String network = propertyInfoRequest.getNetwork();

            if (firmwareVersion != null){
                deviceUpgradeHandle.versionModify(serialNumber, firmwareVersion);
            }
            //如果和缓存的版本号不一致时，更新缓存版本号
            TokenDeviceVo deviceInfo = cacheHandle.getDeviceInfo(serialNumber);
            if (!propertyInfoRequest.getVersion().equals(deviceInfo.getVersion())){
                deviceInfo.setVersion(propertyInfoRequest.getVersion());
                cacheHandle.updateDeviceInfo(serialNumber, deviceInfo);
            }
            if (!protocolVer.equals(parkDevice.getProtocolVer())
                    || (productModel != null && !productModel.equals(parkDevice.getProductModel()))
                    || (firmwareVersion != null && !firmwareVersion.equals(parkDevice.getFirmwareVersion()))
                    || (connectionMode != null && !connectionMode.equals(parkDevice.getCallMethods()))
                    || (wirelessCarrier != null && !wirelessCarrier.equals(parkDevice.getWirelessCarrier()))
                    || (network != null && !network.equals(parkDevice.getNetwork()))
            ) {
                parkDevice.setProtocolVer(protocolVer);
                parkDevice.setProductModel(productModel);
                parkDevice.setFirmwareVersion(firmwareVersion);
                parkDevice.setCallMethods(connectionMode);
                parkDevice.setWirelessCarrier(wirelessCarrier);
                parkDevice.setNetwork(network);
                parkDeviceService.updateDevice(parkDevice);
                log.info("[设备属性上报接口] 更新版本号和协议类型完成, sn[{}], protocolType[{}], protocolVer[{}], productModel[{}], firmwareVersion[{}]",
                        serialNumber, protocolType, protocolVer, productModel, firmwareVersion);
            }
        });
    }
}
