package com.icetech.park.service.report.mor.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.basics.service.redis.RedisMsgListener;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.request.mor.BarrierGateStatusRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.cloudcenter.domain.websocket.WebsocketPushData;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.JsonUtils;
import com.icetech.order.domain.entity.BarrierGateException;
import com.icetech.order.service.BarrierGateExceptionService;
import com.icetech.park.domain.entity.park.Park;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.report.CallService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;

@RefreshScope
@Component
@Slf4j
public class MorBarrierGateStatusServiceImpl extends AbstractService implements CallService<BarrierGateStatusRequest, Void> {

    @Resource
    private BarrierGateExceptionService barrierGateExceptionService;

    @Resource
    private ParkService parkService;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Value("${barrier-gate.alarm.push-message:defaultValue}")
    private String parkId;

    @Value("${barrier-gate.alarm.all.push-message.turn:false}")
    private boolean barrierGateTurn;

    @Override
    public P2cBaseResponse<Void> execute(TokenDeviceVo deviceInfo, P2cBaseRequest<BarrierGateStatusRequest> baseRequest) {
        BarrierGateStatusRequest barrierGateStatusRequest = baseRequest.getBizContent();

        //参数校验
        verifyParams(barrierGateStatusRequest);

        BarrierGateException one = barrierGateExceptionService.selectLimitOne(Wrappers.lambdaQuery(BarrierGateException.class)
                .eq(BarrierGateException::getChannelId, deviceInfo.getId()).orderByDesc(BarrierGateException::getId));
        if (one != null && barrierGateStatusRequest.getStatus().equals(one.getStatus())) {
            return P2cBaseResponse.success(baseRequest, null);
        }
        if (one == null && barrierGateStatusRequest.getStatus() == 1) {
            return P2cBaseResponse.success(baseRequest, null);
        }
        if (barrierGateStatusRequest.getStatus() == 2) {
            BarrierGateException barrierGateException = new BarrierGateException();
            barrierGateException.setChannelId(deviceInfo.getId());
            barrierGateException.setReportSn(deviceInfo.getDeviceNo());
            barrierGateException.setChannelName(deviceInfo.getInandoutName());
            barrierGateException.setParkId(deviceInfo.getParkId());
            barrierGateException.setStatus(2);
            barrierGateException.setAlarmTime(new Date(barrierGateStatusRequest.getEventTime() * 1000));
            barrierGateException.setExceptionImage(barrierGateStatusRequest.getImage());
            barrierGateExceptionService.addBarrierGateException(barrierGateException);
            //推送道闸异常到ws
            this.pushBarrierGateAlarmMessage(deviceInfo);
        }
        if (one != null && barrierGateStatusRequest.getStatus() == 1) {
            BarrierGateException barrierGateException = new BarrierGateException();
            barrierGateException.setId(one.getId());
            barrierGateException.setStatus(1);
            barrierGateException.setRecoverTime(new Date(barrierGateStatusRequest.getEventTime() * 1000));
            barrierGateException.setRecoverImage(barrierGateStatusRequest.getImage());
            barrierGateExceptionService.updateById(barrierGateException);
        }
        return P2cBaseResponse.success(baseRequest, null);

    }

    /**
     * 推送道闸异常告警信息
     *
     * @param deviceInfo 设备信息
     */
    private void pushBarrierGateAlarmMessage(TokenDeviceVo deviceInfo) {
        //直营车场发送道闸异常信息
        if (parkId.contains(String.valueOf(deviceInfo.getParkId())) || barrierGateTurn) {
            log.info("收到道闸异常通道信息:{}", deviceInfo);
            ObjectResponse<Park> findByParkId = parkService.findByParkId(deviceInfo.getParkId());
            if (!ObjectResponse.isSuccess(findByParkId)) {
                return;
            }
            Park park = findByParkId.getData();
            WebsocketPushData pushData = new WebsocketPushData();
            pushData.setRecordType(4);
            pushData.setParkName(park.getParkName());
            pushData.setParkCode(park.getParkCode());
            pushData.setParkId(park.getId());
            pushData.setChannelCode(deviceInfo.getInandoutCode());
            pushData.setChannelType(deviceInfo.getInandoutType());
            pushData.setChannelName(deviceInfo.getInandoutName());
            String data = JsonUtils.toString(pushData);
            log.info("发送道闸异常消息:{}", data);
            redisTemplate.convertAndSend(RedisMsgListener.TOPIC, data);
        }
    }
}
