package com.icetech.park.service.report.mor.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.icetech.basics.domain.entity.park.ParkConfig;
import com.icetech.cloudcenter.api.constants.MqConstants;
import com.icetech.cloudcenter.api.park.ParkService;
import com.icetech.cloudcenter.domain.constants.DataCommonConstants;
import com.icetech.cloudcenter.domain.constants.MorRedisKeyConstants;
import com.icetech.cloudcenter.domain.enumeration.DownServiceEnum;
import com.icetech.cloudcenter.domain.request.CarEnterRequest;
import com.icetech.cloudcenter.domain.request.CarExitRequest;
import com.icetech.cloudcenter.domain.request.mor.BackEventRequest;
import com.icetech.cloudcenter.domain.response.p2c.P2cBaseResponse;
import com.icetech.cloudcenter.domain.vo.ApplyCarVideoVo;
import com.icetech.cloudcenter.domain.vo.p2c.TokenDeviceVo;
import com.icetech.common.constants.OrderStatusConstants;
import com.icetech.common.domain.request.P2cBaseRequest;
import com.icetech.common.domain.response.ObjectResponse;
import com.icetech.common.utils.DateTools;
import com.icetech.common.utils.NumberUtils;
import com.icetech.mq.sender.RabbitSender;
import com.icetech.order.dao.OrderModifyRecordDao;
import com.icetech.order.domain.entity.OrderBack;
import com.icetech.order.domain.entity.OrderCarInfo;
import com.icetech.order.domain.entity.OrderInfo;
import com.icetech.order.domain.entity.OrderModifyRecord;
import com.icetech.order.domain.entity.OrderTags;
import com.icetech.order.service.OrderNotpayService;
import com.icetech.order.service.impl.OrderBackServiceImpl;
import com.icetech.order.service.impl.OrderCarInfoServiceImpl;
import com.icetech.order.service.impl.OrderTagsServiceImpl;
import com.icetech.park.handle.CacheHandle;
import com.icetech.park.service.AbstractService;
import com.icetech.park.service.order.impl.OrderServiceImpl;
import com.icetech.park.service.report.CallService;
import com.icetech.third.dao.send.SendinfoDao;
import com.icetech.third.domain.entity.third.SendInfo;
import com.icetech.third.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 车辆折返上报
 * @author wgq
 */
@Slf4j
@Component
@RefreshScope
public class MorBackEventServiceImpl extends AbstractService implements CallService<BackEventRequest, Void> {
    @Resource
    private OrderServiceImpl orderService;
    @Resource
    private OrderCarInfoServiceImpl orderCarInfoService;
    @Resource
    private OrderBackServiceImpl orderBackService;
    @Resource
    private OrderTagsServiceImpl orderTagsService;
    @Resource
    private RabbitSender rabbitSender;
    @Resource
    private RedisUtils redisUtils;
    @Resource
    private CacheHandle cacheHandle;
    @Resource
    private OrderModifyRecordDao orderModifyRecordDao;
    @Resource
    private SendinfoDao sendinfoDao;
    @Resource
    private ParkService parkService;
    @Resource
    private OrderNotpayService orderNotpayService;
    @Value("${aisle.applyCarVideo.time:30}")
    private Integer aisleApplyCarVideoTime;
    @Value("${aisle.exceptionExit.time:600}")
    private Integer aisleExceptionExitTime;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public P2cBaseResponse<Void> execute(TokenDeviceVo deviceInfo, P2cBaseRequest<BackEventRequest> baseRequest) {
        BackEventRequest backEventRequest = baseRequest.getBizContent();
        //参数校验
        verifyParams(backEventRequest);
        //获取订单信息
        OrderInfo orderInfo = this.getOrderInfo(backEventRequest, deviceInfo);
        if (orderInfo == null) {
            log.warn("[折返事件上报] 未匹配到订单, [{}]", backEventRequest);
            return P2cBaseResponse.success(baseRequest, null);
        }
        //出入场判断
        if (deviceInfo.getInandoutType() == 1) {
            //添加折返记录
            OrderBack orderBack = this.insertOrderBack(orderInfo, backEventRequest, deviceInfo);
            //给当前订单打标记
            this.orderTags(orderInfo);
            //请求车辆在通道的视频
            this.getApplyCarVideo(deviceInfo, orderInfo, backEventRequest, orderBack);
            //标记订单的折返缓存
            redisUtils.set(MorRedisKeyConstants.BACK_TAG + orderInfo.getOrderNum(), orderBack, aisleExceptionExitTime + 5);
            //更新通道缓存
            this.updateEnterChannelCache(deviceInfo, orderInfo, orderBack);
        } else {
            //上次订单离场超过了20分钟
            if (OrderStatusConstants.LEAVED_PARK == orderInfo.getServiceStatus()
                    && NumberUtils.toPrimitive(backEventRequest.getRecTime(), DateTools.unixTimestamp()) - orderInfo.getExitTime() > 1200) {
                log.warn("[折返事件上报] 上次订单离场超过了20分钟, 未匹配到订单, [{}]", backEventRequest);
                return P2cBaseResponse.success(baseRequest, null);
            }
            //获取车场配置
            ObjectResponse<ParkConfig> parkConfigResp = parkService.getParkConfig(orderInfo.getParkId());
            //是否是端网云
            boolean isPnc = parkConfigResp.getData() != null && parkConfigResp.getData().getDataCollection() == 2;
            //端网云或者车辆不在场
            if (isPnc || OrderStatusConstants.IN_PARK != orderInfo.getServiceStatus()) {
                //修改订单为在场
                this.updateOrderInfo(orderInfo);
                //保存订单修改记录
                this.insertOrderModifyRecord(orderInfo, isPnc);
            }
            //添加折返记录
            OrderBack orderBack = this.insertOrderBack(orderInfo, backEventRequest, deviceInfo);
            //给当前订单打标记
            this.orderTags(orderInfo);
            //请求车辆在通道的视频
            this.getApplyCarVideo(deviceInfo, orderInfo, backEventRequest, orderBack);
            //标记订单的折返缓存
            redisUtils.set(MorRedisKeyConstants.BACK_TAG + orderInfo.getOrderNum(), orderBack, aisleExceptionExitTime + 5);
            //更新通道缓存
            this.updateExitChannelCache(deviceInfo, orderInfo, orderBack);
        }
        return P2cBaseResponse.success(baseRequest, null);
    }

    /**
     * 获取订单
     *
     * @param backEventRequest 入参
     * @param deviceVo         通道信息
     * @return orderInfo
     */
    private OrderInfo getOrderInfo(BackEventRequest backEventRequest, TokenDeviceVo deviceVo) {
        //空车牌 & 无牌车 & 未识别
        if (StringUtils.isBlank(backEventRequest.getPlateNum()) || DataCommonConstants.isNoPlate(backEventRequest.getPlateNum())) {
            //通过车牌查找订单
            ObjectResponse<List<OrderInfo>> response = orderService.selectListByParam(deviceVo.getParkId(),
                    backEventRequest.getBackStartTime(), backEventRequest.getBackEndTime(), null);
            if (!ObjectResponse.isSuccess(response)) {
                log.warn("[折返事件上报] 请求失败！, [{}]", backEventRequest);
                return null;
            }
            //订单信息
            List<OrderInfo> infoList = response.getData();
            //通过车牌把订单分组
            Map<String, List<OrderInfo>> listMap = infoList.stream().collect(Collectors.groupingBy(OrderInfo::getOrderNum));
            //通过车牌找到该订单车辆信息
            ObjectResponse<List<OrderCarInfo>> carInfoObjectResponse = orderCarInfoService.getCarInfoList(listMap.keySet(), deviceVo.getParkId());
            OrderInfo orderInfo = null;
            if (ObjectResponse.isSuccess(carInfoObjectResponse)) {
                for (OrderCarInfo datum : carInfoObjectResponse.getData()) {
                    //找到相同入口通道的或者找到相同出口通道的
                    if (deviceVo.getInandoutCode().equals(datum.getEnterChannelId()) || deviceVo.getInandoutCode().equals(datum.getExitChannelId())) {
                        orderInfo = listMap.get(datum.getOrderNum()).get(0);
                    }
                }
            }
            return orderInfo;
        } else {
            //查询最后一次入场时间
            return orderService.selectLimitOne(Wrappers.lambdaQuery(OrderInfo.class)
                    .eq(OrderInfo::getParkId, deviceVo.getParkId())
                    .eq(OrderInfo::getPlateNum, backEventRequest.getPlateNum())
                    .orderByDesc(OrderInfo::getEnterTime));
        }
    }

    /**
     * 保存订单修改记录
     *
     * @param orderInfo 订单信息
     * @param isPnc     是否是断网云
     */
    private void insertOrderModifyRecord(OrderInfo orderInfo, Boolean isPnc) {
        OrderModifyRecord orderModifyRecord = new OrderModifyRecord();
        // 修改订单状态
        orderModifyRecord.setAction(1);
        orderModifyRecord.setCorrectType(4);
        orderModifyRecord.setOrderNum(orderInfo.getOrderNum());
        orderModifyRecord.setParkId(orderInfo.getParkId());
        orderModifyRecord.setBeforeModify(String.valueOf(orderInfo.getServiceStatus()));
        orderModifyRecord.setAfterModify(String.valueOf(OrderStatusConstants.IN_PARK));
        orderModifyRecord.setModifyTime(DateTools.getFormat(new Date()));
        orderModifyRecord.setOperAccount("system.mor");
        orderModifyRecordDao.insert(orderModifyRecord);
        if (isPnc) {
            sendinfoDao.insert(new SendInfo(orderInfo.getParkId(), orderModifyRecord.getId(), DownServiceEnum.订单修改.getServiceType()));
        }
    }

    /**
     * 添加折返记录
     *
     * @param orderInfo 订单信息
     * @param backEvent 上报信息
     * @param deviceVo  通道信息
     */
    private OrderBack insertOrderBack(OrderInfo orderInfo, BackEventRequest backEvent, TokenDeviceVo deviceVo) {
        OrderBack orderBack = new OrderBack();
        BeanUtils.copyProperties(orderInfo, orderBack);
        if (orderInfo.getExitTime() == null) {
            if (backEvent.getRecTime() != null) {
                orderBack.setExitTime(backEvent.getRecTime());
            } else {
                orderBack.setExitTime(backEvent.getBackStartTime() - 2);
            }
        }
        if (NumberUtils.toPrimitive(orderInfo.getNoneEnterFlag()) == 1) {
            orderBack.setEnterTime(null);
        }
        orderBack.setId(null);
        orderBack.setCreateTime(null);
        orderBack.setBackStartTime(backEvent.getBackStartTime());
        orderBack.setBackEndTime(backEvent.getBackEndTime());
        orderBack.setVideoId(backEvent.getVideoId());
        orderBack.setChannelName(deviceVo.getInandoutName());
        orderBack.setChannelId(deviceVo.getId());
        orderBack.setInandoutType(deviceVo.getInandoutType());
        orderBackService.addOrderBack(orderBack);
        return orderBack;
    }

    /**
     * 给当前订单打标记
     *
     * @param orderInfo 订单信息
     */
    private void orderTags(OrderInfo orderInfo) {
        List<OrderTags> listByOrderNum = orderTagsService.getListByOrderNum(orderInfo.getOrderNum(), null);
        Optional<OrderTags> first = listByOrderNum.stream().filter(orderTags ->
                OrderTags.TagIdEnum.BACK_IN_PARK.getType() == orderTags.getTagId()).findFirst();
        if (!first.isPresent()) {
            OrderTags orderTags = new OrderTags();
            orderTags.setOrderNum(orderInfo.getOrderNum());
            orderTags.setParkId(orderInfo.getParkId());
            orderTags.setRegionId(orderInfo.getRegionId());
            orderTags.setTagId(OrderTags.TagIdEnum.BACK_IN_PARK.getType());
            orderTagsService.addOrderTags(orderTags);
        }
        Optional<OrderTags> notPayOrder = listByOrderNum.stream().filter(orderTags ->
                OrderTags.TagIdEnum.NOT_PAY_CAR.getType() == orderTags.getTagId()).findFirst();
        if (notPayOrder.isPresent()) {
            orderNotpayService.deleteByOrderNum(orderInfo.getOrderNum());
        }
    }

    /**
     * 更新订单为在场状态
     *
     * @param orderInfo 订单信息
     */
    private void updateOrderInfo(OrderInfo orderInfo) {
        orderService.update(Wrappers.lambdaUpdate(OrderInfo.class)
                .set(OrderInfo::getExitTime, null)
                .set(OrderInfo::getOddStatus, null)
                .set(OrderInfo::getServiceStatus, OrderStatusConstants.IN_PARK)
                .eq(OrderInfo::getId, orderInfo.getId()));
    }

    /**
     * 请求通道视频
     *
     * @param deviceVo  通道信息
     * @param orderInfo 订单信息
     * @param backEvent 请求参数
     * @param orderBack 折返记录
     */
    private void getApplyCarVideo(TokenDeviceVo deviceVo, OrderInfo orderInfo, BackEventRequest backEvent, OrderBack orderBack) {
        ApplyCarVideoVo vo = new ApplyCarVideoVo();
        vo.setVideoType(2);
        vo.setSn(deviceVo.getDeviceNo());
        vo.setChannelId(deviceVo.getId());
        vo.setOrderNum(orderInfo.getOrderNum());
        vo.setParkId(orderInfo.getParkId());
        vo.setPlateNum(orderInfo.getPlateNum());
        vo.setVideoId(backEvent.getVideoId());
        vo.setRecTime(orderBack.getExitTime());
        rabbitSender.sendMessage(MqConstants.Exchange.AISLE_DELAYED_EXCHANGE, MqConstants.Routing.APPLY_CAR_VIDEO_ROUTING, vo, aisleApplyCarVideoTime * 1000L);
        log.info("[折返事件上报] 请求视频延迟消息发送, [{}]", vo);
    }

    /**
     * 添加或更新入口通道的缓存
     *
     * @param deviceVo  设备通道信息
     * @param orderInfo 订单信息
     * @param orderBack 折返记录
     */
    private void updateEnterChannelCache(TokenDeviceVo deviceVo, OrderInfo orderInfo, OrderBack orderBack) {
        CarEnterRequest enter = cacheHandle.getEntrance(deviceVo.getParkCode(), deviceVo.getInandoutCode());
        if (enter != null && enter.getPlateNum().equals(orderInfo.getPlateNum()) && Math.abs(enter.getEnterTime() - orderBack.getEnterTime()) < 5) {
            //删除缓存是为了下辆车来的时候，不把前车置为异常离场，因为折返了
            log.info("[入场折返事件上报] 删除通道缓存, 车牌号[{}]", orderInfo.getPlateNum());
            cacheHandle.removeEntrace(deviceVo.getParkCode(), deviceVo.getInandoutCode());
            cacheHandle.removeChannelFee(deviceVo.getParkCode(), deviceVo.getInandoutCode());
        }
    }

    /**
     * 添加或更新出口通道的缓存
     *
     * @param deviceVo  设备通道信息
     * @param orderInfo 订单信息
     * @param orderBack 折返记录
     */
    private void updateExitChannelCache(TokenDeviceVo deviceVo, OrderInfo orderInfo, OrderBack orderBack) {
        CarExitRequest exit = cacheHandle.getExit(deviceVo.getParkCode(), deviceVo.getInandoutCode());
        if (exit != null && exit.getPlateNum().equals(orderInfo.getPlateNum()) && Math.abs(exit.getOriginalExitTime() - orderBack.getExitTime()) < 5) {
            //删除缓存是为了下辆车来的时候，不把前车置为异常离场，因为折返了
            log.info("[出场折返事件上报] 删除通道缓存, 车牌号[{}]", orderInfo.getPlateNum());
            cacheHandle.removeExit(deviceVo.getParkCode(), deviceVo.getInandoutCode());
            cacheHandle.removeChannelFee(deviceVo.getParkCode(), deviceVo.getInandoutCode());
        }
    }
}
